(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 3.0,
MathReader 3.0, or any compatible application. The data for the notebook 
starts with the line of stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    566022,      21065]*)
(*NotebookOutlinePosition[    566919,      21096]*)
(*  CellTagsIndexPosition[    566875,      21092]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Simulation of Quantum Cryptography", "Title",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["Colin P. Williams ", "Subsubtitle",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[BoxData[
    \(Off[General::spell1]\)], "Input",
  InitializationCell->True],

Cell[CellGroupData[{

Cell["Copyright Notice", "Section",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
Copyright Colin P. Williams (1997).

This Notebook is intended to be used in conjunction with \"Explorations in \
Quantum Computing\" by Colin P. Williams and Scott H. Clearwater, TELOS, \
Springer-Verlag (1997), ISBN:0-387-94768-X. Permission is hereby granted to \
copy and distribute this Notebook freely for any non-commercial activity \
provided you include this copyright notice at the beginning of all such \
copies. Please send suggestions and bug reports to Colin P. Williams at 
        colin@solstice.jpl.nasa.gov      (818) 306 6512 or 
        cpw@cs.stanford.edu               (415) 728 2118
For information on \"Explorations in Quantum Computing\" check out the TELOS \
web site:  http://www.telospub.com/catalog/PHYSICS/Explorations.html. To \
order call 1-800-777-4643.

All other rights reserved.\
\>", "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["How to use this Notebook", "Section",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
This Notebook contains code for simulating the action of an \
idealized quantum cryptography machine (see \"Explorations in Quantum \
Computing\", Chapter 8).  Such a machine can be used to establish secret \
shared keys between two parties who may then send and receive secure messages \
using a provably secure classical cryptographic scheme such as the one time \
pad (see the Notebook OneTimePads.ma). 

You are encouraged to work through the Notebook, executing commands labelled \
as \"Try me!\" as you go.  To execute a \"Try me!\" command, position your \
cursor over the cell containing the command, click the mouse to select the \
cell and press the SHIFT and RETURN keys simultaneously.

This Notebook contains the following functions:\
\>", "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["DetectabilityProbability", "Input",
  Editable->False,
  AspectRatioFixed->True],

Cell["\<\
computes the probability of detecting eavesdropping as a function \
of the number of bits used in the test for eavesdropping.\
\>", "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["NumberOfBitsForDetectability", "Input",
  Editable->False,
  AspectRatioFixed->True],

Cell["\<\
computes the number of bits that Alice and Bob must compare in \
order to exceed a certain level of detectability.\
\>", "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["DistributeKeyUsingQKD", "Input",
  Editable->False,
  AspectRatioFixed->True],

Cell["\<\
generates a simulation of the quantum key distribution \
process.\
\>", "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Odds of Detecting Eavesdropping as a Function of the Number of Bits \
used for the Eavesdropping Test\
\>", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
Our model of quantum cryptography is idealized in the sense that it \
ignores errors arising from random events during the transmission of the \
polarized photons.  Hence discrepancies between photons transmitted by Alice \
and photons received by Bob, when Alice and Bob both know they used the same \
orientation of polarizers, are attributed to the eavesdropping of a malicious \
intermediary, called Eve. \
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Code", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
DetectabilityProbability[nBits_]:= N[1 - (3/4)^nBits]

DetectabilityProbability::usage =
\t\"DetectabilityProbability[n] returns the probability of detecting \\
\teavesdropping if n bits are used in the eavesdropping test. To be \\
\table to compare a bit, Alice and Bob will have already agreed upon \\
\tthe orientation of their polarizers.\";\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
NumberOfBitsForDetectability[probability_]:=
\t2 * Ceiling[Log[1-probability]/Log[.75]]

NumberOfBitsForDetectability::usage =
\t\"NumberOfBitsForDetectability[p] returns (roughly) the number \\
\tof bits that Alice and Bob need to compare in order to ensure \\
\ta detection probability of p.  Ignoring errors, for a trial to \\
\tdetect the presence of an eavesdropper, Alice and Bob must pick \\
\tthe same orientation of polarizers and yet measure different bits. \\
\tAlice and Bob can each use one of two orientations of polarizers \\
\tand so they will disagree on the choice of orientation about half \\
\tthe time.\";
\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
This shows how the probability of detecting eavesdropping varies \
with the number of bits used in the eavesdropping test.  To be using a bit, \
Alice and Bob will have already agreed that they were using the same \
polarizer orientations for the bit in question.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["\<\
Plot[DetectabilityProbability[n], {n, 1, 20},
     AxesOrigin->{0,0},
     Frame->True,
     GridLines->{Table[i, {i, 1, 20, 2}], Automatic},
     FrameLabel->{\"Number of Bits Tested\", \"Pr(Detecting Eve)\"},
     PlotStyle->Thickness[0.01]];\
\>", "Input",
  AspectRatioFixed->True],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.047619 -0.182319 0.788138 [
[.02381 -0.0125 -3 -9 ]
[.02381 -0.0125 3 0 ]
[.2619 -0.0125 -3 -9 ]
[.2619 -0.0125 3 0 ]
[.5 -0.0125 -6 -9 ]
[.5 -0.0125 6 0 ]
[.7381 -0.0125 -6 -9 ]
[.7381 -0.0125 6 0 ]
[.97619 -0.0125 -6 -9 ]
[.97619 -0.0125 6 0 ]
[ 0 0 -0.125 0 ]
[.5 -0.0125 -65 -21.5625 ]
[.5 -0.0125 65 -9 ]
[-0.0125 .05412 -18 -4.5 ]
[-0.0125 .05412 0 4.5 ]
[-0.0125 .13294 -18 -4.5 ]
[-0.0125 .13294 0 4.5 ]
[-0.0125 .21175 -18 -4.5 ]
[-0.0125 .21175 0 4.5 ]
[-0.0125 .29056 -18 -4.5 ]
[-0.0125 .29056 0 4.5 ]
[-0.0125 .36938 -18 -4.5 ]
[-0.0125 .36938 0 4.5 ]
[-0.0125 .44819 -18 -4.5 ]
[-0.0125 .44819 0 4.5 ]
[-0.0125 .527 -18 -4.5 ]
[-0.0125 .527 0 4.5 ]
[-0.0125 .60582 -6 -4.5 ]
[-0.0125 .60582 0 4.5 ]
[ 0 0 -0.125 0 ]
[-0.01875 .30902 -30.5625 -53 ]
[-0.01875 .30902 -18 53 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 .5 r
.25 Mabswid
.07143 0 m
.07143 .61803 L
s
.16667 0 m
.16667 .61803 L
s
.2619 0 m
.2619 .61803 L
s
.35714 0 m
.35714 .61803 L
s
.45238 0 m
.45238 .61803 L
s
.54762 0 m
.54762 .61803 L
s
.64286 0 m
.64286 .61803 L
s
.7381 0 m
.7381 .61803 L
s
.83333 0 m
.83333 .61803 L
s
.92857 0 m
.92857 .61803 L
s
0 .05412 m
1 .05412 L
s
0 .13294 m
1 .13294 L
s
0 .21175 m
1 .21175 L
s
0 .29056 m
1 .29056 L
s
0 .36938 m
1 .36938 L
s
0 .44819 m
1 .44819 L
s
0 .527 m
1 .527 L
s
0 g
.02381 0 m
.02381 .00625 L
s
[(0)] .02381 -0.0125 0 1 Mshowa
.2619 0 m
.2619 .00625 L
s
[(5)] .2619 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(10)] .5 -0.0125 0 1 Mshowa
.7381 0 m
.7381 .00625 L
s
[(15)] .7381 -0.0125 0 1 Mshowa
.97619 0 m
.97619 .00625 L
s
[(20)] .97619 -0.0125 0 1 Mshowa
.125 Mabswid
.07143 0 m
.07143 .00375 L
s
.11905 0 m
.11905 .00375 L
s
.16667 0 m
.16667 .00375 L
s
.21429 0 m
.21429 .00375 L
s
.30952 0 m
.30952 .00375 L
s
.35714 0 m
.35714 .00375 L
s
.40476 0 m
.40476 .00375 L
s
.45238 0 m
.45238 .00375 L
s
.54762 0 m
.54762 .00375 L
s
.59524 0 m
.59524 .00375 L
s
.64286 0 m
.64286 .00375 L
s
.69048 0 m
.69048 .00375 L
s
.78571 0 m
.78571 .00375 L
s
.83333 0 m
.83333 .00375 L
s
.88095 0 m
.88095 .00375 L
s
.92857 0 m
.92857 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
gsave
.5 -0.0125 -126 -25.5625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(Number) show
105.000000 12.812500 moveto
(of) show
123.000000 12.812500 moveto
(Bits) show
153.000000 12.812500 moveto
(Tested) show
189.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
0 .05412 m
.00625 .05412 L
s
[(0.3)] -0.0125 .05412 1 0 Mshowa
0 .13294 m
.00625 .13294 L
s
[(0.4)] -0.0125 .13294 1 0 Mshowa
0 .21175 m
.00625 .21175 L
s
[(0.5)] -0.0125 .21175 1 0 Mshowa
0 .29056 m
.00625 .29056 L
s
[(0.6)] -0.0125 .29056 1 0 Mshowa
0 .36938 m
.00625 .36938 L
s
[(0.7)] -0.0125 .36938 1 0 Mshowa
0 .44819 m
.00625 .44819 L
s
[(0.8)] -0.0125 .44819 1 0 Mshowa
0 .527 m
.00625 .527 L
s
[(0.9)] -0.0125 .527 1 0 Mshowa
0 .60582 m
.00625 .60582 L
s
[(1)] -0.0125 .60582 1 0 Mshowa
.125 Mabswid
0 .06988 m
.00375 .06988 L
s
0 .08565 m
.00375 .08565 L
s
0 .10141 m
.00375 .10141 L
s
0 .11717 m
.00375 .11717 L
s
0 .1487 m
.00375 .1487 L
s
0 .16446 m
.00375 .16446 L
s
0 .18022 m
.00375 .18022 L
s
0 .19599 m
.00375 .19599 L
s
0 .22751 m
.00375 .22751 L
s
0 .24327 m
.00375 .24327 L
s
0 .25904 m
.00375 .25904 L
s
0 .2748 m
.00375 .2748 L
s
0 .30633 m
.00375 .30633 L
s
0 .32209 m
.00375 .32209 L
s
0 .33785 m
.00375 .33785 L
s
0 .35361 m
.00375 .35361 L
s
0 .38514 m
.00375 .38514 L
s
0 .4009 m
.00375 .4009 L
s
0 .41667 m
.00375 .41667 L
s
0 .43243 m
.00375 .43243 L
s
0 .46395 m
.00375 .46395 L
s
0 .47972 m
.00375 .47972 L
s
0 .49548 m
.00375 .49548 L
s
0 .51124 m
.00375 .51124 L
s
0 .54277 m
.00375 .54277 L
s
0 .55853 m
.00375 .55853 L
s
0 .57429 m
.00375 .57429 L
s
0 .59006 m
.00375 .59006 L
s
0 .03836 m
.00375 .03836 L
s
0 .0226 m
.00375 .0226 L
s
0 .00683 m
.00375 .00683 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
gsave
-0.01875 .30902 -138.281 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
114 10.2813 translate
90 rotate
-114 -10.2813 translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(Pr) show
%%IncludeResource: font Math2Mono
%%IncludeFont: Math2Mono
/Math2Mono findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
75.000000 12.812500 moveto
(H) show
81.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(Detecting) show
141.000000 12.812500 moveto
(Eve) show
%%IncludeResource: font Math2Mono
%%IncludeFont: Math2Mono
/Math2Mono findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
159.000000 12.812500 moveto
(L) show
165.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math2Mono
%%+ font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math2Mono
%%+ Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math2Mono
%%+ font Courier
grestore
.02381 .61178 m
.02381 .61803 L
s
.2619 .61178 m
.2619 .61803 L
s
.5 .61178 m
.5 .61803 L
s
.7381 .61178 m
.7381 .61803 L
s
.97619 .61178 m
.97619 .61803 L
s
.125 Mabswid
.07143 .61428 m
.07143 .61803 L
s
.11905 .61428 m
.11905 .61803 L
s
.16667 .61428 m
.16667 .61803 L
s
.21429 .61428 m
.21429 .61803 L
s
.30952 .61428 m
.30952 .61803 L
s
.35714 .61428 m
.35714 .61803 L
s
.40476 .61428 m
.40476 .61803 L
s
.45238 .61428 m
.45238 .61803 L
s
.54762 .61428 m
.54762 .61803 L
s
.59524 .61428 m
.59524 .61803 L
s
.64286 .61428 m
.64286 .61803 L
s
.69048 .61428 m
.69048 .61803 L
s
.78571 .61428 m
.78571 .61803 L
s
.83333 .61428 m
.83333 .61803 L
s
.88095 .61428 m
.88095 .61803 L
s
.92857 .61428 m
.92857 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .05412 m
1 .05412 L
s
.99375 .13294 m
1 .13294 L
s
.99375 .21175 m
1 .21175 L
s
.99375 .29056 m
1 .29056 L
s
.99375 .36938 m
1 .36938 L
s
.99375 .44819 m
1 .44819 L
s
.99375 .527 m
1 .527 L
s
.99375 .60582 m
1 .60582 L
s
.125 Mabswid
.99625 .06988 m
1 .06988 L
s
.99625 .08565 m
1 .08565 L
s
.99625 .10141 m
1 .10141 L
s
.99625 .11717 m
1 .11717 L
s
.99625 .1487 m
1 .1487 L
s
.99625 .16446 m
1 .16446 L
s
.99625 .18022 m
1 .18022 L
s
.99625 .19599 m
1 .19599 L
s
.99625 .22751 m
1 .22751 L
s
.99625 .24327 m
1 .24327 L
s
.99625 .25904 m
1 .25904 L
s
.99625 .2748 m
1 .2748 L
s
.99625 .30633 m
1 .30633 L
s
.99625 .32209 m
1 .32209 L
s
.99625 .33785 m
1 .33785 L
s
.99625 .35361 m
1 .35361 L
s
.99625 .38514 m
1 .38514 L
s
.99625 .4009 m
1 .4009 L
s
.99625 .41667 m
1 .41667 L
s
.99625 .43243 m
1 .43243 L
s
.99625 .46395 m
1 .46395 L
s
.99625 .47972 m
1 .47972 L
s
.99625 .49548 m
1 .49548 L
s
.99625 .51124 m
1 .51124 L
s
.99625 .54277 m
1 .54277 L
s
.99625 .55853 m
1 .55853 L
s
.99625 .57429 m
1 .57429 L
s
.99625 .59006 m
1 .59006 L
s
.99625 .03836 m
1 .03836 L
s
.99625 .0226 m
1 .0226 L
s
.99625 .00683 m
1 .00683 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.01 w
.07143 .01472 m
.10813 .13227 L
.14816 .23399 L
.18575 .30954 L
.22191 .36767 L
.26039 .41707 L
.29743 .45492 L
.33681 .48686 L
.37474 .51122 L
.41123 .52994 L
.45006 .5458 L
.48744 .55794 L
.52338 .56728 L
.56166 .57524 L
.59849 .58134 L
.63765 .5865 L
.67538 .59043 L
.71166 .59346 L
.75027 .59603 L
.78745 .598 L
.82695 .59966 L
.86501 .60093 L
.90164 .6019 L
.94059 .60272 L
.97619 .60332 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{282, 174},
  ImageMargins->{{34, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004J0000[Q000`40O003h00Oogoo6goo003oOolK
Ool00?mooa]oo`00ogoo6goo003oOolKOol005ioo`<00004Ool007ooOol400000goo001oo`050003
Ool20002Ool3000?Ool20002Ool30009Ool40002Ool30003Ool20002Ool2000=Ool30003Ool20002
Ool20004Ool20003Ool20003Ool30014Ool005moo`06001oo`00001oo`000Woo0P0000Ioo`00Ool0
07oo0002Ool00`00Ool00005Ool00`00Oomoo`0=Ool01000Oomoo`000Woo00<007ooOol02Goo00@0
07ooOol0009oo`05001oogooOol00002Ool01@00Oomoogoo00003Goo00D007ooOomoo`0000Ioo`04
001oogoo0002Ool00`00Ool00004Ool01000Oomoo`00A7oo001OOol01P00Ool00000Ool0009oo`80
0006Ool007oo001oo`000Woo00<007oo00000P000goo00<007ooOol03Goo00@007ooOol0009oo`03
001oogoo00Uoo`04001oogoo0002Ool01@00Oomoogoo00001Goo00<007ooOol037oo00@007ooOomo
o`<000=oo`05001oogooOol00004Ool30002Ool01000Oomoo`00A7oo001OOol200001Goo00000000
Ool0100000Aoo`00Oomoo`<0009oo`<0009oo`@000ioo`80009oo`<000Yoo`<0009oo`80009oo`@0
00=oo`8000eoo`04001oogooOol30003Ool200000goo000000020002Ool30003Ool30014Ool005mo
o`800003Ool007oo00Yoo`03001oogoo01ioo`03001oogoo00Uoo`04001oogoo0006Ool00`00Oomo
o`0@Ool01@00Ool007oo00002goo00<007ooOol02Woo00<007ooOol0@Woo001NOol200000goo0000
000:Ool2000QOol00`00Oomoo`07Ool40003Ool00`00Oomoo`0DOol5000GOol20014Ool00?mooa]o
o`00ogoo6goo003oOolKOol002aoo`03001oogoo03Eoo`03001oogoo035oo`D0009oo`03001oogoo
02moo`D0009oo`03001oogoo031oo`<000=oo`03001oogoo00Ioo`00:goo00<007oo0000=Goo00<0
07oo0000=7oo00L007ooOomoo`00Ool0000bOol01`00Oomoogoo001oo`00035oo`03001oogoo009o
o`03001oo`0000Moo`00:goo00<007oo0000=goo00<007ooOol0<Woo00L007ooOomoo`00Ool0000b
Ool00`00Oomoo`03Ool00`00Oomoo`0`Ool01`00Oomoogoo001oo`0000Moo`00:goo00<007oo0000
=Goo0P00=Goo00L007ooOomoo`00Ool0000bOol01000Oomoogoo0P00<goo00L007ooOomoo`00Ool0
0007Ool002]oo`03001oo`0003Eoo`03001oogoo03Aoo`07001oogooOol007oo0000<Woo00D007oo
Oomoo`0003=oo`03001oo`00009oo`03001oo`0000Moo`00;7oo00<007ooOol0=7oo0`00<goo0P00
17oo00<007ooOol0<7oo0P000goo0`00<Woo00<007ooOol00Woo00<007ooOol01Woo003oOolKOol0
0?mooa]oo`00ogoo6goo003oOolKOol00?mooa]oo`009gookP001Goo000WOol00`00Oomoo`02Ool0
0`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`09Ool0
0`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomoo`08Ool0
0`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool0
0`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`09Ool0
0`00Oomoo`02Ool00`00Oomoo`03Ool002Moo`03001oogoo009oo`03001oogoo00Qoo`0300moogoo
01Aoo`0300moogoo01Aoo`03001oogoo01=oo`0300moogoo01Aoo`0300moogoo00Qoo`03001oogoo
00Uoo`0300moogoo01=oo`0300moogoo01Aoo`03001oogoo01Aoo`0300moogoo01=oo`0300moogoo
00Uoo`04001oogooOol20005Ool002Moo`03001oogoo00eoo`03001oogoo01Aoo`0300moogoo01Ao
o`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Ao
o`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`009goo00<007oo
Ool037oo0`005Goo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo
00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo
00<007ooOol00goo000WOol00`00Oomoo`0=Ool2000EOol00`0?Oomoo`0DOol00`0?Oomoo`0COol0
0`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol0
0`0?Oomoo`0COol00`0?Oomoo`0=Ool20005Ool002Moo`03001oogoo00eoo`8001Eoo`0300moogoo
01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo
01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`009goo00<0
07ooOol03Goo0`0057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol0
57oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol0
3Woo00<007ooOol00goo000WOol00`00Oomoo`0=Ool00`0?0000000DOol00`0?Oomoo`0DOol00`0?
Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?
Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool002Moo`03001oogoo00eo
o`0300l0000001Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Ao
o`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00eo
o`8000Eoo`009goo00<007ooOol03Goo00@03`00000001=oo`0300moogoo01Aoo`0300moogoo01=o
o`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Ao
o`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`005goo00<007ooOol00Woo00@007oo
Oomoo`8000Eoo`03001oogoo00eoo`0400moo`00000COol00`0?Oomoo`0DOol00`0?Oomoo`0COol0
0`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol0
0`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool001Ioo`03001oo`0000Uoo`03001oogoo
009oo`03001oogoo00eoo`0400moo`00000COol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomo
o`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomo
o`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool001Ioo`03001oo`0000Qoo`03001oogoo00=oo`80
01003`<00=L03`8000Eoo`005Woo00<007oo00002Goo00<007ooOol00Woo00<007ooOol03Goo00<0
3gooOol00P004Woo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo
00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo
00<007ooOol00goo000FOol00`00Ool00007Ool00`00Ool00004Ool00`00Oomoo`0=Ool00`0?Oomo
o`02000BOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?
Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00
Oomoo`03Ool001Moo`03001oogoo00Moo`03001oogoo00=oo`03001oogoo00eoo`0300moogoo00<0
015oo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo
01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo
00=oo`009goo00<007ooOol03Goo00@03gooOomoo`80015oo`0300moogoo01Aoo`0300moogoo01=o
o`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Ao
o`0300moogoo01=oo`0300moogoo00eoo`8000Eoo`009goo00<007ooOol03Goo00@03gooOomoo`80
015oo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo
01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo
00=oo`009goo00<007ooOol03Goo00@03gooOomoo`<0011oo`0300moogoo01Aoo`0300moogoo01=o
o`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Ao
o`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`009goo00<007ooOol03Goo00<03goo
Ool00Woo0P0047oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo
00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Goo
0P001Goo000WOol00`00Oomoo`0=Ool00`0?Oomoo`02Ool2000@Ool00`0?Oomoo`0DOol00`0?Oomo
o`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomo
o`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool002Moo`03001oogoo00eoo`03
00moogoo009oo`<000moo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo
01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo
00ioo`03001oogoo00=oo`009goo00<007ooOol03Goo00<03gooOol00goo0P003goo00<03gooOol0
57oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol0
57oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<007ooOol00goo0005Ool00`00
Oomoo`03Ool00`00Oomoo`0IOol00`00Oomoo`0=Ool00`0?Oomoo`03Ool2000?Ool00`0?Oomoo`0D
Ool00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0D
Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0=Ool20005Ool000Eoo`L001]oo`03001o
ogoo00eoo`0300moogoo00=oo`<000ioo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Ao
o`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=o
o`0300moogoo00ioo`03001oogoo00=oo`001Goo00L007ooOomoo`00Ool0000KOol00`00Oomoo`0=
Ool00`0?Oomoo`04Ool2000>Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?
Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?
Oomoo`0>Ool00`00Oomoo`03Ool000Eoo`05001oogooOol0000MOol00`00Oomoo`0=Ool00`0?Oomo
o`04Ool2000>Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol0
0`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>Ool0
0`00Oomoo`03Ool000Ioo`<001ioo`03001oogoo00eoo`0300moogoo00Aoo`<000eoo`0300moogoo
01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo
01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00eoo`8000Eoo`009goo00<007ooOol0
3Goo00<03gooOol01Goo0P003Goo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<0
3gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<0
3gooOol03Woo00<007ooOol00goo000GOol00`00Oomoo`02Ool00`00Oomoo`02Ool30003Ool00`00
Oomoo`0=Ool00`0?Oomoo`05Ool2000=Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0D
Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0C
Ool00`0?Oomoo`0>Ool00`00Oomoo`03Ool000Moo`05001oogooOol0000:Ool00`00Ool00009Ool0
0`00Oomoo`02Ool00`00Oomoo`0=Ool00`0?Oomoo`05Ool3000<Ool00`0?Oomoo`0DOol00`0?Oomo
o`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomo
o`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool000Moo`D000Yoo`03001oo`00
00Moo`@000=oo`8001L03`800=403`8000Eoo`0027oo00@007ooOol000Yoo`03001oo`0000Moo`03
001oo`0000Aoo`03001oogoo00eoo`0300moogoo00Ioo`8000aoo`0300moogoo01Aoo`0300moogoo
01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo
01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`001goo00D007ooOomoo`0000Yo
o`03001oo`0000Qoo`8000Aoo`03001oogoo00eoo`0300moogoo00Ioo`<000]oo`0300moogoo01Ao
o`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Ao
o`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`001goo00<007oo
Ool03Goo00<007ooOol027oo00<007ooOol00Woo00<007ooOol03Goo00<03gooOol01goo0P002goo
00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo
00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Goo0P001Goo000WOol0
0`00Oomoo`0=Ool00`0?Oomoo`07Ool3000:Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomo
o`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomo
o`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool000Qoo`@001]oo`03001oogoo00eoo`0300moogoo
00Qoo`8000Yoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`03
00moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03
001oogoo00=oo`001goo00<007ooOol00Woo00<007ooOol067oo00<007ooOol03Goo00<03gooOol0
27oo0P002Woo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<0
3gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<0
07ooOol00goo0006Ool00`00Oomoo`04Ool00`00Oomoo`0GOol00`00Oomoo`0=Ool00`0?Oomoo`08
Ool30009Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?
Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0=Ool20005
Ool002Moo`03001oogoo00eoo`0300moogoo00Uoo`8000Uoo`0300moogoo01Aoo`0300moogoo01=o
o`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Ao
o`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`001Goo00<007ooOol00goo00<007oo
Ool06Goo00<007ooOol03Goo00<03gooOol02Goo0`0027oo00<03gooOol057oo00<03gooOol04goo
00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo
00<03gooOol04goo00<03gooOol03Woo00<007ooOol00goo0005Ool7000KOol00`00Oomoo`0=Ool0
0`0?Oomoo`0:Ool20008Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomo
o`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomo
o`0>Ool00`00Oomoo`03Ool000Eoo`03001oogoo00=oo`03001oogoo01Uoo`03001oogoo00eoo`03
00moogoo00Yoo`8000Qoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo
01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo
00eoo`8000Eoo`001Goo00<007ooOol00goo00<007ooOol06Goo00<007ooOol03Goo00<03gooOol0
2Woo0`001goo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<0
3gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<0
07ooOol00goo0006Ool01@00Oomoogoo000077oo00<007ooOol03Goo00<03gooOol02goo0P001goo
00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo
00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<007ooOol00goo
0007Ool3000MOol00`00Oomoo`0=Ool00`0?Oomoo`0;Ool30006Ool00`0?Oomoo`0DOol00`0?Oomo
o`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomo
o`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool002Moo`03001oogoo00eoo`03
00moogoo00aoo`8000Ioo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo
01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo
00eoo`8000Eoo`005goo00<007ooOol00Woo00<007ooOol00Woo00<007ooOol00goo00<007ooOol0
3Goo00<03gooOol037oo0P001Woo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<0
3gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<0
3gooOol03Woo00<007ooOol00goo0008Ool3000;Ool00`00Ool00007Ool00`00Ool00004Ool00`00
Oomoo`0=Ool00`0?Oomoo`0<Ool30005Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0D
Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0C
Ool00`0?Oomoo`0>Ool00`00Oomoo`03Ool000Moo`05001oo`00Ool0000:Ool00`00Ool00009Ool0
0`00Oomoo`02Ool2000N00l2003:00l20005Ool000Moo`05001oo`00Ool0000:Ool00`00Ool00007
Ool20005Ool00`00Oomoo`0=Ool00`0?Oomoo`0=Ool30004Ool00`0?Oomoo`0DOol00`0?Oomoo`0C
Ool00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0D
Ool00`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool000Moo`05001oo`00Ool0000:Ool0
0`00Ool00007Ool00`00Oomoo`04Ool00`00Oomoo`0=Ool00`0?Oomoo`0>Ool20004Ool00`0?Oomo
o`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomo
o`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool000Qoo`80
0003Ool007oo00Yoo`03001oogoo00Ioo`<000Aoo`03001oogoo00eoo`0300moogoo00ioo`8000Ao
o`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=o
o`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=o
o`009goo00<007ooOol03Goo00<03gooOol03Woo0`000goo00<03gooOol057oo00<03gooOol04goo
00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo
00<03gooOol04goo00<03gooOol03Goo0P001Goo0007Ool00`00Oomoo`0MOol00`00Oomoo`0=Ool0
0`0?Oomoo`0?Ool20003Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomo
o`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomo
o`0>Ool00`00Oomoo`03Ool000Ioo`D001aoo`03001oogoo00eoo`0300moogoo00moo`<0009oo`03
00moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`03
00moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`00
1goo00D007ooOomoo`0001]oo`03001oogoo00eoo`0300moogoo011oo`80009oo`0300moogoo01Ao
o`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Ao
o`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`001goo00D007oo
Oomoo`0001]oo`03001oogoo00eoo`0300moogoo015oo`04001oogoo00lFOol00`0?Oomoo`0COol0
0`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol0
0`0?Oomoo`0COol00`0?Oomoo`0=Ool20005Ool000Moo`05001oogooOol0000KOol00`00Oomoo`0=
Ool00`0?Oomoo`0AOol200000goo00moo`0EOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomo
o`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomo
o`0>Ool00`00Oomoo`03Ool000Yoo`03001oogoo01Yoo`03001oogoo00eoo`0300moogoo019oo`03
001oo`0?01Ioo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`03
00moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`00
9goo00<007ooOol03Goo00<03gooOol04Woo0P0000<03gooOol057oo00<03gooOol04goo00<03goo
Ool057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03goo
Ool04goo00<03gooOol03Woo00<007ooOol00goo000WOol00`00Oomoo`0=Ool00`0?Oomoo`0COol0
0`0000moo`0EOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol0
0`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0=Ool20005Ool000Qoo`<0
01aoo`03001oogoo00eoo`0300moogoo01=oo`8001Ioo`0300moogoo01=oo`0300moogoo01Aoo`03
00moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`03
00moogoo00ioo`03001oogoo00=oo`001goo00D007oo001oo`0001]oo`03001oogoo00eoo`0300mo
ogoo01Aoo`03001oogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300mo
ogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001o
ogoo00=oo`001goo00D007oo001oo`0001]oo`03001oogoo00eoo`0300moogoo01Aoo`8001Eoo`03
00moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`03
00moogoo01Aoo`0300moogoo01=oo`0300moogoo00eoo`8000Eoo`001goo00D007oo001oo`0001]o
o`03001oogoo00eoo`0300moogoo01Aoo`0300l007oo01Aoo`0300moogoo01=oo`0300moogoo01Ao
o`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=o
o`0300moogoo00ioo`03001oogoo00=oo`0027oo0P0000=oo`00Ool02Woo00<007ooOol00Woo00@0
07ooOomoo`8000Eoo`03001oogoo00eoo`0300moogoo01Aoo`0300l0000001Aoo`0300moogoo01=o
o`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Ao
o`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`005Woo00<007oo00001goo00<007oo
000017oo00<007ooOol03Goo00<03gooOol057oo00<03goo000057oo00<03gooOol04goo00<03goo
Ool057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03goo
Ool04goo00<03gooOol03Woo00<007ooOol00goo000FOol00`00Ool00007Ool00`00Ool00004Ool2
000W00l2003100l20005Ool000Qoo`<000]oo`03001oo`0000Moo`8000Eoo`03001oogoo00eoo`03
00moogoo01Aoo`0400moogoo000COol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol0
0`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>Ool0
0`00Oomoo`03Ool000Moo`05001oogooOol0000:Ool00`00Ool00007Ool00`00Oomoo`04Ool00`00
Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomoo`02000BOol00`0?Oomoo`0COol00`0?Oomoo`0DOol0
0`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol0
0`0?Oomoo`0>Ool00`00Oomoo`03Ool000Moo`05001oogooOol0000;Ool00`00Oomoo`07Ool20004
Ool00`00Oomoo`0=Ool00`0?Oomoo`0DOol01@0?Oomoogoo00004Woo00<03gooOol04goo00<03goo
Ool057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03goo
Ool04goo00<03gooOol03Woo00<007ooOol00goo0007Ool01@00Oomoogoo00006goo00<007ooOol0
3Goo00<03gooOol057oo00@03gooOomoo`80015oo`0300moogoo01=oo`0300moogoo01Aoo`0300mo
ogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300mo
ogoo00eoo`8000Eoo`001goo0P000Woo00<007ooOol06Goo00<007ooOol03Goo00<03gooOol057oo
00<03gooOol00Woo00<007ooOol03goo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo
00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo
00<007ooOol00goo000WOol00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomoo`02Ool2000@Ool0
0`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol0
0`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool000Moo`03001oogoo
01eoo`03001oogoo00eoo`0300moogoo01Aoo`0300moogoo00=oo`8000moo`0300moogoo01=oo`03
00moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`03
00moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`001Woo1@0077oo00<007ooOol03Goo00<0
3gooOol057oo00<03gooOol017oo00<007ooOol03Goo00<03gooOol04goo00<03gooOol057oo00<0
3gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<0
3gooOol03Goo0P001Goo0007Ool01@00Oomoogoo00006goo00<007ooOol03Goo00<03gooOol057oo
00<03gooOol017oo0P003Woo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03goo
Ool04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<007oo
Ool00goo0007Ool01@00Oomoogoo00006goo00<007ooOol03Goo00<03gooOol057oo00<03gooOol0
1Goo00<007ooOol037oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol0
4goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<007ooOol0
0goo0007Ool01@00Oomoogoo00006goo00<007ooOol03Goo00<03gooOol057oo00<03gooOol01Goo
0P003Goo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03goo
Ool057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Goo0P001Goo000:Ool00`00Oomo
o`0JOol00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomoo`06Ool2000<Ool00`0?Oomoo`0COol0
0`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol0
0`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool002Moo`03001oogoo00eoo`0300moogoo
01Aoo`0300moogoo00Moo`03001oogoo00Yoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo
01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo
00ioo`03001oogoo00=oo`009goo00<007ooOol03Goo00<03gooOol057oo00<03gooOol01goo0P00
2goo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol0
57oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<007ooOol00goo0007Ool01@00
Oomoogoo00006goo00<007ooOol03Goo00<03gooOol057oo00<03gooOol027oo00<007ooOol02Goo
00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo
00<03gooOol057oo00<03gooOol04goo00<03gooOol03Goo0P001Goo0007Ool01@00Oomoogoo0000
6goo00<007ooOol03Goo00<03gooOol057oo00<03gooOol027oo0P002Woo00<03gooOol04goo00<0
3gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<0
3gooOol04goo00<03gooOol03Woo00<007ooOol00goo0005Ool00`00Ool00004000;Ool00`00Oomo
o`02Ool00`00Oomoo`02Ool00`00Oomoo`03Ool00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomo
o`09Ool20009Ool00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol0
0`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool0
00]oo`03001oogoo00Qoo`03001oo`0000Qoo`03001oogoo00=oo`03001oogoo00eoo`0300moogoo
01Aoo`0300moogoo00Yoo`03001oogoo00Moo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo
01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo
00ioo`03001oogoo00=oo`002goo00<007ooOol027oo00<007oo000027oo00<007ooOol00goo0P00
<P0?0P00]P0?0P001Goo000FOol00`00Ool00009Ool00`00Oomoo`02Ool00`00Oomoo`0=Ool00`0?
Oomoo`0DOol00`0?Oomoo`0;Ool20007Ool00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0D
Ool00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>
Ool00`00Oomoo`03Ool000Moo`05001oogooOol0000:Ool00`00Ool00007Ool00`00Ool00004Ool0
0`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomoo`0<Ool20006Ool00`0?Oomoo`0COol00`0?Oomo
o`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomo
o`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool000Moo`D000]oo`03001oogoo00Ioo`<000Aoo`03
001oogoo00eoo`0300moogoo01Aoo`0300moogoo00eoo`03001oogoo00Aoo`0300moogoo01=oo`03
00moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`03
00moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`001goo00D007ooOomoo`0001]oo`03001o
ogoo00eoo`0300moogoo01Aoo`0300moogoo00eoo`8000Eoo`0300moogoo01=oo`0300moogoo01Ao
o`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=o
o`0300moogoo00eoo`8000Eoo`001goo00<007ooOol07Goo00<007ooOol03Goo00<03gooOol057oo
00<03gooOol03Woo0P0017oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03goo
Ool04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<007oo
Ool00goo0007Ool01@00Oomoogoo00006goo00<007ooOol03Goo00<03gooOol057oo00<03gooOol0
3goo0P000goo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<0
3gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<007ooOol00goo0008
Ool4000KOol00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomoo`0@Ool01@00Oomoogoo00l05Goo
00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo
00<03gooOol04goo00<03gooOol03Goo0P001Goo000;Ool00`00Oomoo`0IOol00`00Oomoo`0=Ool0
0`0?Oomoo`0DOol00`0?Oomoo`0@Ool20002Ool00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomo
o`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomo
o`0>Ool00`00Oomoo`03Ool002Moo`03001oogoo00eoo`0300moogoo01Aoo`0300moogoo015oo`80
0003Ool03goo01Aoo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Ao
o`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`0027oo0`0077oo
00<007ooOol03Goo00<03gooOol057oo00<03gooOol04Woo0P0000<03gooOol04goo00<03gooOol0
57oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol0
4goo00<03gooOol03Woo00<007ooOol00goo0007Ool01`00Oomoogoo001oo`0001Uoo`03001oogoo
00eoo`0300moogoo01Aoo`0300moogoo019oo`<001Eoo`0300moogoo01Aoo`0300moogoo01Aoo`03
00moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00eoo`80
00Eoo`001goo00L007ooOomoo`00Ool0000IOol00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomo
o`0COol3000DOol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol0
0`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool000Moo`04001oogoo
0002Ool00`00Oomoo`0GOol00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomoo`0DOol3000COol0
0`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol0
0`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool000Eoo`03001oo`0000D001Yoo`03001o
ogoo00eoo`0300moogoo01Aoo`0300moogoo01Aoo`0400l00000000BOol00`0?Oomoo`0DOol00`0?
Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?
Oomoo`0>Ool00`00Oomoo`03Ool000Eoo`L001]oo`03001oogoo00eoo`0300moogoo01Aoo`0300mo
ogoo01Aoo`0300moo`000080015oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`03
00moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00eoo`8000Eoo`001Goo00@0
07ooOol0009oo`03001oogoo01Uoo`03001oogoo00eoo`0300moogoo01Aoo`0300moogoo01Aoo`03
00moogoo00<0011oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Ao
o`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`001Goo00<007oo
00000P0000=oo`00Ool02Woo00<007ooOol00Woo00<007ooOol00Woo00<007ooOol00goo00<007oo
Ool03Goo00<03gooOol057oo00<03gooOol057oo00@03gooOomoo`<000moo`0300moogoo01Aoo`03
00moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`03
00moogoo00ioo`03001oogoo00=oo`001Goo0P000goo0P002Woo00<007oo00001goo00<007oo0000
17oo00<007ooOol03Goo00<03gooOol057oo00<03gooOol057oo00<03gooOol00Woo0`003Woo00<0
3gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<0
3gooOol04goo00<03gooOol03Woo00<007ooOol00goo000FOol00`00Ool00007Ool00`00Ool00004
Ool2001200l3002U00l20005Ool000Moo`03001oogoo00aoo`03001oo`0000Qoo`03001oogoo00=o
o`03001oogoo00eoo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo00Aoo`<000aoo`0300mo
ogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300mo
ogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`001goo0`0037oo00<007oo00001goo00<007oo
000017oo00<007ooOol03Goo00<03gooOol057oo00<03gooOol057oo00<03gooOol01Goo10002Woo
00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo
00<03gooOol04goo00<03gooOol03Woo00<007ooOol00goo0007Ool00`00Oomoo`02000;Ool00`00
Oomoo`07Ool00`00Oomoo`03Ool00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?
Oomoo`06Ool40009Ool00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0D
Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0=Ool20005Ool000]oo`03001oogoo01Uo
o`03001oogoo00eoo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo00Qoo`<000Qoo`0300mo
ogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300mo
ogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`001goo00@007oo000001aoo`03001oogoo00eo
o`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo00Uoo`<000Moo`0300moogoo01Aoo`0300mo
ogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300mo
ogoo00ioo`03001oogoo00=oo`001goo0P007Woo00<007ooOol03Goo00<03gooOol057oo00<03goo
Ool057oo00<03gooOol02Woo0`001Woo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo
00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<007ooOol00goo
0007Ool00`00Oomoo`0MOol00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomo
o`0;Ool40004Ool00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol0
0`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0=Ool20005Ool002Moo`03001oogoo00eoo`03
00moogoo01Aoo`0300moogoo01Aoo`0300moogoo00aoo`@000=oo`0300moogoo01Aoo`0300moogoo
01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo
00ioo`03001oogoo00=oo`0027oo0`0077oo00<007ooOol03Goo00<03gooOol057oo00<03gooOol0
57oo00<03gooOol03Woo100000=oo`0?Ool05Goo00<03gooOol057oo00<03gooOol04goo00<03goo
Ool057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<007ooOol00goo0007Ool0
1@00Ool007oo00006goo00<007ooOol03Goo00<03gooOol057oo00<03gooOol057oo00<03gooOol0
3goo100000<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03goo
Ool057oo00<03gooOol04goo00<03gooOol03Woo00<007ooOol00goo0007Ool01@00Ool007oo0000
6goo00<007ooOol03Goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04Goo0`005Woo00<0
3gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<0
3gooOol03Goo0P001Goo0007Ool01@00Ool007oo00006goo00<007ooOol03Goo00<03gooOol057oo
00<03gooOol057oo00<03gooOol04Woo100057oo00<03gooOol057oo00<03gooOol04goo00<03goo
Ool057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<007ooOol00goo0006Ool0
2000Ool00000Ool007oo000IOol00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?
Oomoo`0COol4000COol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0D
Ool00`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool000Moo`03001oogoo009oo`03001o
ogoo01Qoo`03001oogoo00eoo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300mo
o`0000<0015oo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`03
00moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`0027oo10006goo00<007ooOol03Goo00<0
3gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol01@003goo00<03gooOol057oo
00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Goo
0P001Goo000GOol00`00Oomoo`02Ool01000Oomoogoo0P001Goo00<007ooOol03Goo00<03gooOol0
57oo00<03gooOol057oo00<03gooOol04goo00<03gooOol00Woo1@003Goo00<03gooOol057oo00<0
3gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<0
07ooOol00goo000FOol00`00Ool00009Ool00`00Oomoo`02Ool00`00Oomoo`0=Ool00`0?Oomoo`0D
Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`04Ool5000;Ool00`0?Oomoo`0DOol00`0?
Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00
Oomoo`03Ool001Ioo`03001oo`0000Qoo`8000Aoo`8005/03`D008X03`8000Eoo`005Woo00<007oo
00001goo00<007oo000017oo00<007ooOol03Goo00<03gooOol057oo00<03gooOol057oo00<03goo
Ool04goo00<03gooOol027oo1@001goo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo
00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<007ooOol00goo000FOol00`00Ool0
0007Ool00`00Ool00004Ool00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomo
o`0COol00`0?Oomoo`0:Ool50005Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol0
0`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool001Moo`03001oogoo
00Moo`8000Aoo`03001oogoo00eoo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`03
00moogoo00aoo`H0009oo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo
01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`009goo00<007ooOol03Goo00<0
3gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo1P0000<03gooOol057oo
00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Goo
0P001Goo000WOol00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol0
0`0?Oomoo`0AOol6000DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomo
o`0COol00`0?Oomoo`0>Ool00`00Oomoo`03Ool002Moo`03001oogoo00eoo`0300moogoo01Aoo`03
00moogoo01Aoo`0300moogoo01=oo`0300moogoo01=oo`L0015oo`0300moogoo01=oo`0300moogoo
01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03001oogoo00=oo`009goo00<0
07ooOol03Goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<0
3goo00001P003Woo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo
00<03gooOol03Woo00<007ooOol00goo000WOol00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomo
o`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`02Ool8000:Ool00`0?Oomoo`0COol0
0`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0=Ool20005Ool002Moo`03
001oogoo00eoo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`03
00moogoo00Eoo`X000Eoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo
01=oo`0300moogoo00ioo`03001oogoo00=oo`009goo00<007ooOol03Goo00<03gooOol057oo00<0
3gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol02Woo2@0000=oo`0?Ool057oo
00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol03Woo00<007ooOol00goo
000WOol00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomo
o`0DOol00`0?Oomoo`0>Ool:000BOol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol0
0`0?Oomoo`0>Ool00`00Oomoo`03Ool002Moo`03001oogoo00eoo`0300moogoo01Aoo`0300moogoo
01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300moogoo01=oo`h000Uoo`0300moogoo01Aoo`03
00moogoo01Aoo`0300moogoo01=oo`0300moogoo00eoo`8000Eoo`009goo00<007ooOol03Goo00<0
3gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00@0
3gooOomooa80000300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo00ioo`03
001oogoo00=oo`009goo00<007ooOol03Goo00<03gooOol057oo00<03gooOol057oo00<03gooOol0
4goo00<03gooOol057oo00<03gooOol057oo00<03gooOol02Goo4`003Woo00<03gooOol057oo00<0
3gooOol04goo00<03gooOol03Woo00<007ooOol00goo000WOol00`00Oomoo`0=Ool00`0?Oomoo`0D
Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0B
OolC0005Ool00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0=Ool20005Ool002Moo`03001o
ogoo00eoo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300mo
ogoo01Aoo`0300moogoo01=oo`0300moogoo00Eooa8000Qoo`8000Yoo`0300moogoo01=oo`0300mo
ogoo00ioo`03001oogoo00=oo`007goo1@000goo00<007ooOol03Goo00<03gooOol057oo00<03goo
Ool057oo00<03gooOol04goo00<03gooOol057oo00<03gooOol057oo00<03gooOol04goo00<03goo
Ool03Woo6`000Woo00<03gooOol04goo00<03gooOol03Woo00<007ooOol00goo000QOol00`00Oomo
o`03Ool00`00Oomoo`0=Ool00`0?Oomoo`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomo
o`0DOol00`0?Oomoo`0DOol00`0?Oomoo`0COol00`0?Oomoo`0DOol00`0?Ool0000g0004Ool00`00
Oomoo`03Ool0025oo`03001oogoo00=oo`8000ioo`0300moogoo01Aoo`0300moogoo01Aoo`0300mo
ogoo01=oo`0300moogoo01Aoo`0300moogoo01Aoo`0300moogoo01=oo`0300moogoo01Aoo`0300mo
ogoo015oobH000=oo`8000Eoo`008Goo00<007ooOol00goo00<007ooOol00Woo00<007ooOol027oo
00<03gooOol057oo00<03gooOol057oo00<007ooOol04goo00<03gooOol057oo00<03gooOol027oo
00<007ooOol02Goo00<03gooOol04goo00<03gooOol057oo00<007ooOol057oo00<03gooOol04goo
00<03gooOol02Goo00<007ooOol00Woo00<007ooOol00goo000QOol00`00Oomoo`03Ool00`00Oomo
o`02Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomo
o`09Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomo
o`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomo
o`08Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomo
o`09Ool00`00Oomoo`02Ool00`00Oomoo`03Ool0021oo`8000Eoonh000Eoo`00ogoo6goo0000\
\>"],
  ImageRangeCache->{{{0, 281}, {173, 0}} -> {-3.94814, 0.100336, 0.08837, 
  0.00533928}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
This shows how many bits Alice and Bob must compare in order to \
ensure that they can detect the presence of an eavesdropper with a \
probability greater than or equal to p.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["\<\
Plot[NumberOfBitsForDetectability[p], {p, 0, 0.999},
\t\tPlotPoints->200,
\t \tFrame->True,
     \tFrameLabel->{\"Detectability Probability p\", \"Number of Bits \
Needed\"},
     \tPlotStyle->Thickness[0.01]];\
\>", "Input",
  AspectRatioFixed->True],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.953334 0.0147151 0.0117721 [
[.02381 -0.0125 -3 -9 ]
[.02381 -0.0125 3 0 ]
[.21448 -0.0125 -9 -9 ]
[.21448 -0.0125 9 0 ]
[.40514 -0.0125 -9 -9 ]
[.40514 -0.0125 9 0 ]
[.59581 -0.0125 -9 -9 ]
[.59581 -0.0125 9 0 ]
[.78648 -0.0125 -9 -9 ]
[.78648 -0.0125 9 0 ]
[.97714 -0.0125 -3 -9 ]
[.97714 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[.5 -0.0125 -83 -21.5625 ]
[.5 -0.0125 83 -9 ]
[-0.0125 .01472 -6 -4.5 ]
[-0.0125 .01472 0 4.5 ]
[-0.0125 .13244 -12 -4.5 ]
[-0.0125 .13244 0 4.5 ]
[-0.0125 .25016 -12 -4.5 ]
[-0.0125 .25016 0 4.5 ]
[-0.0125 .36788 -12 -4.5 ]
[-0.0125 .36788 0 4.5 ]
[-0.0125 .4856 -12 -4.5 ]
[-0.0125 .4856 0 4.5 ]
[-0.0125 .60332 -12 -4.5 ]
[-0.0125 .60332 0 4.5 ]
[ 0 0 -0.125 0 ]
[-0.01875 .30902 -24.5625 -65 ]
[-0.01875 .30902 -12 65 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.02381 0 m
.02381 .00625 L
s
[(0)] .02381 -0.0125 0 1 Mshowa
.21448 0 m
.21448 .00625 L
s
[(0.2)] .21448 -0.0125 0 1 Mshowa
.40514 0 m
.40514 .00625 L
s
[(0.4)] .40514 -0.0125 0 1 Mshowa
.59581 0 m
.59581 .00625 L
s
[(0.6)] .59581 -0.0125 0 1 Mshowa
.78648 0 m
.78648 .00625 L
s
[(0.8)] .78648 -0.0125 0 1 Mshowa
.97714 0 m
.97714 .00625 L
s
[(1)] .97714 -0.0125 0 1 Mshowa
.125 Mabswid
.07148 0 m
.07148 .00375 L
s
.11914 0 m
.11914 .00375 L
s
.16681 0 m
.16681 .00375 L
s
.26214 0 m
.26214 .00375 L
s
.30981 0 m
.30981 .00375 L
s
.35748 0 m
.35748 .00375 L
s
.45281 0 m
.45281 .00375 L
s
.50048 0 m
.50048 .00375 L
s
.54814 0 m
.54814 .00375 L
s
.64348 0 m
.64348 .00375 L
s
.69114 0 m
.69114 .00375 L
s
.73881 0 m
.73881 .00375 L
s
.83414 0 m
.83414 .00375 L
s
.88181 0 m
.88181 .00375 L
s
.92948 0 m
.92948 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
gsave
.5 -0.0125 -144 -25.5625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(Detectability) show
147.000000 12.812500 moveto
(Probability) show
219.000000 12.812500 moveto
(p) show
225.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
0 .01472 m
.00625 .01472 L
s
[(0)] -0.0125 .01472 1 0 Mshowa
0 .13244 m
.00625 .13244 L
s
[(10)] -0.0125 .13244 1 0 Mshowa
0 .25016 m
.00625 .25016 L
s
[(20)] -0.0125 .25016 1 0 Mshowa
0 .36788 m
.00625 .36788 L
s
[(30)] -0.0125 .36788 1 0 Mshowa
0 .4856 m
.00625 .4856 L
s
[(40)] -0.0125 .4856 1 0 Mshowa
0 .60332 m
.00625 .60332 L
s
[(50)] -0.0125 .60332 1 0 Mshowa
.125 Mabswid
0 .03826 m
.00375 .03826 L
s
0 .0618 m
.00375 .0618 L
s
0 .08535 m
.00375 .08535 L
s
0 .10889 m
.00375 .10889 L
s
0 .15598 m
.00375 .15598 L
s
0 .17952 m
.00375 .17952 L
s
0 .20307 m
.00375 .20307 L
s
0 .22661 m
.00375 .22661 L
s
0 .2737 m
.00375 .2737 L
s
0 .29724 m
.00375 .29724 L
s
0 .32079 m
.00375 .32079 L
s
0 .34433 m
.00375 .34433 L
s
0 .39142 m
.00375 .39142 L
s
0 .41497 m
.00375 .41497 L
s
0 .43851 m
.00375 .43851 L
s
0 .46205 m
.00375 .46205 L
s
0 .50914 m
.00375 .50914 L
s
0 .53269 m
.00375 .53269 L
s
0 .55623 m
.00375 .55623 L
s
0 .57977 m
.00375 .57977 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
gsave
-0.01875 .30902 -144.281 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
126 10.2813 translate
90 rotate
-126 -10.2813 translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(Number) show
105.000000 12.812500 moveto
(of) show
123.000000 12.812500 moveto
(Bits) show
153.000000 12.812500 moveto
(Needed) show
189.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
.02381 .61178 m
.02381 .61803 L
s
.21448 .61178 m
.21448 .61803 L
s
.40514 .61178 m
.40514 .61803 L
s
.59581 .61178 m
.59581 .61803 L
s
.78648 .61178 m
.78648 .61803 L
s
.97714 .61178 m
.97714 .61803 L
s
.125 Mabswid
.07148 .61428 m
.07148 .61803 L
s
.11914 .61428 m
.11914 .61803 L
s
.16681 .61428 m
.16681 .61803 L
s
.26214 .61428 m
.26214 .61803 L
s
.30981 .61428 m
.30981 .61803 L
s
.35748 .61428 m
.35748 .61803 L
s
.45281 .61428 m
.45281 .61803 L
s
.50048 .61428 m
.50048 .61803 L
s
.54814 .61428 m
.54814 .61803 L
s
.64348 .61428 m
.64348 .61803 L
s
.69114 .61428 m
.69114 .61803 L
s
.73881 .61428 m
.73881 .61803 L
s
.83414 .61428 m
.83414 .61803 L
s
.88181 .61428 m
.88181 .61803 L
s
.92948 .61428 m
.92948 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .01472 m
1 .01472 L
s
.99375 .13244 m
1 .13244 L
s
.99375 .25016 m
1 .25016 L
s
.99375 .36788 m
1 .36788 L
s
.99375 .4856 m
1 .4856 L
s
.99375 .60332 m
1 .60332 L
s
.125 Mabswid
.99625 .03826 m
1 .03826 L
s
.99625 .0618 m
1 .0618 L
s
.99625 .08535 m
1 .08535 L
s
.99625 .10889 m
1 .10889 L
s
.99625 .15598 m
1 .15598 L
s
.99625 .17952 m
1 .17952 L
s
.99625 .20307 m
1 .20307 L
s
.99625 .22661 m
1 .22661 L
s
.99625 .2737 m
1 .2737 L
s
.99625 .29724 m
1 .29724 L
s
.99625 .32079 m
1 .32079 L
s
.99625 .34433 m
1 .34433 L
s
.99625 .39142 m
1 .39142 L
s
.99625 .41497 m
1 .41497 L
s
.99625 .43851 m
1 .43851 L
s
.99625 .46205 m
1 .46205 L
s
.99625 .50914 m
1 .50914 L
s
.99625 .53269 m
1 .53269 L
s
.99625 .55623 m
1 .55623 L
s
.99625 .57977 m
1 .57977 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.01 w
.02381 .03826 m
.02847 .03826 L
.03355 .03826 L
.03832 .03826 L
.04291 .03826 L
.0478 .03826 L
.0525 .03826 L
.0575 .03826 L
.06232 .03826 L
.06695 .03826 L
.07188 .03826 L
.07662 .03826 L
.08119 .03826 L
.08604 .03826 L
.09072 .03826 L
.09569 .03826 L
.10048 .03826 L
.10509 .03826 L
.10999 .03826 L
.11471 .03826 L
.11972 .03826 L
.12456 .03826 L
.1292 .03826 L
.13415 .03826 L
.13891 .03826 L
.14349 .03826 L
.14837 .03826 L
.15306 .03826 L
.15805 .03826 L
.16285 .03826 L
.16748 .03826 L
.17239 .03826 L
.17713 .03826 L
.18168 .03826 L
.18653 .03826 L
.1912 .03826 L
.19616 .03826 L
.20094 .03826 L
.20553 .03826 L
.21043 .03826 L
.21513 .03826 L
.22014 .03826 L
.22496 .03826 L
.2296 .03826 L
.23454 .03826 L
.23929 .03826 L
.24386 .03826 L
.24872 .03826 L
.2534 .03826 L
.25579 .03826 L
Mistroke
.25838 .03826 L
.25959 .03826 L
.26087 .03826 L
.26149 .03826 L
.2618 .03826 L
.26193 .03826 L
.26208 .03826 L
.26221 .0618 L
.26234 .0618 L
.26261 .0618 L
.26318 .0618 L
.26788 .0618 L
.27287 .0618 L
.27768 .0618 L
.28231 .0618 L
.28724 .0618 L
.29198 .0618 L
.29654 .0618 L
.30139 .0618 L
.30606 .0618 L
.31103 .0618 L
.31582 .0618 L
.32042 .0618 L
.32532 .0618 L
.33003 .0618 L
.33504 .0618 L
.33987 .0618 L
.34452 .0618 L
.34946 .0618 L
.35422 .0618 L
.35879 .0618 L
.36366 .0618 L
.36835 .0618 L
.37334 .0618 L
.37814 .0618 L
.38276 .0618 L
.38767 .0618 L
.3924 .0618 L
.39695 .0618 L
.4018 .0618 L
.40646 .0618 L
.41142 .0618 L
.41619 .0618 L
.42078 .0618 L
.42567 .0618 L
.43038 .0618 L
.43538 .0618 L
.43788 .0618 L
.43898 .0618 L
.43957 .0618 L
Mistroke
.4402 .0618 L
.44051 .0618 L
.44066 .0618 L
.4408 .0618 L
.44093 .08535 L
.44107 .08535 L
.44136 .08535 L
.44245 .08535 L
.44483 .08535 L
.44985 .08535 L
.45468 .08535 L
.45933 .08535 L
.46428 .08535 L
.46905 .08535 L
.47363 .08535 L
.47851 .08535 L
.4832 .08535 L
.48819 .08535 L
.493 .08535 L
.49763 .08535 L
.50255 .08535 L
.50728 .08535 L
.51184 .08535 L
.51669 .08535 L
.52136 .08535 L
.52632 .08535 L
.5311 .08535 L
.5357 .08535 L
.5406 .08535 L
.54531 .08535 L
.55031 .08535 L
.55514 .08535 L
.55978 .08535 L
.56472 .08535 L
.56947 .08535 L
.57186 .08535 L
.57291 .08535 L
.57404 .08535 L
.57435 .08535 L
.57451 .08535 L
.57467 .08535 L
.57482 .08535 L
.57497 .10889 L
.57512 .10889 L
.57525 .10889 L
.57639 .10889 L
.57891 .10889 L
.58368 .10889 L
.58827 .10889 L
.59316 .10889 L
Mistroke
.59786 .10889 L
.60285 .10889 L
.60767 .10889 L
.6123 .10889 L
.61723 .10889 L
.62197 .10889 L
.62653 .10889 L
.63139 .10889 L
.63606 .10889 L
.64103 .10889 L
.64582 .10889 L
.65043 .10889 L
.65533 .10889 L
.66004 .10889 L
.66506 .10889 L
.66989 .10889 L
.6721 .10889 L
.67336 .10889 L
.67398 .10889 L
.67454 .10889 L
.67482 .10889 L
.67513 .10889 L
.67527 .10889 L
.67542 .10889 L
.67555 .13244 L
.6757 .13244 L
.67694 .13244 L
.67948 .13244 L
.68433 .13244 L
.68899 .13244 L
.69395 .13244 L
.69873 .13244 L
.70332 .13244 L
.70821 .13244 L
.71292 .13244 L
.71792 .13244 L
.72274 .13244 L
.72738 .13244 L
.73232 .13244 L
.73707 .13244 L
.74163 .13244 L
.74418 .13244 L
.7465 .13244 L
.74765 .13244 L
.7489 .13244 L
.74947 .13244 L
.75008 .13244 L
.75038 .13244 L
.75066 .13244 L
.75078 .13244 L
Mistroke
.75091 .13244 L
.75105 .15598 L
.75118 .15598 L
.75576 .15598 L
.76064 .15598 L
.76534 .15598 L
.77033 .15598 L
.77514 .15598 L
.77977 .15598 L
.78469 .15598 L
.78944 .15598 L
.79399 .15598 L
.79885 .15598 L
.80352 .15598 L
.80594 .15598 L
.80653 .15598 L
.80685 .15598 L
.80715 .15598 L
.80733 .15598 L
.80742 .15598 L
.80751 .17952 L
.80767 .17952 L
.80784 .17952 L
.80848 .17952 L
.81306 .17952 L
.81793 .17952 L
.82261 .17952 L
.8276 .17952 L
.8324 .17952 L
.83701 .17952 L
.84193 .17952 L
.84437 .17952 L
.84666 .17952 L
.84772 .17952 L
.84884 .17952 L
.84943 .17952 L
.84959 .17952 L
.84968 .17952 L
.84977 .17952 L
.84984 .17952 L
.84993 .20307 L
.85008 .20307 L
.85121 .20307 L
.85613 .20307 L
.86088 .20307 L
.86545 .20307 L
.87031 .20307 L
.87498 .20307 L
.87736 .20307 L
.87862 .20307 L
Mistroke
.87995 .20307 L
.88063 .20307 L
.88096 .20307 L
.88126 .20307 L
.88153 .20307 L
.88167 .20307 L
.88183 .22661 L
.88198 .22661 L
.88214 .22661 L
.88244 .22661 L
.88474 .22661 L
.88973 .22661 L
.89454 .22661 L
.89916 .22661 L
.9015 .22661 L
.90274 .22661 L
.90408 .22661 L
.9047 .22661 L
.90504 .22661 L
.90521 .22661 L
.90536 .22661 L
.9055 .22661 L
.90563 .25016 L
.90577 .25016 L
.90592 .25016 L
.90653 .25016 L
.90882 .25016 L
.91376 .25016 L
.91851 .25016 L
.9209 .25016 L
.92194 .25016 L
.92249 .25016 L
.92308 .25016 L
.92325 .25016 L
.92341 .25016 L
.92355 .2737 L
.9237 .2737 L
.92428 .2737 L
.92543 .2737 L
.92795 .2737 L
.93031 .2737 L
.9329 .2737 L
.93409 .2737 L
.93537 .2737 L
.93598 .2737 L
.93629 .2737 L
.93657 .2737 L
.9367 .2737 L
.93683 .2737 L
.93695 .29724 L
Mistroke
.9371 .29724 L
.93767 .29724 L
.94234 .29724 L
.94356 .29724 L
.94489 .29724 L
.94549 .29724 L
.94614 .29724 L
.94646 .29724 L
.94661 .29724 L
.94675 .29724 L
.94688 .29724 L
.94702 .32079 L
.94716 .32079 L
.94731 .32079 L
.94986 .32079 L
.95218 .32079 L
.95333 .32079 L
.95365 .32079 L
.95399 .32079 L
.95427 .32079 L
.95435 .32079 L
.95444 .32079 L
.95459 .34433 L
.95473 .34433 L
.95487 .34433 L
.95517 .34433 L
.95578 .34433 L
.95687 .34433 L
.95814 .34433 L
.95876 .34433 L
.95934 .34433 L
.95986 .34433 L
.95998 .34433 L
.96012 .34433 L
.96027 .36788 L
.96041 .36788 L
.96158 .36788 L
.9628 .36788 L
.96344 .36788 L
.9638 .36788 L
.96397 .36788 L
.96413 .36788 L
.96428 .36788 L
.96436 .36788 L
.96444 .39142 L
.96453 .39142 L
.96462 .39142 L
.96479 .39142 L
.96539 .39142 L
.96655 .39142 L
Mistroke
.96686 .39142 L
.96719 .39142 L
.96733 .39142 L
.96747 .39142 L
.96755 .39142 L
.96763 .41497 L
.96778 .41497 L
.96842 .41497 L
.96912 .41497 L
.96941 .41497 L
.96958 .41497 L
.96973 .41497 L
.96987 .41497 L
.97003 .43851 L
.97017 .43851 L
.9703 .43851 L
.9709 .43851 L
.97123 .43851 L
.97139 .43851 L
.97154 .43851 L
.97168 .43851 L
.97183 .46205 L
.97191 .46205 L
.97199 .46205 L
.97214 .46205 L
.97243 .46205 L
.97269 .46205 L
.97294 .46205 L
.97306 .46205 L
.9732 .4856 L
.97334 .4856 L
.97349 .4856 L
.97376 .4856 L
.97391 .4856 L
.97399 .4856 L
.97407 .4856 L
.97416 .50914 L
.97425 .50914 L
.97441 .50914 L
.97456 .50914 L
.97471 .50914 L
.97479 .50914 L
.97488 .53269 L
.97503 .53269 L
.97518 .53269 L
.97532 .53269 L
.9754 .53269 L
.97548 .55623 L
.97563 .55623 L
.97577 .55623 L
Mistroke
.97592 .57977 L
.97605 .57977 L
.97619 .60332 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{282, 174},
  ImageMargins->{{34, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004J0000[Q000`40O003h00Oogoo6goo003oOolK
Ool00?mooa]oo`00QWoo0P00B7oo0P004goo0`00<Woo0027Ool00`00Oomoo`17Ool00`00Oomoo`0B
Ool00`00Oomoo`0aOol004Uoo`@000=oo`8000=oo`8000=oo`80009oo`<000=oo`80009oo`P0009o
o`<0009oo`<0009oo`<000=oo`8000=oo`03001oogoo01=oo`<000=oo`<000=oo`800003Ool00000
0080009oo`P0009oo`<0009oo`<0009oo`<000=oo`8000=oo`03001oogoo019oo`<0035oo`00BWoo
00H007ooOol007oo0004Ool01P00Oomoo`00Ool000Aoo`03001oogoo009oo`05001oogoo001oo`03
0002Ool01000Oomoo`000Woo00<007ooOol00Woo00<007ooOol00Woo00D007ooOomoo`00009oo`05
001oo`00Ool0000EOol00`00Oomoo`03Ool01@00Oomoogoo00000Woo00<007oo00000Woo00<007oo
00000P000Woo00@007ooOol0009oo`03001oogoo009oo`03001oogoo009oo`05001oogooOol00002
Ool01@00Ool007oo00004goo00@007ooOol0031oo`00BWoo00D007ooOol007oo00<0009oo`03001o
ogoo009oo`<0009oo`03001oogoo009oo`03001oogoo00Aoo`04001oogoo0002Ool01000Oomoo`00
17oo00<007ooOol00Woo00D007ooOomoo`0000Aoo`03001oo`0001Eoo`<000=oo`05001oogooOol0
0002Ool00`00Ool00002Ool01@00Oomoogoo00000Woo00@007ooOol0009oo`03001oogoo009oo`03
001oogoo009oo`05001oogooOol00004Ool00`00Ool0000COol01000Oomoo`00<7oo001:Ool01@00
Oomoo`00Ool00`0000=oo`0000000P000Woo0`000Woo0`0000=oo`0000000P000goo0P000Woo0`00
0Woo0P0017oo00@007ooOomoo`80009oo`@00004Ool00000Ool2000DOol01@00Oomoo`00Ool01000
0Woo0P000Woo0`000goo0P000Woo0`000Woo0P0017oo00@007ooOomoo`80009oo`@00004Ool00000
Ool2000AOol4000aOol004Yoo`04001oogoo0006Ool00`00Oomoo`0<Ool00`00Oomoo`07Ool00`00
Oomoo`08Ool00`00Oomoo`06Ool00`00Oomoo`0JOol01000Oomoo`002goo00<007ooOol01goo00<0
07ooOol027oo00<007ooOol01Woo00<007ooOol0C7oo0019Ool4000OOol20005Ool01000Oomoogoo
0P0017oo00<007ooOol07Goo10002goo0P0027oo0P001Goo00@007ooOomoo`8000Aoo`03001oogoo
051oo`00ogoo6goo003oOolKOol00?mooa]oo`00:7oo00<007ooOol09Woo00<007ooOol00Woo00@0
07ooOomoo`<0025oo`03001oogoo009oo`03001oogoo009oo`<0025oo`03001oogoo009oo`04001o
ogooOol2000ROol00`00Oomoo`02Ool00`00Oomoo`02Ool00`00Oomoo`0TOol5000;Ool002Moo`03
001oo`0002Ioo`03001oo`0000Moo`03001oogoo021oo`03001oo`0000Uoo`03001oogoo01moo`03
001oo`0000Moo`03001oo`00021oo`03001oo`0000Moo`03001oo`0002Moo`03001oogoo00]oo`00
9goo00<007oo00009Woo00<007oo000027oo00<007ooOol07goo00<007oo00001goo100087oo00<0
07oo00001goo00<007oo000087oo00<007oo00001goo00<007oo00009goo00<007ooOol02goo000W
Ool00`00Ool0000VOol00`00Ool00008Ool00`00Oomoo`0OOol00`00Ool00007Ool00`00Ool0000Q
Ool00`00Ool00007Ool2000QOol00`00Ool00008Ool00`00Oomoo`0VOol00`00Oomoo`0;Ool002Mo
o`03001oo`0002Ioo`03001oo`0000Moo`03001oo`00021oo`03001oo`0000Qoo`80025oo`03001o
o`0000Moo`03001oogoo021oo`03001oo`0000Moo`03001oo`0002Moo`03001oogoo00]oo`00:7oo
00<007ooOol09Woo00<007ooOol01goo00<007ooOol087oo00<007ooOol027oo00<007ooOol087oo
00<007ooOol01goo0P008Goo00<007ooOol01goo00<007ooOol09Goo0P003Goo003oOolKOol00?mo
oa]oo`00ogoo6goo003oOolKOol00?mooa]oo`008gool0001goo000SOol00`00Oomoo`03Ool00`00
Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00
Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00
Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00
Oomoo`09Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00
Oomoo`03Ool00`00Oomoo`05Ool001eoo`03001oogoo00=oo`03001oogoo00=oo`03001oogoo02Yo
o`03001oogoo02]oo`03001oogoo02Yoo`03001oogoo02]oo`03001oogoo02Yoo`03001oogoo00=o
o`03001oogoo00Eoo`0077oo00<007oo000017oo00<007ooOol0k7oo00<007ooOol01Goo000LOol0
0`00Ool00004Ool3003ZOol30007Ool001aoo`03001oo`0000Aoo`03001oogoo0>aoo`03001oogoo
00Eoo`0077oo00<007oo000017oo00<007ooOol0k7oo00<007ooOol01Goo000MOol00`00Oomoo`03
Ool00`00Oomoo`3/Ool00`00Oomoo`05Ool002=oo`03001oogoo0>aoo`03001oogoo00Eoo`008goo
0P000goo>`00[Woo0P001goo000SOol00`00Oomoo`02Oolk002_Ool00`00Oomoo`05Ool002=oo`03
001oogoo03]oo`800:moo`03001oogoo00Eoo`0027oo00<007ooOol00goo00<007ooOol04Woo00<0
07ooOol0>goo0P00[goo00<007ooOol01Goo0008Ool7000DOol00`00Oomoo`0kOol2002_Ool00`00
Oomoo`05Ool000Qoo`<000=oo`03001oogoo019oo`03001oogoo03]oo`8002Uoo`03001oogoo08=o
o`03001oogoo00Eoo`002goo0P005Woo0P00?7oo;000Q7oo0P001goo0008Ool00`00Oomoo`02Ool2
000DOol00`00Oomoo`0kOol/0025Ool00`00Oomoo`05Ool000Qoo`L001Aoo`03001oogoo06Eoo`80
08Eoo`03001oogoo00Eoo`0027oo00<007ooOol067oo00<007ooOol0IGoo0P00QGoo00<007ooOol0
1Goo000:Ool00`00Oomoo`0FOol00`00Oomoo`1UOol20025Ool00`00Oomoo`05Ool000Yoo`@001Eo
o`8006Ioob8006Aoo`8000Moo`003Woo00<007ooOol04Woo00<007ooOol0IGoo8P00IGoo00<007oo
Ool01Goo000>Ool00`00Oomoo`0BOol00`00Oomoo`25Ool2001UOol00`00Oomoo`05Ool000Yoo`04
001oogoo000EOol00`00Oomoo`25Ool2001UOol00`00Oomoo`05Ool000Yoo`D001Aoo`03001oogoo
08Eoo`8006Eoo`03001oogoo00Eoo`003Woo00<007ooOol04Woo00<007ooOol0QGoo0P00IGoo00<0
07ooOol01Goo000:Ool01@00Oomoogoo000057oo0P00QWoo6P00C7oo0P001goo000:Ool5000DOol0
0`00Oomoo`25OolJ001=Ool00`00Oomoo`05Ool000Yoo`05001oogooOol0000DOol00`00Oomoo`2M
Ool2001=Ool00`00Oomoo`05Ool000]oo`@001Aoo`03001oogoo09eoo`8004eoo`03001oogoo00Eo
o`002Woo00D007ooOomoo`0000Moo`D0009oo`03001oogoo00=oo`03001oogoo09eoo`8004eoo`03
001oogoo00Eoo`002goo10002Goo00L007ooOomoo`00Ool00004Ool00`00Oomoo`2MOol2000AOol0
0`00Oomoo`0iOol00`00Oomoo`05Ool000ioo`03001oogoo00Moo`07001oogooOol007oo000017oo
0`00WGoo5000>Goo0`001goo0008Ool00`00Oomoo`03Ool00`00Oomoo`07Ool01`00Oomoogoo001o
o`0000Aoo`03001oogoo09eooa@003]oo`03001oogoo00Eoo`0027oo1`002Goo00L007ooOomoo`00
Ool00004Ool00`00Oomoo`2_Ool2000kOol00`00Oomoo`05Ool000Yoo`04001oogoo0009Ool20004
Ool00`00Oomoo`03Ool00`00Oomoo`2_Ool2000kOol00`00Oomoo`05Ool000Yoo`05001oogooOol0
000DOol00`00Oomoo`2_Ool2000kOol00`00Oomoo`05Ool000Yoo`05001oogooOol0000DOol2002`
Ool@000/Ool20007Ool000]oo`<001Eoo`03001oogoo0:mooa0002eoo`03001oogoo00Eoo`008goo
00<007ooOol0_Goo0P00;Goo00<007ooOol01Goo000SOol00`00Oomoo`2mOol2000]Ool00`00Oomo
o`05Ool000]oo`<001Eoo`03001oogoo0;eoo`8002eoo`03001oogoo00Eoo`002Woo00D007oo001o
o`0001Aoo`03001oogoo0;eoo`8002eoo`03001oogoo00Eoo`002Woo00D007oo001oo`0001Aoo`80
0;ioo``0029oo`8000Moo`002Woo00D007oo001oo`0001Aoo`03001oogoo0;eoo``002=oo`03001o
ogoo00Eoo`002goo0P0000=oo`00Ool04goo00<007ooOol0agoo0P008goo00<007ooOol01Goo000S
Ool00`00Oomoo`37Ool2000SOol00`00Oomoo`05Ool002=oo`03001oogoo0<Moo`8002=oo`03001o
ogoo00Eoo`002Woo00D007ooOomoo`0001Aoo`800<Qoo`X001Yoo`8000Moo`002Woo1@0057oo00<0
07ooOol0agoo2P006goo00<007ooOol01Goo000;Ool01000Oomoo`0057oo00<007ooOol0cgoo0P00
6goo00<007ooOol01Goo000:Ool01@00Oomoogoo000057oo00<007ooOol0cgoo0P006goo00<007oo
Ool01Goo000:Ool00`00Oomoo`0FOol00`00Oomoo`3?Ool2000KOol00`00Oomoo`05Ool002=oo`03
001oogoo0<moo`8001]oo`03001oogoo00Eoo`008goo0P00d7oo1`005Goo0P001goo000;Ool3000E
Ool00`00Oomoo`3?Ool7000FOol00`00Oomoo`05Ool000Yoo`05001oogooOol0000DOol00`00Oomo
o`3DOol2000FOol00`00Oomoo`05Ool000Yoo`05001oogooOol0000DOol00`00Oomoo`3DOol2000F
Ool00`00Oomoo`05Ool000Yoo`05001oogooOol00008Ool30003Ool00`00Oomoo`03Ool00`00Oomo
o`3DOol2000FOol00`00Oomoo`05Ool000]oo`<000Uoo`03001oogoo009oo`03001oo`0000Aoo`03
001oogoo0=Aoo`8000=oo`03001oogoo011oo`03001oogoo00Eoo`0067oo00L007ooOomoo`00Ool0
0004Ool3003DOol7000?Ool30007Ool001Qoo`07001oogooOol007oo000017oo00<007ooOol0e7oo
1`004Goo00<007ooOol01Goo000GOol00`00Ool00002Ool00`00Ool00004Ool00`00Oomoo`3IOol2
000AOol00`00Oomoo`05Ool000Yoo`05001oogooOol00009Ool00`00Oomoo`02Ool00`00Oomoo`03
Ool00`00Oomoo`3IOol2000AOol00`00Oomoo`05Ool000Uoo`H001Aoo`03001oogoo0=Uoo`80015o
o`03001oogoo00Eoo`0027oo00<007oo00000goo00<007ooOol04Woo0P00fWoo1@003Goo0P001goo
0008Ool00`00Ool00003Ool00`00Oomoo`0BOol00`00Oomoo`3IOol5000>Ool00`00Oomoo`05Ool0
02=oo`03001oogoo0=aoo`8000ioo`03001oogoo00Eoo`008goo00<007ooOol0g7oo0P003Woo00<0
07ooOol01Goo000SOol00`00Oomoo`3LOol2000>Ool00`00Oomoo`05Ool002=oo`03001oogoo0=ao
o`8000ioo`03001oogoo00Eoo`008goo0P00gGoo10002goo0P001goo0008Ool00`00Oomoo`03Ool0
0`00Oomoo`0BOol00`00Oomoo`3LOol4000<Ool00`00Oomoo`05Ool000Qoo`L001Aoo`03001oogoo
0=ioo`8000aoo`03001oogoo00Eoo`0027oo00@007ooOol0009oo`03001oogoo019oo`03001oogoo
0=ioo`8000aoo`03001oogoo00Eoo`0027oo00@007ooOol0009oo`03001oogoo019oo`03001oogoo
0=ioo`8000aoo`03001oogoo00Eoo`0027oo00@007ooOol0009oo`03001oogoo019oo`03001oogoo
0=ioo`<000]oo`03001oogoo00Eoo`002Goo0P0000=oo`0000005Goo0P00ggoo10002Goo0P001goo
000SOol00`00Oomoo`3NOol4000:Ool00`00Oomoo`05Ool002=oo`03001oogoo0>1oo`8000Yoo`03
001oogoo00Eoo`002Woo00D007ooOomoo`0001Aoo`03001oogoo0>1oo`8000Yoo`03001oogoo00Eo
o`002Woo00D007ooOomoo`0001Aoo`03001oogoo0>1oo`8000Yoo`03001oogoo00Eoo`0027oo00<0
07oo0000100057oo0P00hGoo0`0027oo0P001goo000>Ool00`00Oomoo`0BOol00`00Oomoo`3POol3
0009Ool00`00Oomoo`05Ool000ioo`03001oogoo019oo`03001oogoo0>5oo`8000Uoo`03001oogoo
00Eoo`008goo00<007ooOol0hGoo0P002Goo00<007ooOol01Goo000:Ool00`00Oomoo`0:Ool20004
Ool00`00Oomoo`03Ool00`00Oomoo`3QOol20009Ool00`00Oomoo`05Ool000Uoo`D000]oo`06001o
ogoo001oo`0017oo00<007ooOol0hGoo0P002Goo00<007ooOol01Goo000:Ool01@00Oomoogoo0000
2Goo00L007ooOomoo`00Ool00004Ool3003QOol30006Ool30007Ool000Yoo`05001oogooOol0000:
Ool01P00Oomoo`00Ool000Aoo`03001oogoo0>5oo`<000Qoo`03001oogoo00Eoo`002Woo00D007oo
Oomoo`0000Qoo`03001oo`00009oo`03001oo`0000Aoo`03001oogoo0>9oo`8000Qoo`03001oogoo
00Eoo`003Goo00<007ooOol027oo00<007ooOol00Woo00<007ooOol00goo00<007ooOol0hWoo0P00
27oo00<007ooOol01Goo000SOol00`00Oomoo`3ROol20008Ool00`00Oomoo`05Ool002=oo`800>=o
o`<000Ioo`8000Moo`002goo00@007ooOol001Aoo`03001oogoo0>9oo`<000Moo`03001oogoo00Eo
o`002Woo00D007oo001oo`0001Aoo`03001oogoo0>=oo`8000Moo`03001oogoo00Eoo`002Woo00D0
07oo001oo`0001Aoo`03001oogoo0>=oo`8000Moo`03001oogoo00Eoo`002Woo00D007oo001oo`00
01Aoo`03001oogoo0>=oo`8000Moo`03001oogoo00Eoo`002Woo00@007ooOol001Eoo`03001oogoo
0>=oo`8000Moo`03001oogoo00Eoo`008goo0P00i7oo0`001Goo0P001goo000SOol00`00Oomoo`3S
Ool30006Ool00`00Oomoo`05Ool002=oo`03001oogoo0>Aoo`8000Ioo`03001oogoo00Eoo`0027oo
00<007ooOol00goo00<007ooOol04Woo00<007ooOol0i7oo0P001Woo00<007ooOol01Goo0008Ool7
000DOol00`00Oomoo`3TOol20006Ool00`00Oomoo`05Ool000Qoo`<000=oo`03001oogoo019oo`03
001oogoo0>Aoo`8000Ioo`03001oogoo00Eoo`002goo0P005Woo0P00iGoo0P001Goo0P001goo0008
Ool00`00Oomoo`02Ool2000DOol00`00Oomoo`3TOol20006Ool00`00Oomoo`05Ool000Qoo`L001Ao
o`03001oogoo0>Aoo`8000Ioo`03001oogoo00Eoo`0027oo00<007ooOol067oo00<007ooOol0i7oo
0P001Woo00<007ooOol01Goo000SOol00`00Oomoo`3TOol20006Ool00`00Oomoo`05Ool000]oo`<0
01Eoo`800>Eoo`8000Eoo`8000Moo`002Woo00D007oo001oo`0001Aoo`03001oogoo0>Aoo`8000Io
o`03001oogoo00Eoo`002Woo00D007oo001oo`0001Aoo`03001oogoo0>Aoo`8000Ioo`03001oogoo
00Eoo`002Woo00D007oo001oo`0001Aoo`03001oogoo0>Aoo`8000Ioo`03001oogoo00Eoo`002goo
0P0000=oo`00Ool027oo0`000Woo00<007ooOol00goo00<007ooOol0i7oo0P001Woo00<007ooOol0
1Goo000IOol01P00Oomoo`00Ool000Aoo`03001oogoo0>Aoo`8000Ioo`03001oogoo00Eoo`005goo
100000Aoo`00Ool000Aoo`<00>Aoo`<000=oo`<000Moo`002goo0`002Goo00<007oo00000Woo00<0
07oo000017oo00<007ooOol0i7oo0`001Goo00<007ooOol01Goo000:Ool01@00Ool007oo00002Goo
0P000Woo00<007oo000017oo00<007ooOol0iGoo0P001Goo00<007ooOol01Goo000:Ool01@00Ool0
07oo00002Woo00D007ooOomoo`0000Eoo`03001oogoo0>Eoo`8000Eoo`03001oogoo00Eoo`002Woo
00D007oo001oo`0001Aoo`03001oogoo0>Eoo`8000Eoo`03001oogoo00Eoo`002goo0P0000=oo`00
Ool04goo00<007ooOol0iGoo0P001Goo00<007ooOol01Goo000SOol2003VOol20004Ool20007Ool0
02=oo`03001oogoo0>Eoo`8000Eoo`03001oogoo00Eoo`002goo0`005Goo00<007ooOol0iGoo0P00
1Goo00<007ooOol01Goo000:Ool01@00Oomoogoo000057oo00<007ooOol0iGoo0P001Goo00<007oo
Ool01Goo000:Ool01@00Oomoogoo000057oo00<007ooOol0iGoo0P001Goo00<007ooOol01Goo0008
Ool00`00Ool00002Ool00`00Oomoo`0COol2003VOol20004Ool20007Ool000Qoo`L001Aoo`03001o
ogoo0>Eoo`8000Eoo`03001oogoo00Eoo`003Woo00<007ooOol04Woo00<007ooOol0iGoo0P001Goo
00<007ooOol01Goo000SOol00`00Oomoo`3UOol20005Ool00`00Oomoo`05Ool000]oo`<001Eoo`03
001oogoo0>Eoo`8000Eoo`03001oogoo00Eoo`002Woo00D007oo001oo`0001Aoo`03001oogoo0>Eo
o`8000Eoo`03001oogoo00Eoo`002Woo00D007oo001oo`0001Aoo`800>Ioo`8000Aoo`8000Moo`00
2Woo00D007oo001oo`0001Aoo`03001oogoo0>Eoo`8000Eoo`03001oogoo00Eoo`002goo0P0000=o
o`00Ool04goo00<007ooOol0iGoo0P001Goo00<007ooOol01Goo000SOol00`00Oomoo`3UOol20005
Ool00`00Oomoo`05Ool002=oo`03001oogoo0>Eoo`8000Eoo`03001oogoo00Eoo`002goo0`005Goo
0P00iWoo0P0017oo0P001goo000:Ool01@00Oomoogoo000057oo00<007ooOol0iGoo0P001Goo00<0
07ooOol01Goo000:Ool01@00Oomoogoo000057oo00<007ooOol0iGoo0P001Goo00<007ooOol01Goo
0008Ool00`00Ool00002Ool00`00Oomoo`0COol00`00Oomoo`3UOol20005Ool00`00Oomoo`05Ool0
00Qoo`L000Uoo`03001oogoo009oo`03001oogoo00=oo`03001oogoo0>Eoo`8000Eoo`03001oogoo
00Eoo`003Woo00<007ooOol01Woo00<007oo00000Woo00<007oo000017oo00<007ooOol0iGoo0P00
1Goo00<007ooOol01Goo000IOol01P00Oomoo`00Ool000Aoo`<00>Eoo`8000=oo`<000Moo`005goo
0P000goo00<007oo000017oo00<007ooOol0iGoo0P001Goo00<007ooOol01Goo000GOol00`00Oomo
o`02Ool00`00Ool00004Ool00`00Oomoo`03Ool00`00Oomoo`0ZOol00`00Oomoo`0[Ool00`00Oomo
o`0ZOol00`00Oomoo`0[Ool00`00Oomoo`0ZOol00`00Oomoo`03Ool00`00Oomoo`05Ool001Moo`<0
00=oo`03001oogoo00=oo`03001oogoo00=oo`03001oogoo00Qoo`03001oogoo00Uoo`03001oogoo
00Qoo`03001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo
00Uoo`03001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo
00Uoo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00Qoo`03001oogoo
00Uoo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo00=oo`03001oogoo00Eoo`008gool000
1goo0000\
\>"],
  ImageRangeCache->{{{0, 281}, {173, 0}} -> {-0.18172, -9.98872, 0.00439623, 
  0.356018}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Encoding Bits Strings in Polarized Photons", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Code", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
EncodeBitString[bitString_]:=
\tEncodeBits[Map[ToExpression, Characters[bitString]]]

EncodeBits[bits_]:=
\tMap[{#, ChooseBasisForEncoding[], ket[#]}&, bits]
\t
ChooseBasisForEncoding[]:=
\tIf[CoinToss[]===Heads, RectilinearBasis, DiagonalBasis]
\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[" Encode the Bit String \"101101110100\"", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["EncodeBitString[\"101101110100\"]", "Input",
  AspectRatioFixed->True],

Cell[OutputFormData[
"\<\
{{1, DiagonalBasis, ket[1]}, {0, RectilinearBasis, ket[0]}, {1, \
RectilinearBasis, ket[1]}, 
  {1, DiagonalBasis, ket[1]}, {0, DiagonalBasis, ket[0]}, {1, DiagonalBasis, \
ket[1]}, {1, RectilinearBasis, ket[1]}, 
  {1, RectilinearBasis, ket[1]}, {0, RectilinearBasis, ket[0]}, {1, \
RectilinearBasis, ket[1]}, 
  {0, DiagonalBasis, ket[0]}, {0, RectilinearBasis, ket[0]}}\
\>", 
"\<\
{{1, DiagonalBasis, ket[1]}, {0, RectilinearBasis, ket[0]}, {1, \
RectilinearBasis, ket[1]}, 
 
  {1, DiagonalBasis, ket[1]}, {0, DiagonalBasis, ket[0]}, {1, DiagonalBasis, \
ket[1]}, {1, RectilinearBasis, ket[1]}, 
 
  {1, RectilinearBasis, ket[1]}, {0, RectilinearBasis, ket[0]}, {1, \
RectilinearBasis, ket[1]}, 
 
  {0, DiagonalBasis, ket[0]}, {0, RectilinearBasis, ket[0]}}\
\>"], "Output",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Simulating Quantum Key Distribution", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["?DistributeKeyUsingQKD", "Input",
  AspectRatioFixed->True],

Cell["\<\
DistributeKeyUsingQKD[p, n, options] generates a simulation of  \
quantum key distribution.  The desired probability of
   detecting  eavesdropping is p and the desired number of bits in the  \
secret shared key is n. If the option
   Eavesdropping->True is  given, then the actions of a malicious \
intermediary Eve are  simulated. If the option
   Eavesdropping->False is given, then  the simulator runs without Eve \
intervening.\
\>", "Print"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
The following code simulates quantum key distribution in the \
absence of eavesdropping. The final output is shows Alice's secret key (i.e. \
sequence of bits) and Bob's secret key. Both keys should be identical.\
\>", 
  "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["\<\
SeedRandom[123456];
DistributeKeyUsingQKD[0.75, 5, Eavesdropping->False]\
\>", "Input",
  AspectRatioFixed->True],

Cell["\<\
evePresentQ = False
{2, 3, 5, 6, 10, 11, 12, 16, 18, 19, 20, 24, 27, 28, 29}\
\>", "Print",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .10323 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.00245776 0.952381 [
[ 0 0 0 0 ]
[ 1 .10323 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .10323 L
0 .10323 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.00245776 0.97619 0.0331797 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
0 g
.06338 .02265 m
.07641 .00961 L
s
.0914 .01613 m
.10983 .01613 L
s
.12482 .02265 m
.13785 .00961 L
s
.15554 .02265 m
.16858 .00961 L
s
.18626 .02265 m
.1993 .00961 L
s
.21699 .00961 m
.23002 .02265 L
s
.24771 .00961 m
.26074 .02265 L
s
.27573 .01613 m
.29416 .01613 L
s
.30915 .00961 m
.32219 .02265 L
s
.33987 .02265 m
.35291 .00961 L
s
.37711 .00691 m
.37711 .02535 L
s
.40783 .00691 m
.40783 .02535 L
s
.43856 .00691 m
.43856 .02535 L
s
.46928 .00691 m
.46928 .02535 L
s
.49348 .02265 m
.50652 .00961 L
s
.5242 .00961 m
.53724 .02265 L
s
.56144 .00691 m
.56144 .02535 L
s
.59217 .00691 m
.59217 .02535 L
s
.61637 .00961 m
.6294 .02265 L
s
.64439 .01613 m
.66283 .01613 L
s
.68433 .00691 m
.68433 .02535 L
s
.70854 .00961 m
.72157 .02265 L
s
.73926 .00961 m
.75229 .02265 L
s
.7765 .00691 m
.7765 .02535 L
s
.80722 .00691 m
.80722 .02535 L
s
.83142 .00961 m
.84446 .02265 L
s
.86215 .02265 m
.87518 .00961 L
s
.89939 .00691 m
.89939 .02535 L
s
.92359 .02265 m
.93662 .00961 L
s
.95431 .02265 m
.96735 .00961 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0362519 0.97619 0.0669739 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
.5 Mabswid
.06338 .00961 m
.07641 .02265 L
s
.06338 .02265 m
.07641 .00961 L
s
.0914 .01613 m
.10983 .01613 L
s
.10061 .02535 m
.10061 .00691 L
s
.12482 .00961 m
.13785 .02265 L
s
.12482 .02265 m
.13785 .00961 L
s
.15554 .00961 m
.16858 .02265 L
s
.15554 .02265 m
.16858 .00961 L
s
.18626 .00961 m
.1993 .02265 L
s
.18626 .02265 m
.1993 .00961 L
s
.21699 .00961 m
.23002 .02265 L
s
.21699 .02265 m
.23002 .00961 L
s
.24771 .00961 m
.26074 .02265 L
s
.24771 .02265 m
.26074 .00961 L
s
.27573 .01613 m
.29416 .01613 L
s
.28495 .02535 m
.28495 .00691 L
s
.30915 .00961 m
.32219 .02265 L
s
.30915 .02265 m
.32219 .00961 L
s
.33987 .00961 m
.35291 .02265 L
s
.33987 .02265 m
.35291 .00961 L
s
.3679 .01613 m
.38633 .01613 L
s
.37711 .02535 m
.37711 .00691 L
s
.39862 .01613 m
.41705 .01613 L
s
.40783 .02535 m
.40783 .00691 L
s
.42934 .01613 m
.44777 .01613 L
s
.43856 .02535 m
.43856 .00691 L
s
.46006 .01613 m
.47849 .01613 L
s
.46928 .02535 m
.46928 .00691 L
s
.49348 .00961 m
.50652 .02265 L
s
.49348 .02265 m
.50652 .00961 L
s
.5242 .00961 m
.53724 .02265 L
s
.5242 .02265 m
.53724 .00961 L
s
.55223 .01613 m
.57066 .01613 L
s
.56144 .02535 m
.56144 .00691 L
s
.58295 .01613 m
.60138 .01613 L
s
.59217 .02535 m
.59217 .00691 L
s
.61637 .00961 m
.6294 .02265 L
s
.61637 .02265 m
.6294 .00961 L
s
.64439 .01613 m
.66283 .01613 L
s
.65361 .02535 m
.65361 .00691 L
s
.67512 .01613 m
.69355 .01613 L
s
.68433 .02535 m
.68433 .00691 L
s
.70854 .00961 m
.72157 .02265 L
s
.70854 .02265 m
.72157 .00961 L
s
.73926 .00961 m
.75229 .02265 L
s
.73926 .02265 m
.75229 .00961 L
s
.76728 .01613 m
.78571 .01613 L
s
.7765 .02535 m
.7765 .00691 L
s
.798 .01613 m
.81644 .01613 L
s
.80722 .02535 m
.80722 .00691 L
s
.83142 .00961 m
.84446 .02265 L
s
.83142 .02265 m
.84446 .00961 L
s
.86215 .00961 m
.87518 .02265 L
s
.86215 .02265 m
.87518 .00961 L
s
.89017 .01613 m
.9086 .01613 L
s
.89939 .02535 m
.89939 .00691 L
s
.92359 .00961 m
.93662 .02265 L
s
.92359 .02265 m
.93662 .00961 L
s
.95431 .00961 m
.96735 .02265 L
s
.95431 .02265 m
.96735 .00961 L
s
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0700461 0.97619 0.100768 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
[(1)] .06989 .01613 0 0 Mshowa
[(1)] .10061 .01613 0 0 Mshowa
[(1)] .13134 .01613 0 0 Mshowa
[(1)] .16206 .01613 0 0 Mshowa
[(1)] .19278 .01613 0 0 Mshowa
[(0)] .2235 .01613 0 0 Mshowa
[(0)] .25422 .01613 0 0 Mshowa
[(1)] .28495 .01613 0 0 Mshowa
[(0)] .31567 .01613 0 0 Mshowa
[(1)] .34639 .01613 0 0 Mshowa
[(0)] .37711 .01613 0 0 Mshowa
[(0)] .40783 .01613 0 0 Mshowa
[(0)] .43856 .01613 0 0 Mshowa
[(0)] .46928 .01613 0 0 Mshowa
[(1)] .5 .01613 0 0 Mshowa
[(0)] .53072 .01613 0 0 Mshowa
[(0)] .56144 .01613 0 0 Mshowa
[(0)] .59217 .01613 0 0 Mshowa
[(0)] .62289 .01613 0 0 Mshowa
[(1)] .65361 .01613 0 0 Mshowa
[(0)] .68433 .01613 0 0 Mshowa
[(0)] .71505 .01613 0 0 Mshowa
[(0)] .74578 .01613 0 0 Mshowa
[(0)] .7765 .01613 0 0 Mshowa
[(0)] .80722 .01613 0 0 Mshowa
[(0)] .83794 .01613 0 0 Mshowa
[(1)] .86866 .01613 0 0 Mshowa
[(0)] .89939 .01613 0 0 Mshowa
[(1)] .93011 .01613 0 0 Mshowa
[(1)] .96083 .01613 0 0 Mshowa
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{393.875, 40.625},
  ImageMargins->{{34, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`4000690000:1000`40O003h00OogooRWoo000ROooo
F]IAF]HGOol0029oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo01Eoo`008Woo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol00Woo00<007ooOol00goo00=JeWoo
Ool00Woo00<007ooOol00goo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00goo00<007oo
Ool00Woo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol00Woo00<007ooOol00goo00=JeWoo
Ool00Woo00<007ooOol00goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol00goo00<007oo
Ool00goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol00goo00<007ooOol00Woo00=JeWoo
Ool00goo00<007ooOol00Woo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol05Goo000ROol00e[FOomoo`05Ool0
1000Oomooe[F2Woo00=JeWooOol01Goo00@007ooOomJePMoo`04001oogooF]H7Ool01@00Oomoogoo
F]H00Woo00<007ooOol01Goo00AJeWooOol000Moo`03F]Ioogoo00Qoo`04F]Ioogoo0007Ool00e[F
Oomoo`05Ool01@00OomoogooF]H017oo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo
00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol01Goo
00D007ooOomooe[F009oo`03001oogoo00Eoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo
009oo`03001oogoo00=oo`04F]Ioogoo0007Ool00e[FOomoo`08Ool00e[FOomoo`03Ool00`00Oomo
o`03Ool015[FOomoo`001goo00AJeWooOol000Moo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Io
ogoo00=oo`03001oogoo009oo`05F]IoogooOol00007Ool00e[FOomoo`05Ool01000Oomooe[F17oo
00<007ooOol00goo00=JeWooOol01Goo00@007ooOomJePMoo`04001oogooF]HGOol0029oo`03F]Io
ogoo00Aoo`05001oogooOomJeP0:Ool00e[FOomoo`04Ool01@00OomoogooF]H01Woo00D007ooOomo
oe[F00Ioo`03001oogoo009oo`05F]IoogooOol00006Ool01E[FOomoogoo00001Woo00=JeWooOol0
27oo00EJeWooOomoo`0000Ioo`03F]Ioogoo00Aoo`03001oogoo009oo`03F]Ioogoo009oo`03001o
ogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Io
ogoo00=oo`03001oogoo009oo`03F]Ioogoo00Aoo`03001oogoo009oo`05F]IoogooOol00006Ool0
0e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01E[FOomoogoo0000
1Woo00=JeWooOol027oo00=JeWooOol00goo00<007ooOol00goo00EJeWooOomoo`0000Ioo`05F]Io
ogooOol00006Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool0
0e[FOomoo`02Ool00`00Oomoo`04Ool00e[FOomoo`04Ool01@00OomoogooF]H017oo00<007ooOol0
0goo00=JeWooOol017oo00D007ooOomooe[F00Ioo`05001oogooOomJeP0GOol0029oo`03F]Ioogoo
00=oo`03001oogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo
00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo
00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03001oogoo
00=oo`03F]Ioogoo00=oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo
009oo`03001oogoo00=oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo
009oo`03F]Ioogoo00=oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo
009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo
00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo
00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo
00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo00=oo`03F]Ioogoo00=oo`03001oogoo
009oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo
00=oo`03001oogoo009oo`03F]Ioogoo01Eoo`008Woo00=JeWooOol00Woo00<007ooOol00goo00=J
eWoo00001`0000=ooe[FOol00goo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol00goo00<007ooOol00goo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00goo00<0
07ooOol00Woo00=JeWoo00001`0000=ooe[FOol017oo00<007ooOol00Woo00=JeWooOol00goo00<0
07ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol00goo00<007ooOol00Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00goo00<0
07ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol00Woo00<007ooOol00goo00=JeWooOol00goo00<007ooOol00Woo00=JeWoo00001`0000=o
oe[FOol017oo00<007ooOol00goo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00goo00<0
07ooOol00Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00goo00<007ooOol00Woo00=J
eWooOol00goo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<0
07ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol05Goo000ROol01E[FOomoogoo00001Woo00=JeWooOol027oo00EJeWooOomoo`0000Ioo`05
F]IoogooOol00006Ool00e[FOomoo`02Ool00`00Oomoo`04Ool00e[FOomoo`04Ool01@00Oomoogoo
F]H01Woo00D007ooOomooe[F00Yoo`03F]Ioogoo00Aoo`05001oogooOomJeP04Ool00`00Oomoo`04
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`03
Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`02Ool00`00Oomoo`04
Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool00e[FOomoo`04Ool01@00OomoogooF]H02Woo00=JeWooOol00goo00<007oo
Ool00goo00=JeWooOol017oo00D007ooOomooe[F00Ioo`05001oogooOomJeP05Ool00`00Oomoo`02
Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`04Ool00`00Oomoo`02Ool01E[FOomoogoo
00001Woo00=JeWooOol00Woo00<007ooOol00goo00EJeWooOomoo`0000Ioo`05F]IoogooOol00006
Ool00e[FOomoo`0EOol0029oo`04F]Ioogoo0007Ool00e[FOomoo`08Ool015[FOomoo`001goo00AJ
eWooOol000Moo`05F]IoogooOol00007Ool00e[FOomoo`05Ool01000Oomooe[F1goo00@007ooOomJ
ePYoo`03F]Ioogoo00Eoo`04001oogooF]H3Ool00`00Oomoo`05Ool00e[FOomoo`02Ool00`00Oomo
o`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomo
o`03Ool00`00Oomoo`02Ool01E[FOomoogoo00001goo00=JeWooOol017oo00D007ooOomooe[F00Ao
o`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00Eoo`04001oogooF]H:
Ool00e[FOomoo`03Ool00`00Oomoo`03Ool00e[FOomoo`05Ool01000Oomooe[F1goo00@007ooOomJ
ePEoo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00Eoo`05001oogoo
OomJeP02Ool00`00Oomoo`05Ool00e[FOomoo`02Ool00`00Oomoo`03Ool015[FOomoo`001goo00AJ
eWooOol000Moo`03F]Ioogoo01Eoo`008Woo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol0
0Woo00<007ooOol00goo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00goo00<007ooOol0
0Woo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol0
0Woo00<007ooOol00goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol00goo00<007ooOol0
0goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol0
0goo00<007ooOol00Woo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol00Woo00<007ooOol0
0goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol05Goo000ROol00e[FOomoo`08Ool00e[F
Oomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[FOomoo`08Ool00e[F
Oomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[FOomoo`08Ool00e[F
Oomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[FOomoo`08Ool00e[F
Oomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[F
Oomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[F
Oomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`0EOol0029o
oomJeU5JeQMoo`00ogooRWoo000QOoooF]ICF]HFOol0025oo`03F]Ioogoo00Qoo`03F]Ioogoo00Uo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Uo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo01Aoo`008Goo00=JeWoo
Ool027oo00=JeWooOol00goo00=JeWooOol00goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol00Woo00=JeWooOol00goo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol00goo00=JeWooOol00goo00=JeWooOol00Woo00=JeWoo
Ool00goo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00goo00=JeWooOol00Woo00=JeWoo
Ool02Goo00=JeWooOol027oo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00goo00=JeWoo
Ool00Woo00=JeWooOol027oo00=JeWooOol00goo00=JeWooOol00goo00=JeWooOol00Woo00=JeWoo
Ool00goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol00goo00=JeWooOol00goo00=JeWoo
Ool00Woo00=JeWooOol00goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol00goo00=JeWoo
Ool00goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol057oo000QOol015[FOomooe[F17oo
00AJeWooOomJePEoo`03F]Ioogoo00=oo`04F]IoogooF]H4Ool015[FOomooe[F0Woo00=JeWooOol0
0Woo00AJeWooOomJeP9oo`03F]Ioogoo009oo`04F]IoogooF]H2Ool00e[FOomoo`02Ool015[FOomo
oe[F0goo00EJeWooOomooe[F00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`04F]IoogooF]H4Ool0
15[FOomooe[F0Woo00=JeWooOol00Woo00AJeWooOomJePEoo`03F]Ioogoo00=oo`03F]Ioogoo009o
o`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo009o
o`05F]IoogooOomJeP03Ool01E[FOomoogooF]H00Woo00=JeWooOol00Woo00AJeWooOomJePAoo`03
F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`04F]IoogooF]H4Ool015[FOomooe[F1Goo
00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00goo00AJeWooOomJePAoo`04F]IoogooF]H2
Ool00e[FOomoo`02Ool015[FOomooe[F1Goo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol0
0goo00AJeWooOomJePAoo`04F]IoogooF]H2Ool00e[FOomoo`02Ool015[FOomooe[F1Goo00=JeWoo
Ool00goo00AJeWooOomJePAoo`04F]IoogooF]H2Ool00e[FOomoo`02Ool015[FOomooe[F5Woo000Q
Ool01E[FOomoogooF]H00Woo00EJeWooOomooe[F00Eoo`03F]Ioogoo00=oo`05F]IoogooOomJeP02
Ool01E[FOomoogooF]H00goo00AJeWooOomJeP=oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H0
0goo00AJeWooOomJeP=oo`03F]Ioogoo009oo`03F]Iooe[F00Aoo`03F]Ioogoo009oo`03F]Ioogoo
00=oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H00goo00AJeWooOomJeP=oo`03F]Ioogoo00=o
o`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=o
o`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Iooe[F00Aoo`05F]IoogooOomJ
eP02Ool01E[FOomoogooF]H017oo00=JeWooOol00goo00=JeWooOol00goo00=JeWooOol00Woo00EJ
eWooOomooe[F009oo`05F]IoogooOomJeP05Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomo
o`03Ool01E[FOomoogooF]H00Woo00EJeWooOomooe[F00=oo`04F]IoogooF]H3Ool00e[FOomoo`03
Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool01E[FOomoogooF]H00Woo00EJeWoo
Oomooe[F00=oo`04F]IoogooF]H3Ool00e[FOomoo`03Ool00e[FOomoo`03Ool01E[FOomoogooF]H0
0Woo00EJeWooOomooe[F00=oo`04F]IoogooF]H3Ool00e[FOomoo`0DOol0025oo`03F]Ioogoo009o
o`9JePAoo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009o
o`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo00=oo`03F]Io
ogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009o
o`9JePAoo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Io
ogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Io
ogoo00=oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00=o
o`03F]Ioogoo009oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Io
ogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Io
ogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`9JePAo
o`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`9JePAo
o`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo01Aoo`008Goo00=JeWooOol00Woo0U[F17oo00=JeWoo
Ool01e[F0Woo00=JeWooOol00Woo0U[F17oo00=JeWooOol00Woo0U[F17oo00=JeWooOol00Woo0U[F
17oo00=JeWooOol00Woo0U[F17oo00=JeWooOol00goo00=JeWooOol00goo00=JeWooF]H01e[F00=o
oe[FOol00goo0U[F17oo00=JeWooOol00Woo0U[F17oo00=JeWooOol01e[F0Woo00=JeWooF]H01e[F
00=ooe[FOol025[F00=ooe[FOol025[F00=ooe[FOol017oo00=JeWooOol00goo00=JeWooOol00Woo
0U[F17oo00=JeWooF]H01e[F00=ooe[FOol025[F00=ooe[FOol00goo0U[F17oo00=JeWooOol01e[F
0Woo00=JeWooF]H01e[F00=ooe[FOol00goo0U[F17oo00=JeWooOol00Woo0U[F17oo00=JeWooOol0
1e[F0Woo00=JeWooF]H01e[F00=ooe[FOol00goo0U[F17oo00=JeWooOol00Woo0U[F17oo00=JeWoo
Ool01e[F0Woo00=JeWooOol00Woo0U[F17oo00=JeWooOol00Woo0U[F17oo00=JeWooOol057oo000Q
Ool01E[FOomoogooF]H00Woo00EJeWooOomooe[F00Eoo`03F]Ioogoo00=oo`05F]IoogooOomJeP02
Ool01E[FOomoogooF]H00goo00AJeWooOomJeP=oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H0
0goo00AJeWooOomJeP=oo`03F]Ioogoo009oo`03F]Iooe[F00Aoo`03F]Ioogoo009oo`03F]Ioogoo
00=oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H00goo00AJeWooOomJeP=oo`03F]Ioogoo00=o
o`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=o
o`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Iooe[F00Aoo`05F]IoogooOomJ
eP02Ool01E[FOomoogooF]H017oo00=JeWooOol00goo00=JeWooOol00goo00=JeWooOol00Woo00EJ
eWooOomooe[F009oo`05F]IoogooOomJeP05Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomo
o`03Ool01E[FOomoogooF]H00Woo00EJeWooOomooe[F00=oo`04F]IoogooF]H3Ool00e[FOomoo`03
Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool01E[FOomoogooF]H00Woo00EJeWoo
Oomooe[F00=oo`04F]IoogooF]H3Ool00e[FOomoo`03Ool00e[FOomoo`03Ool01E[FOomoogooF]H0
0Woo00EJeWooOomooe[F00=oo`04F]IoogooF]H3Ool00e[FOomoo`0DOol0025oo`04F]IoogooF]H4
Ool015[FOomooe[F1Goo00=JeWooOol00goo00AJeWooOomJePAoo`04F]IoogooF]H2Ool00e[FOomo
o`02Ool015[FOomooe[F0Woo00=JeWooOol00Woo00AJeWooOomJeP9oo`03F]Ioogoo009oo`04F]Io
ogooF]H3Ool01E[FOomoogooF]H00goo00=JeWooOol00Woo00=JeWooOol00goo00AJeWooOomJePAo
o`04F]IoogooF]H2Ool00e[FOomoo`02Ool015[FOomooe[F1Goo00=JeWooOol00goo00=JeWooOol0
0Woo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00goo00=JeWooOol0
0Woo00EJeWooOomooe[F00=oo`05F]IoogooOomJeP02Ool00e[FOomoo`02Ool015[FOomooe[F17oo
00=JeWooOol00goo00=JeWooOol00goo00=JeWooOol00Woo00AJeWooOomJePAoo`04F]IoogooF]H5
Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool015[FOomooe[F17oo00AJeWooOomJ
eP9oo`03F]Ioogoo009oo`04F]IoogooF]H5Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomo
o`03Ool015[FOomooe[F17oo00AJeWooOomJeP9oo`03F]Ioogoo009oo`04F]IoogooF]H5Ool00e[F
Oomoo`03Ool015[FOomooe[F17oo00AJeWooOomJeP9oo`03F]Ioogoo009oo`04F]IoogooF]HFOol0
025oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo
00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo009oo`03F]Ioogoo
00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo
009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo
009oo`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo
00=oo`03F]Ioogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo
009oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo
00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo
00=oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo01Aoo`008Goo00=J
eWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol057oo000QOoooF]ICF]HFOol00?moohYoo`008Woooe[FDE[F5goo000ROol00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`0E
Ool0029oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Io
ogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Io
ogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Io
ogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Io
ogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Io
ogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Io
ogoo00Qoo`03F]Ioogoo01Eoo`008Woo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol05Goo000ROol00e[FOomoo`050003Ool00e[F
Oomoo`050003Ool00e[FOomoo`050003Ool00e[FOomoo`050003Ool015[FOomoogoo1@000goo00=J
eWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol01@000goo
00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol01@0017oo00=JeWooOol00Woo00<007ooOol0
0goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol0
0Woo00<007ooOol00goo00=JeWooOol01@0017oo00=JeWooOol00Woo00<007ooOol00goo00=JeWoo
Ool00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol01@000goo00=JeWooOol00goo00<007ooOol00goo00=JeWooOol00Woo00<0
07ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol017oo00=JeWooOol01@000goo
00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol01@000goo00=JeWooOol01@000goo00=JeWoo
Ool05Goo000ROol00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool0
0e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`03Ool0
0`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol0
0Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`04Ool0
1e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool0
1e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001oogoo00Aoo`07F]IoogooOol007oo0000
17oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo0000
17oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007oo000017oo00MJeWooOomo
o`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomo
o`00Ool00004Ool01e[FOomoogoo001oo`0000Eoo`03F]Ioogoo009oo`03001oogoo00=oo`07F]Io
ogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol0
0goo00=JeWooOol05Goo000ROol00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00
Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[F
Oomoo`03Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo
00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00
Oomoo`04Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00
Ool00004Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001oogoo00Aoo`07F]Ioogoo
Ool007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07F]Ioogoo
Ool007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007oo000017oo
00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo
00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Eoo`03F]Ioogoo009oo`03001oogoo
00=oo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo
00<007ooOol00goo00=JeWooOol05Goo000ROol00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomo
o`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomo
o`03Ool00e[FOomoo`03Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol0
07oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[FOomo
o`02Ool00`00Oomoo`04Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJ
eWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001oogoo00Ao
o`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Ao
o`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<0
07oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol0
07oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Eoo`03F]Ioogoo009o
o`03001oogoo00=oo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol00Woo00<007ooOol00goo00=JeWooOol05Goo000ROol00e[FOomoo`02Ool00`00Oomoo`03
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool00e[FOomoo`03Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07
F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004
Ool00e[FOomoo`02Ool00`00Oomoo`04Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo
000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03
001oogoo00Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo
001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00=JeWoo
Ool00Woo00<007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07
F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Eoo`03
F]Ioogoo009oo`03001oogoo00=oo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol05Goo000ROol015[FOomoogoo0P00
1Goo00AJeWooOomoo`8000Eoo`04F]IoogooOol20005Ool015[FOomoogoo0P001Goo00=JeWooOol0
0Woo0P001Goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00AJ
eWooOomoo`8000Eoo`03F]Ioogoo009oo`03001oogoo00=oo`04F]IoogooOol20006Ool00e[FOomo
o`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomo
o`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool015[FOomoogoo0P001Woo00=JeWooOol00Woo00<0
07ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol00Woo00<007ooOol00goo00AJeWooOomoo`8000Eoo`03F]Ioogoo00=oo`03001oogoo00=o
o`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009o
o`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00Ao
o`04F]IoogooOol20005Ool00e[FOomoo`02Ool00`00Oomoo`03Ool015[FOomoogoo0P001Goo00AJ
eWooOomoo`8000Eoo`03F]Ioogoo01Eoo`008Woo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol05Goo000ROoooF]IAF]HGOol00?mo
ohYoo`00\
\>"],
  ImageRangeCache->{{{0, 392.875}, {39.625, 0}} -> {-0.0373468, -0.00258119, 
  0.00273546, 0.00273546}, {{13.625, 379.188}, {38.625, 
  26.875}} -> {-2.05242, -0.114373, 0.0893679, 0.0893679}, {{13.625, 
  379.188}, {25.6875, 13.875}} -> {-1.98835, -1.26689, 0.0890417, 
  0.0890417}, {{13.625, 379.188}, {12.6875, 0.9375}} -> {-2.05242, -2.43235, 
  0.0893679, 0.0893679}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .10323 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.00245776 0.952381 [
[ 0 0 0 0 ]
[ 1 .10323 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .10323 L
0 .10323 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.00245776 0.97619 0.0331797 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
[(0)] .06989 .01613 0 0 Mshowa
[(1)] .10061 .01613 0 0 Mshowa
[(1)] .13134 .01613 0 0 Mshowa
[(1)] .16206 .01613 0 0 Mshowa
[(1)] .19278 .01613 0 0 Mshowa
[(0)] .2235 .01613 0 0 Mshowa
[(0)] .25422 .01613 0 0 Mshowa
[(0)] .28495 .01613 0 0 Mshowa
[(1)] .31567 .01613 0 0 Mshowa
[(1)] .34639 .01613 0 0 Mshowa
[(0)] .37711 .01613 0 0 Mshowa
[(0)] .40783 .01613 0 0 Mshowa
[(0)] .43856 .01613 0 0 Mshowa
[(1)] .46928 .01613 0 0 Mshowa
[(0)] .5 .01613 0 0 Mshowa
[(0)] .53072 .01613 0 0 Mshowa
[(0)] .56144 .01613 0 0 Mshowa
[(0)] .59217 .01613 0 0 Mshowa
[(0)] .62289 .01613 0 0 Mshowa
[(1)] .65361 .01613 0 0 Mshowa
[(1)] .68433 .01613 0 0 Mshowa
[(1)] .71505 .01613 0 0 Mshowa
[(1)] .74578 .01613 0 0 Mshowa
[(0)] .7765 .01613 0 0 Mshowa
[(1)] .80722 .01613 0 0 Mshowa
[(1)] .83794 .01613 0 0 Mshowa
[(1)] .86866 .01613 0 0 Mshowa
[(0)] .89939 .01613 0 0 Mshowa
[(1)] .93011 .01613 0 0 Mshowa
[(0)] .96083 .01613 0 0 Mshowa
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0362519 0.97619 0.0669739 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
.5 Mabswid
.06068 .01613 m
.07911 .01613 L
s
.06989 .02535 m
.06989 .00691 L
s
.0914 .01613 m
.10983 .01613 L
s
.10061 .02535 m
.10061 .00691 L
s
.12482 .00961 m
.13785 .02265 L
s
.12482 .02265 m
.13785 .00961 L
s
.15284 .01613 m
.17127 .01613 L
s
.16206 .02535 m
.16206 .00691 L
s
.18626 .00961 m
.1993 .02265 L
s
.18626 .02265 m
.1993 .00961 L
s
.21699 .00961 m
.23002 .02265 L
s
.21699 .02265 m
.23002 .00961 L
s
.24501 .01613 m
.26344 .01613 L
s
.25422 .02535 m
.25422 .00691 L
s
.27843 .00961 m
.29146 .02265 L
s
.27843 .02265 m
.29146 .00961 L
s
.30645 .01613 m
.32488 .01613 L
s
.31567 .02535 m
.31567 .00691 L
s
.33987 .00961 m
.35291 .02265 L
s
.33987 .02265 m
.35291 .00961 L
s
.3679 .01613 m
.38633 .01613 L
s
.37711 .02535 m
.37711 .00691 L
s
.39862 .01613 m
.41705 .01613 L
s
.40783 .02535 m
.40783 .00691 L
s
.43204 .00961 m
.44507 .02265 L
s
.43204 .02265 m
.44507 .00961 L
s
.46276 .00961 m
.4758 .02265 L
s
.46276 .02265 m
.4758 .00961 L
s
.49078 .01613 m
.50922 .01613 L
s
.5 .02535 m
.5 .00691 L
s
.5242 .00961 m
.53724 .02265 L
s
.5242 .02265 m
.53724 .00961 L
s
.55493 .00961 m
.56796 .02265 L
s
.55493 .02265 m
.56796 .00961 L
s
.58295 .01613 m
.60138 .01613 L
s
.59217 .02535 m
.59217 .00691 L
s
.61637 .00961 m
.6294 .02265 L
s
.61637 .02265 m
.6294 .00961 L
s
.64439 .01613 m
.66283 .01613 L
s
.65361 .02535 m
.65361 .00691 L
s
.67781 .00961 m
.69085 .02265 L
s
.67781 .02265 m
.69085 .00961 L
s
.70584 .01613 m
.72427 .01613 L
s
.71505 .02535 m
.71505 .00691 L
s
.73656 .01613 m
.75499 .01613 L
s
.74578 .02535 m
.74578 .00691 L
s
.76728 .01613 m
.78571 .01613 L
s
.7765 .02535 m
.7765 .00691 L
s
.8007 .00961 m
.81374 .02265 L
s
.8007 .02265 m
.81374 .00961 L
s
.82873 .01613 m
.84716 .01613 L
s
.83794 .02535 m
.83794 .00691 L
s
.86215 .00961 m
.87518 .02265 L
s
.86215 .02265 m
.87518 .00961 L
s
.89017 .01613 m
.9086 .01613 L
s
.89939 .02535 m
.89939 .00691 L
s
.92359 .00961 m
.93662 .02265 L
s
.92359 .02265 m
.93662 .00961 L
s
.95161 .01613 m
.97005 .01613 L
s
.96083 .02535 m
.96083 .00691 L
s
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0700461 0.97619 0.100768 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
0 g
.06338 .02265 m
.07641 .00961 L
s
.0914 .01613 m
.10983 .01613 L
s
.12482 .02265 m
.13785 .00961 L
s
.15554 .02265 m
.16858 .00961 L
s
.18626 .02265 m
.1993 .00961 L
s
.21699 .00961 m
.23002 .02265 L
s
.24771 .00961 m
.26074 .02265 L
s
.27573 .01613 m
.29416 .01613 L
s
.30915 .00961 m
.32219 .02265 L
s
.33987 .02265 m
.35291 .00961 L
s
.37711 .00691 m
.37711 .02535 L
s
.40783 .00691 m
.40783 .02535 L
s
.43856 .00691 m
.43856 .02535 L
s
.46928 .00691 m
.46928 .02535 L
s
.49348 .02265 m
.50652 .00961 L
s
.5242 .00961 m
.53724 .02265 L
s
.56144 .00691 m
.56144 .02535 L
s
.59217 .00691 m
.59217 .02535 L
s
.61637 .00961 m
.6294 .02265 L
s
.64439 .01613 m
.66283 .01613 L
s
.68433 .00691 m
.68433 .02535 L
s
.70854 .00961 m
.72157 .02265 L
s
.73926 .00961 m
.75229 .02265 L
s
.7765 .00691 m
.7765 .02535 L
s
.80722 .00691 m
.80722 .02535 L
s
.83142 .00961 m
.84446 .02265 L
s
.86215 .02265 m
.87518 .00961 L
s
.89939 .00691 m
.89939 .02535 L
s
.92359 .02265 m
.93662 .00961 L
s
.95431 .02265 m
.96735 .00961 L
s
MathSubEnd
P
% End of sub-graphic
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{393.875, 40.625},
  ImageMargins->{{34, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`4000690000:1000`40O003h00OogooRWoo000ROooo
F]IAF]HGOol0029oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo01Eoo`008Woo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol05Goo000ROol00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`0EOol0
029oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00D000=oo`03F]Ioogoo00D000=oo`03
F]Ioogoo00D000=oo`04F]IoogooOol50003Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomo
o`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`050003Ool00e[F
Oomoo`050004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool0
0e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`050003Ool00e[FOomoo`02Ool00`00Oomoo`04
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`050003Ool015[FOomo
ogoo1@000goo00=JeWooOol01@000goo00=JeWooOol01@000goo00=JeWooOol00Woo00<007ooOol0
0goo00=JeWooOol01@000goo00=JeWooOol01@0017oo00=JeWooOol01@000goo00=JeWooOol00Woo
00<007ooOol00goo00=JeWooOol01@000goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol0
5Goo000ROol01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo
009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00=oo`03001oogoo
00=oo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`00
00Aoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00Aoo`07F]Ioogoo
Ool007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo
009oo`03001oogoo00=oo`07F]IoogooOol007oo00001Goo00MJeWooOomoo`00Ool00004Ool01e[F
Oomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool00e[F
Oomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`03Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00
Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo
009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00Aoo`03F]Ioogoo009oo`03001oogoo
00=oo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00
Ool00004Ool00e[FOomoo`0EOol0029oo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<0
07ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol00goo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Ao
o`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<0
07ooOol017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol0
07oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00005Ool01e[FOomo
ogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomo
ogoo001oo`0000Aoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00=oo`03001oogoo00=o
o`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`07F]IoogooOol0
07oo000017oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol017oo00=J
eWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomo
o`03Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo01Eoo`008Woo00MJeWooOomoo`00Ool00004
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool00e[FOomoo`03Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07
F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03
Ool00e[FOomoo`02Ool00`00Oomoo`04Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo
000017oo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo
001oo`0000Eoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo
001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00=JeWoo
Ool00goo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007oo
Ool00goo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02
Ool00`00Oomoo`04Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`03
F]Ioogoo009oo`03001oogoo00=oo`07F]IoogooOol007oo000017oo00=JeWooOol05Goo000ROol0
1e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001o
ogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00=oo`03001oogoo00=oo`07F]Io
ogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`03F]Io
ogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00Aoo`07F]IoogooOol007oo0000
17oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001o
ogoo00=oo`07F]IoogooOol007oo00001Goo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001o
o`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool0
0`00Oomoo`03Ool00e[FOomoo`03Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool0
0e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001o
ogoo00=oo`03F]Ioogoo009oo`03001oogoo00Aoo`03F]Ioogoo009oo`03001oogoo00=oo`07F]Io
ogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool0
0e[FOomoo`0EOol0029oo`03F]Ioogoo009oo`03001oogoo00=oo`04F]IoogooOol20005Ool015[F
Oomoogoo0P001Goo00AJeWooOomoo`8000Eoo`03F]Ioogoo009oo`8000Eoo`03F]Ioogoo009oo`03
001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`04
F]IoogooOol20005Ool015[FOomoogoo0P001Woo00=JeWooOol00Woo00<007ooOol00goo00=JeWoo
Ool00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00AJeWooOomoo`8000Eoo`03
F]Ioogoo009oo`03001oogoo00Aoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03
001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`04
F]IoogooOol20005Ool00e[FOomoo`02Ool20005Ool015[FOomoogoo0P001Goo00AJeWooOomoo`80
00Eoo`03F]Ioogoo009oo`03001oogoo00=oo`04F]IoogooOol20005Ool015[FOomoogoo0P001Woo
00AJeWooOomoo`8000Eoo`03F]Ioogoo009oo`03001oogoo00=oo`04F]IoogooOol20005Ool00e[F
Oomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`0EOol0029oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Uoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo01Eoo`008Woooe[FDE[F
5goo003oOon:Ool0025ooomJeU=JeQIoo`008Goo00=JeWooOol027oo00=JeWooOol02Goo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol057oo000QOol00e[FOomoo`03Ool0
0e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`03Ool00e[FOomoo`08Ool00e[FOomoo`02Ool0
0e[FOomoo`03Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`03Ool00e[FOomoo`03Ool0
0e[FOomoo`08Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`08Ool00e[FOomoo`03Ool0
0e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`03Ool00e[FOomoo`03Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`03Ool0
0e[FOomoo`02Ool00e[FOomoo`08Ool00e[FOomoo`03Ool00e[FOomoo`03Ool00e[FOomoo`08Ool0
0e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool0
0e[FOomoo`03Ool00e[FOomoo`08Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`08Ool0
0e[FOomoo`03Ool00e[FOomoo`03Ool00e[FOomoo`08Ool00e[FOomoo`02Ool00e[FOomoo`03Ool0
0e[FOomoo`0DOol0025oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo
00=oo`04F]IoogooF]H4Ool015[FOomooe[F17oo00=JeWooOol00goo00AJeWooOomJePAoo`04F]Io
ogooF]H2Ool00e[FOomoo`02Ool015[FOomooe[F1Goo00=JeWooOol00goo00AJeWooOomJePAoo`04
F]IoogooF]H5Ool00e[FOomoo`02Ool015[FOomooe[F17oo00AJeWooOomJePEoo`03F]Ioogoo00=o
o`03F]Ioogoo009oo`03F]Ioogoo00=oo`04F]IoogooF]H4Ool015[FOomooe[F0Woo00=JeWooOol0
0Woo00AJeWooOomJePEoo`03F]Ioogoo00=oo`04F]IoogooF]H4Ool015[FOomooe[F0Woo00=JeWoo
Ool00Woo00AJeWooOomJePEoo`03F]Ioogoo009oo`04F]IoogooF]H4Ool015[FOomooe[F1Goo00=J
eWooOol00goo00AJeWooOomJePAoo`04F]IoogooF]H5Ool00e[FOomoo`02Ool00e[FOomoo`03Ool0
0e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`03Ool015[FOomooe[F17oo00AJeWooOomJePAo
o`03F]Ioogoo00=oo`04F]IoogooF]H4Ool015[FOomooe[F1Goo00=JeWooOol00goo00AJeWooOomJ
ePAoo`04F]IoogooF]H4Ool00e[FOomoo`03Ool00e[FOomoo`0DOol0025oo`03F]Ioogoo00=oo`03
F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`05F]IoogooOomJeP02Ool01E[FOomoogoo
F]H017oo00=JeWooOol00goo00EJeWooOomooe[F009oo`05F]IoogooOomJeP03Ool015[FOomooe[F
0goo00=JeWooOol00goo00=JeWooOol00goo00EJeWooOomooe[F009oo`05F]IoogooOomJeP05Ool0
0e[FOomoo`02Ool01E[FOomoogooF]H00Woo00EJeWooOomooe[F00Eoo`03F]Ioogoo00=oo`03F]Io
ogoo009oo`03F]Ioogoo00=oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H00goo00AJeWooOomJ
eP=oo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H00goo
00AJeWooOomJeP=oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`05F]IoogooOomJeP02Ool01E[FOomo
ogooF]H01Goo00=JeWooOol00goo00EJeWooOomooe[F009oo`05F]IoogooOomJeP05Ool00e[FOomo
o`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`03Ool01E[FOomo
ogooF]H00Woo00EJeWooOomooe[F00Aoo`03F]Ioogoo00=oo`05F]IoogooOomJeP02Ool01E[FOomo
ogooF]H01Goo00=JeWooOol00goo00EJeWooOomooe[F009oo`05F]IoogooOomJeP04Ool00e[FOomo
o`03Ool00e[FOomoo`0DOol0025oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03
F]Ioogoo00=oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo
009oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo
009oo`9JePAoo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo
00=oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`9JePAoo`03
F]Ioogoo009oo`9JePAoo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`9JePAoo`03
F]Ioogoo009oo`9JePAoo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo009oo`9JePAoo`03
F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo00=oo`03F]Ioogoo
009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo
009oo`9JePAoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo
00=oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03
F]Ioogoo01Aoo`008Goo00=JeWooF]H01e[F00Aooe[FOomoo`MJeP9oo`03F]Ioogoo009oo`9JePAo
o`03F]Iooe[F00MJeP03OomJeWoo00=oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo00MJ
eP9oo`03F]Ioogoo009oo`9JePAoo`03F]Iooe[F00MJeP03OomJeWoo00=oo`9JePAoo`03F]Ioogoo
00MJeP9oo`03F]Iooe[F00MJeP03OomJeWoo00=oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Io
ogoo00MJeP9oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Iooe[F00MJeP03
OomJeWoo00=oo`9JePAoo`03F]Ioogoo00MJeP9oo`03F]Ioogoo009oo`9JePAoo`03F]Iooe[F00MJ
eP03OomJeWoo00QJeP04OomJeWooOol7F]H2Ool00e[FOomoo`02Ool2F]H4Ool00e[FOomJeP07F]H0
0gooF]Ioo`03Ool2F]H4Ool00e[FOomoo`07F]H2Ool00e[FOomoo`02Ool2F]H4Ool00e[FOomJeP07
F]H00gooF]Ioo`0EOol0025oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Io
ogoo00=oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H017oo00=JeWooOol00goo00EJeWooOomo
oe[F009oo`05F]IoogooOomJeP03Ool015[FOomooe[F0goo00=JeWooOol00goo00=JeWooOol00goo
00EJeWooOomooe[F009oo`05F]IoogooOomJeP05Ool00e[FOomoo`02Ool01E[FOomoogooF]H00Woo
00EJeWooOomooe[F00Eoo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`05F]Ioogoo
OomJeP02Ool01E[FOomoogooF]H00goo00AJeWooOomJeP=oo`03F]Ioogoo00=oo`03F]Ioogoo00=o
o`05F]IoogooOomJeP02Ool01E[FOomoogooF]H00goo00AJeWooOomJeP=oo`03F]Ioogoo00=oo`03
F]Ioogoo009oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H01Goo00=JeWooOol00goo00EJeWoo
Oomooe[F009oo`05F]IoogooOomJeP05Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02
Ool00e[FOomoo`03Ool00e[FOomoo`03Ool01E[FOomoogooF]H00Woo00EJeWooOomooe[F00Aoo`03
F]Ioogoo00=oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H01Goo00=JeWooOol00goo00EJeWoo
Oomooe[F009oo`05F]IoogooOomJeP04Ool00e[FOomoo`03Ool00e[FOomoo`0DOol0025oo`03F]Io
ogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`04F]IoogooF]H4Ool015[F
Oomooe[F17oo00=JeWooOol00goo00AJeWooOomJePAoo`04F]IoogooF]H2Ool00e[FOomoo`02Ool0
15[FOomooe[F1Goo00=JeWooOol00goo00AJeWooOomJePAoo`04F]IoogooF]H5Ool00e[FOomoo`02
Ool015[FOomooe[F17oo00AJeWooOomJePEoo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo
00=oo`04F]IoogooF]H4Ool015[FOomooe[F0Woo00=JeWooOol00Woo00AJeWooOomJePEoo`03F]Io
ogoo00=oo`04F]IoogooF]H4Ool015[FOomooe[F0Woo00=JeWooOol00Woo00AJeWooOomJePEoo`03
F]Ioogoo009oo`04F]IoogooF]H4Ool015[FOomooe[F1Goo00=JeWooOol00goo00AJeWooOomJePAo
o`04F]IoogooF]H5Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03
Ool00e[FOomoo`03Ool015[FOomooe[F17oo00AJeWooOomJePAoo`03F]Ioogoo00=oo`04F]Ioogoo
F]H4Ool015[FOomooe[F1Goo00=JeWooOol00goo00AJeWooOomJePAoo`04F]IoogooF]H4Ool00e[F
Oomoo`03Ool00e[FOomoo`0DOol0025oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=o
o`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00=o
o`03F]Ioogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo009o
o`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo00=o
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00Qo
o`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo009o
o`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo00Qo
o`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo00=o
o`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo01Aoo`008Goo00=JeWoo
Ool027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool057oo000QOoooF]ICF]HFOol00?moohYoo`008Woooe[FDE[F5goo000ROol00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`09Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`0EOol0
029oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo
00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo
00Uoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo
00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00Uoo`03F]Ioogoo
00Qoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo
00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03001oogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo
00Qoo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo
00Uoo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo
00Qoo`03F]Ioogoo01Eoo`008Woo00=JeWooOol01Goo00@007ooOomJePYoo`03F]Ioogoo00Eoo`04
001oogooF]H7Ool01000Oomooe[F1goo00D007ooOomooe[F009oo`03001oogoo00Eoo`04F]Ioogoo
0007Ool00e[FOomoo`08Ool015[FOomoo`001goo00=JeWooOol01Goo00D007ooOomooe[F00Aoo`03
001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00=oo`03001oogoo009oo`03
F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00Eoo`05001oogooOomJeP02Ool00`00Oomoo`05
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool015[FOomoo`00
1goo00=JeWooOol027oo00=JeWooOol00goo00<007ooOol00goo00AJeWooOol000Moo`04F]Ioogoo
0007Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool01E[FOomo
ogoo00001goo00=JeWooOol01Goo00@007ooOomJePAoo`03001oogoo00=oo`03F]Ioogoo00Eoo`04
001oogooF]H7Ool01000Oomooe[F5goo000ROol00e[FOomoo`04Ool01@00OomoogooF]H02Woo00=J
eWooOol017oo00D007ooOomooe[F00Ioo`05001oogooOomJeP06Ool00`00Oomoo`02Ool01E[FOomo
ogoo00001Woo00EJeWooOomoo`0000Ioo`03F]Ioogoo00Qoo`05F]IoogooOol00006Ool00e[FOomo
o`04Ool00`00Oomoo`02Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomo
o`03Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomo
o`04Ool00`00Oomoo`02Ool01E[FOomoogoo00001Woo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol00Woo00<007ooOol00goo00EJeWooOomoo`0000Ioo`03F]Ioogoo00Qoo`03F]Ioogoo00=o
o`03001oogoo00=oo`05F]IoogooOol00006Ool01E[FOomoogoo00001Woo00=JeWooOol00goo00<0
07ooOol00Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00Woo00<007ooOol017oo00=J
eWooOol017oo00D007ooOomooe[F00Aoo`03001oogoo00=oo`03F]Ioogoo00Aoo`05001oogooOomJ
eP06Ool01@00OomoogooF]H05goo000ROol00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`08
Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03
Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03
Ool00e[FOomoo`08Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`03Ool00`00Oomoo`03
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`03
Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`03
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`08Ool00e[FOomoo`03
Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03
Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03
Ool00`00Oomoo`03Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`02Ool00`00Oomoo`03
Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`0E
Ool0029oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioo`0000L00003OomJeWoo00=oo`03001o
ogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00=oo`03001oogoo00=oo`03F]Io
ogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioo`0000L00003OomJ
eWoo00Aoo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo00=oo`03F]Ioogoo009oo`03001o
ogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Io
ogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo00=oo`03F]Ioogoo009oo`03001o
ogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Io
ogoo00=oo`03001oogoo009oo`03F]Ioo`0000L00003OomJeWoo00Aoo`03001oogoo00=oo`03F]Io
ogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001o
ogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo00=oo`03F]Io
ogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001o
ogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo01Eoo`008Woo00EJeWooOomoo`00
00Ioo`03F]Ioogoo00Qoo`05F]IoogooOol00006Ool01E[FOomoogoo00001Woo00=JeWooOol00Woo
00<007ooOol017oo00=JeWooOol017oo00D007ooOomooe[F00Ioo`05001oogooOomJeP0:Ool00e[F
Oomoo`04Ool01@00OomoogooF]H017oo00<007ooOol017oo00=JeWooOol00Woo00<007ooOol00goo
00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00goo
00<007ooOol00Woo00=JeWooOol00Woo00<007ooOol017oo00=JeWooOol00goo00<007ooOol00Woo
00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol017oo
00D007ooOomooe[F00Yoo`03F]Ioogoo00=oo`03001oogoo00=oo`03F]Ioogoo00Aoo`05001oogoo
OomJeP06Ool01@00OomoogooF]H01Goo00<007ooOol00Woo00=JeWooOol00goo00<007ooOol00Woo
00=JeWooOol017oo00<007ooOol00Woo00EJeWooOomoo`0000Ioo`03F]Ioogoo009oo`03001oogoo
00=oo`05F]IoogooOol00006Ool01E[FOomoogoo00001Woo00=JeWooOol05Goo000ROol015[FOomo
o`001goo00=JeWooOol027oo00AJeWooOol000Moo`04F]Ioogoo0007Ool01E[FOomoogoo00001goo
00=JeWooOol01Goo00@007ooOomJePMoo`04001oogooF]H:Ool00e[FOomoo`05Ool01000Oomooe[F
0goo00<007ooOol01Goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol0
0goo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00goo00<007ooOol00Woo00EJeWooOomo
o`0000Moo`03F]Ioogoo00Aoo`05001oogooOomJeP04Ool00`00Oomoo`03Ool00e[FOomoo`02Ool0
0`00Oomoo`03Ool00e[FOomoo`05Ool01000Oomooe[F2Woo00=JeWooOol00goo00<007ooOol00goo
00=JeWooOol01Goo00@007ooOomJePMoo`04001oogooF]H5Ool00`00Oomoo`02Ool00e[FOomoo`03
Ool00`00Oomoo`02Ool00e[FOomoo`05Ool01@00OomoogooF]H00Woo00<007ooOol01Goo00=JeWoo
Ool00Woo00<007ooOol00goo00AJeWooOol000Moo`04F]Ioogoo0007Ool00e[FOomoo`0EOol0029o
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Uo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Uo
o`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00=o
o`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00Uoo`03F]Ioogoo00Qo
o`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03001oogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00Uo
o`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo01Eoo`008Woo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol02Goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol05Goo000ROoooF]IAF]HGOol00?moohYoo`00\
\>"],
  ImageRangeCache->{{{0, 392.875}, {39.625, 0}} -> {-0.0373468, -0.00258119, 
  0.00273546, 0.00273546}, {{13.625, 379.188}, {38.625, 
  26.875}} -> {-2.05242, -0.114373, 0.0893679, 0.0893679}, {{13.625, 
  379.188}, {25.6875, 13.875}} -> {-1.98835, -1.26689, 0.0890417, 
  0.0890417}, {{13.625, 379.188}, {12.6875, 0.9375}} -> {-2.05242, -2.43235, 
  0.0893679, 0.0893679}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .13871 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.00330261 0.952381 [
[ 0 0 0 0 ]
[ 1 .13871 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .13871 L
0 .13871 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.00330261 0.97619 0.0340246 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
gsave
.06989 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .10061 .01613 0 0 Mshowa
[(1)] .13134 .01613 0 0 Mshowa
gsave
.16206 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.19278 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .2235 .01613 0 0 Mshowa
gsave
.25422 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.28495 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.31567 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .34639 .01613 0 0 Mshowa
gsave
.37711 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .40783 .01613 0 0 Mshowa
gsave
.43856 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.46928 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.5 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .53072 .01613 0 0 Mshowa
gsave
.56144 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .59217 .01613 0 0 Mshowa
gsave
.62289 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .65361 .01613 0 0 Mshowa
gsave
.68433 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.71505 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.74578 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.7765 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.80722 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.83794 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .86866 .01613 0 0 Mshowa
[(0)] .89939 .01613 0 0 Mshowa
gsave
.93011 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.96083 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0370968 0.97619 0.0678187 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
.5 Mabswid
.0914 .01613 m
.10983 .01613 L
s
.10061 .02535 m
.10061 .00691 L
s
.12482 .00961 m
.13785 .02265 L
s
.12482 .02265 m
.13785 .00961 L
s
.21699 .00961 m
.23002 .02265 L
s
.21699 .02265 m
.23002 .00961 L
s
.33987 .00961 m
.35291 .02265 L
s
.33987 .02265 m
.35291 .00961 L
s
.39862 .01613 m
.41705 .01613 L
s
.40783 .02535 m
.40783 .00691 L
s
.5242 .00961 m
.53724 .02265 L
s
.5242 .02265 m
.53724 .00961 L
s
.58295 .01613 m
.60138 .01613 L
s
.59217 .02535 m
.59217 .00691 L
s
.64439 .01613 m
.66283 .01613 L
s
.65361 .02535 m
.65361 .00691 L
s
.86215 .00961 m
.87518 .02265 L
s
.86215 .02265 m
.87518 .00961 L
s
.89017 .01613 m
.9086 .01613 L
s
.89939 .02535 m
.89939 .00691 L
s
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0708909 0.97619 0.101613 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
.5 Mabswid
.0914 .01613 m
.10983 .01613 L
s
.10061 .02535 m
.10061 .00691 L
s
.12482 .00961 m
.13785 .02265 L
s
.12482 .02265 m
.13785 .00961 L
s
.21699 .00961 m
.23002 .02265 L
s
.21699 .02265 m
.23002 .00961 L
s
.33987 .00961 m
.35291 .02265 L
s
.33987 .02265 m
.35291 .00961 L
s
.39862 .01613 m
.41705 .01613 L
s
.40783 .02535 m
.40783 .00691 L
s
.5242 .00961 m
.53724 .02265 L
s
.5242 .02265 m
.53724 .00961 L
s
.58295 .01613 m
.60138 .01613 L
s
.59217 .02535 m
.59217 .00691 L
s
.64439 .01613 m
.66283 .01613 L
s
.65361 .02535 m
.65361 .00691 L
s
.86215 .00961 m
.87518 .02265 L
s
.86215 .02265 m
.87518 .00961 L
s
.89017 .01613 m
.9086 .01613 L
s
.89939 .02535 m
.89939 .00691 L
s
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.104685 0.97619 0.135407 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
gsave
.06989 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .10061 .01613 0 0 Mshowa
[(1)] .13134 .01613 0 0 Mshowa
gsave
.16206 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.19278 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .2235 .01613 0 0 Mshowa
gsave
.25422 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.28495 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.31567 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .34639 .01613 0 0 Mshowa
gsave
.37711 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .40783 .01613 0 0 Mshowa
gsave
.43856 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.46928 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.5 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .53072 .01613 0 0 Mshowa
gsave
.56144 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .59217 .01613 0 0 Mshowa
gsave
.62289 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .65361 .01613 0 0 Mshowa
gsave
.68433 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.71505 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.74578 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.7765 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.80722 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.83794 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .86866 .01613 0 0 Mshowa
[(0)] .89939 .01613 0 0 Mshowa
gsave
.93011 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.96083 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{339.75, 47.125},
  ImageMargins->{{34, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40005C0000;a000`40O003h00OogooE7oo003oOomD
Ool001aooomJeREJeQ=oo`0077oo00=JeWooOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=J
eWooOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01goo00=J
eWooOol01Woo00=JeWooOol01goo00=JeWooOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=J
eWooOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01goo00=J
eWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=J
eWooOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01goo00=J
eWooOol01goo00=JeWooOol01Woo00=JeWooOol04Goo000LOol00e[FOomoo`07Ool00e[FOomoo`07
Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07
Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`07
Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07
Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07
Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07
Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001aoo`03F]Io
ogoo00Moo`03F]Ioo`0000@000=oo`03F]Ioo`0000@0009oo`03F]Ioogoo00Moo`03F]Ioogoo00Mo
o`05F]IoogooOol00004Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomo
o`050002Ool00e[FOomoo`07Ool01E[FOomoogoo00001Goo00=JeWooOol01Woo00=JeWooOol01goo
00=JeWooOol01goo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Moo`05F]IoogooOol00005Ool00e[F
Oomoo`07Ool00e[FOol000040002Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool0
0e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`050002Ool01E[FOomoogoo
00001Goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol04Goo000LOol00e[FOomoo`07Ool0
1E[FOomoogoo00001Goo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`06F]Io
ogoo001oo`000goo00=JeWooOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol00Woo
00<007ooOol00Woo00=JeWooOol01goo00IJeWooOol007oo0004Ool00e[FOomoo`06Ool00e[FOomo
o`07Ool00e[FOomoo`07Ool01U[FOomoo`00Ool000=oo`03F]Ioogoo00Moo`06F]Ioogoo001oo`00
17oo00=JeWooOol01goo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Io
ogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001o
ogoo009oo`06F]Ioogoo001oo`0017oo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol04Goo
000LOol00e[FOomoo`07Ool01E[FOomoogoo00001Goo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Mo
o`03F]Ioogoo00Moo`06F]Ioogoo001oo`000goo00=JeWooOol01goo00=JeWooOol01goo00=JeWoo
Ool01Woo00=JeWooOol00Woo00<007ooOol00Woo00=JeWooOol01goo00IJeWooOol007oo0004Ool0
0e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool01U[FOomoo`00Ool000=oo`03F]Ioogoo
00Moo`06F]Ioogoo001oo`0017oo00=JeWooOol01goo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Mo
o`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Io
o`03F]Ioogoo009oo`03001oogoo009oo`06F]Ioogoo001oo`0017oo00=JeWooOol01goo00=JeWoo
Ool01Woo00=JeWooOol04Goo000LOol00e[FOomoo`07Ool01E[FOomoogoo00001Goo00EJeWooOomo
o`0000Aoo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`06F]Ioogoo001oo`000goo00=JeWooOol01goo
00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol00Woo00<007ooOol00Woo00=JeWooOol01goo
00IJeWooOol007oo0004Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool01U[FOomo
o`00Ool000=oo`03F]Ioogoo00Moo`06F]Ioogoo001oo`0017oo00=JeWooOol01goo00EJeWooOomo
o`0000Aoo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Io
ogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001oogoo009oo`06F]Ioogoo001oo`0017oo
00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol04Goo000LOol00e[FOomoo`07Ool01E[FOomo
ogoo00001Goo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`06F]Ioogoo001o
o`000goo00=JeWooOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol00Woo00<007oo
Ool00Woo00=JeWooOol01goo00IJeWooOol007oo0004Ool00e[FOomoo`06Ool00e[FOomoo`07Ool0
0e[FOomoo`07Ool01U[FOomoo`00Ool000=oo`03F]Ioogoo00Moo`06F]Ioogoo001oo`0017oo00=J
eWooOol01goo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Io
o`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001oogoo009o
o`06F]Ioogoo001oo`0017oo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol04Goo000LOol0
0e[FOomoo`07Ool00e[FOomoo`020005Ool00e[FOomoo`020004Ool00e[FOomoo`07Ool00e[FOomo
o`07Ool01E[FOomoogoo000017oo00=JeWooOol01goo00=JeWooOol01goo00=JeWooOol01Woo00AJ
eWooOomoo`8000Aoo`03F]Ioogoo00Moo`05F]IoogooOol00005Ool00e[FOomoo`06Ool00e[FOomo
o`07Ool00e[FOomoo`07Ool01E[FOomoogoo000017oo00=JeWooOol01goo00EJeWooOomoo`0000Eo
o`03F]Ioogoo00Moo`03F]Ioogoo008000Aoo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo
00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`04F]IoogooOol20004Ool01E[F
Oomoogoo00001Goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol04Goo000LOoooF]HUF]HC
Ool00?mooeAoo`0077oooe[F9E[F4goo000LOol00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomo
o`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomo
o`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomo
o`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001aoo`03F]Ioogoo00Moo`03
F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03
F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03
F]Ioogoo00Moo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03
F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo009oo`03F]Ioogoo009oo`03
F]Ioogoo00Moo`05F]IoogooOomJeP04Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06
Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`02
Ool00e[FOomoo`02Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001aoo`03F]Io
ogoo00Moo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Iooe[F00=oo`04F]IoogooF]H9Ool00e[F
Oomoo`07Ool00e[FOomJeP04Ool00e[FOomJeP09Ool00e[FOomoo`07Ool00e[FOomoo`06Ool015[F
Oomooe[F0goo00AJeWooOomJePUoo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Ioo`03
F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Iooe[F00Aoo`03F]Iooe[F00Uoo`03F]Ioogoo009oo`03
F]Ioogoo009oo`03F]Ioogoo00Moo`05F]IoogooOomJeP04Ool00e[FOomoo`07Ool00e[FOomoo`07
Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool015[FOomooe[F
0goo00AJeWooOomJePAoo`03F]Ioogoo009oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo
015oo`0077oo00=JeWooOol01goo00=JeWooOol00Woo00=JeWooOol00Woo00IJeWooOomJeWooF]H3
Ool00e[FOomoo`07Ool00e[FOomoo`07Ool015[FOomooe[F0Woo00AJeWooOomJePUoo`03F]Ioogoo
00Moo`03F]Ioogoo00Ioo`07F]IoogooOomJeWooF]H00goo00=JeWooOol01goo00=JeWooOol00Woo
00=JeWooOol00Woo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01goo00AJeWooOomJeP9o
o`04F]IoogooF]H9Ool00e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`07Ool01E[FOomoogoo
F]H017oo00=JeWooOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWoo
Ool01goo00=JeWooOol01Woo00MJeWooOomooe[FOomJeP03Ool00e[FOomoo`02Ool00e[FOomoo`02
Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001aoo`03F]Ioogoo00Moo`03F]Io
oe[F00IJeP03OomJeWoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`04F]Io
ogooOol2F]H3Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`02Ool0
0e[FOomoo`02Ool00e[FOomoo`07Ool00e[FOomJeP06F]H00gooF]Ioo`07Ool00e[FOomoo`07Ool0
0e[FOomoo`07Ool015[FOomoogoo0U[F0goo00=JeWooOol01goo00=JeWooF]H01U[F00=ooe[FOol0
27oo25[F00=ooe[FOol027oo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWoo
Ool01goo00=JeWooOol01Woo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooF]H01U[F00=ooe[F
Ool027oo00=JeWooOol01Woo00=JeWooOol04Goo000LOol00e[FOomoo`07Ool00e[FOomoo`02Ool0
0e[FOomoo`02Ool01U[FOomooe[FOomJeP=oo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`04F]Ioogoo
F]H2Ool015[FOomooe[F2Goo00=JeWooOol01goo00=JeWooOol01Woo00MJeWooOomooe[FOomJeP03
Ool00e[FOomoo`07Ool00e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`06Ool00e[FOomoo`07
Ool00e[FOomoo`07Ool015[FOomooe[F0Woo00AJeWooOomJePUoo`03F]Ioogoo009oo`03F]Ioogoo
009oo`03F]Ioogoo00Moo`05F]IoogooOomJeP04Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[F
Oomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool01e[FOomoogooF]Iooe[F
00=oo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo
015oo`0077oo00=JeWooOol01goo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooF]H00goo00AJ
eWooOomJePUoo`03F]Ioogoo00Moo`03F]Iooe[F00Aoo`03F]Iooe[F00Uoo`03F]Ioogoo00Moo`03
F]Ioogoo00Ioo`04F]IoogooF]H3Ool015[FOomooe[F2Goo00=JeWooOol00Woo00=JeWooOol00Woo
00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01goo00=JeWooF]H017oo00=JeWooF]H02Goo
00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooOol01goo00EJeWooOomooe[F00Aoo`03F]Ioogoo
00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo
00Ioo`04F]IoogooF]H3Ool015[FOomooe[F17oo00=JeWooOol00Woo00=JeWooOol01goo00=JeWoo
Ool01Woo00=JeWooOol04Goo000LOol00e[FOomoo`07Ool00e[FOomoo`02Ool00e[FOomoo`02Ool0
0e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool0
0e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`02Ool0
0e[FOomoo`02Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool0
0e[FOomoo`07Ool00e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`07Ool01E[FOomoogooF]H0
17oo00=JeWooOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol0
1goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooOol0
1goo00=JeWooOol01Woo00=JeWooOol04Goo000LOol00e[FOomoo`07Ool00e[FOomoo`07Ool00e[F
Oomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[F
Oomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[F
Oomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[F
Oomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[F
Oomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[F
Oomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001aooomJeREJeQ=oo`00
77oooe[F9E[F4goo000LOol00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomo
o`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomo
o`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomo
o`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomo
o`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomo
o`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001aoo`03F]Ioogoo00Moo`03F]Ioogoo009oo`03
F]Ioogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03
F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03
F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03
F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Moo`05
F]IoogooOomJeP04Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07
Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`02Ool00e[FOomoo`02
Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001aoo`03F]Ioogoo00Moo`03F]Io
ogoo009oo`03F]Ioogoo009oo`03F]Iooe[F00=oo`04F]IoogooF]H9Ool00e[FOomoo`07Ool00e[F
OomJeP04Ool00e[FOomJeP09Ool00e[FOomoo`07Ool00e[FOomoo`06Ool015[FOomooe[F0goo00AJ
eWooOomJePUoo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03
F]Ioogoo00Moo`03F]Iooe[F00Aoo`03F]Iooe[F00Uoo`03F]Ioogoo009oo`03F]Ioogoo009oo`03
F]Ioogoo00Moo`05F]IoogooOomJeP04Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06
Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool015[FOomooe[F0goo00AJeWooOomJ
ePAoo`03F]Ioogoo009oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo015oo`0077oo00=J
eWooOol01goo00=JeWooOol00Woo00=JeWooOol00Woo00IJeWooOomJeWooF]H3Ool00e[FOomoo`07
Ool00e[FOomoo`07Ool015[FOomooe[F0Woo00AJeWooOomJePUoo`03F]Ioogoo00Moo`03F]Ioogoo
00Ioo`07F]IoogooOomJeWooF]H00goo00=JeWooOol01goo00=JeWooOol00Woo00=JeWooOol00Woo
00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01goo00AJeWooOomJeP9oo`04F]IoogooF]H9
Ool00e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`07Ool01E[FOomoogooF]H017oo00=JeWoo
Ool01goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01goo00=JeWoo
Ool01Woo00MJeWooOomooe[FOomJeP03Ool00e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`07
Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001aoo`03F]Ioogoo00Moo`03F]Iooe[F00IJeP03OomJ
eWoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`04F]IoogooOol2F]H3Ool0
0e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`02Ool00e[FOomoo`02Ool0
0e[FOomoo`07Ool00e[FOomJeP06F]H00gooF]Ioo`07Ool00e[FOomoo`07Ool00e[FOomoo`07Ool0
15[FOomoogoo0U[F0goo00=JeWooOol01goo00=JeWooF]H01U[F00=ooe[FOol027oo25[F00=ooe[F
Ool027oo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01goo00=JeWoo
Ool01Woo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooF]H01U[F00=ooe[FOol027oo00=JeWoo
Ool01Woo00=JeWooOol04Goo000LOol00e[FOomoo`07Ool00e[FOomoo`02Ool00e[FOomoo`02Ool0
1U[FOomooe[FOomJeP=oo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`04F]IoogooF]H2Ool015[FOomo
oe[F2Goo00=JeWooOol01goo00=JeWooOol01Woo00MJeWooOomooe[FOomJeP03Ool00e[FOomoo`07
Ool00e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07
Ool015[FOomooe[F0Woo00AJeWooOomJePUoo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo
00Moo`05F]IoogooOomJeP04Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[F
Oomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool01e[FOomoogooF]Iooe[F00=oo`03F]Ioogoo
009oo`03F]Ioogoo009oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo015oo`0077oo00=J
eWooOol01goo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooF]H00goo00AJeWooOomJePUoo`03
F]Ioogoo00Moo`03F]Iooe[F00Aoo`03F]Iooe[F00Uoo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`04
F]IoogooF]H3Ool015[FOomooe[F2Goo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooOol01Woo
00=JeWooOol01goo00=JeWooOol01goo00=JeWooF]H017oo00=JeWooF]H02Goo00=JeWooOol00Woo
00=JeWooOol00Woo00=JeWooOol01goo00EJeWooOomooe[F00Aoo`03F]Ioogoo00Moo`03F]Ioogoo
00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`04F]Ioogoo
F]H3Ool015[FOomooe[F17oo00=JeWooOol00Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWoo
Ool04Goo000LOol00e[FOomoo`07Ool00e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`06Ool0
0e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool0
0e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`02Ool00e[FOomoo`02Ool0
0e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool0
0e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`07Ool01E[FOomoogooF]H017oo00=JeWooOol0
1goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01goo00=JeWooOol0
1Woo00=JeWooOol01goo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooOol01goo00=JeWooOol0
1Woo00=JeWooOol04Goo000LOol00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[F
Oomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[F
Oomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[F
Oomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[F
Oomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[F
Oomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[F
Oomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001aooomJeREJeQ=oo`0077oooe[F9U[F4Woo
000LOol00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomo
o`07Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomo
o`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomo
o`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomo
o`07Ool00e[FOomoo`0@Ool001aoo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03
F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03
F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03
F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03
F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03
F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03
F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo011oo`0077oo00=JeWooOol01goo00=JeWoo0000
10000Woo00=JeWooOol01@000Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol00Woo00<0
07ooOol00Woo00=JeWooOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01@000Woo
00=JeWooOol01goo00EJeWooOomoo`0000Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo
00Moo`05F]IoogooOol00004Ool00e[FOomoo`07Ool00e[FOomoo`02Ool00`00Oomoo`02Ool00e[F
Oomoo`07Ool00e[FOol000040002Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`07Ool0
0e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOol000040002Ool00e[FOomoo`02
Ool00`00Oomoo`02Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`0@Ool001aoo`03F]Io
ogoo00Moo`05F]IoogooOol00004Ool00e[FOomoo`02Ool00`00Oomoo`02Ool00e[FOomoo`07Ool0
0e[FOomoo`06Ool01e[FOomoogoo001oo`0000=oo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Io
ogoo00Ioo`03F]Ioogoo009oo`03001oogoo009oo`03F]Ioogoo00Moo`06F]Ioogoo001oo`0017oo
00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01goo00IJeWooOol007oo0003Ool00e[FOomo
o`07Ool01e[FOomoogoo001oo`0000=oo`03F]Ioogoo00Moo`05F]IoogooOol00004Ool00e[FOomo
o`07Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomo
o`07Ool01E[FOomoogoo000017oo00MJeWooOomoo`00Ool00003Ool00e[FOomoo`07Ool00e[FOomo
o`07Ool00e[FOomoo`0@Ool001aoo`03F]Ioogoo00Moo`05F]IoogooOol00004Ool00e[FOomoo`02
Ool00`00Oomoo`02Ool00e[FOomoo`07Ool00e[FOomoo`06Ool01e[FOomoogoo001oo`0000=oo`03
F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001oogoo009oo`03
F]Ioogoo00Moo`06F]Ioogoo001oo`0017oo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol0
1goo00IJeWooOol007oo0003Ool00e[FOomoo`07Ool01e[FOomoogoo001oo`0000=oo`03F]Ioogoo
00Moo`05F]IoogooOol00004Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[F
Oomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool01E[FOomoogoo000017oo00MJeWooOomoo`00
Ool00003Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`0@Ool001aoo`03F]Ioogoo00Mo
o`05F]IoogooOol00004Ool00e[FOomoo`02Ool00`00Oomoo`02Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool01e[FOomoogoo001oo`0000=oo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Io
o`03F]Ioogoo009oo`03001oogoo009oo`03F]Ioogoo00Moo`06F]Ioogoo001oo`0017oo00=JeWoo
Ool01Woo00=JeWooOol01goo00=JeWooOol01goo00IJeWooOol007oo0003Ool00e[FOomoo`07Ool0
1e[FOomoogoo001oo`0000=oo`03F]Ioogoo00Moo`05F]IoogooOol00004Ool00e[FOomoo`07Ool0
0e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool0
1E[FOomoogoo000017oo00MJeWooOomoo`00Ool00003Ool00e[FOomoo`07Ool00e[FOomoo`07Ool0
0e[FOomoo`0@Ool001aoo`03F]Ioogoo00Moo`05F]IoogooOol00004Ool00e[FOomoo`02Ool00`00
Oomoo`02Ool00e[FOomoo`07Ool00e[FOomoo`06Ool01e[FOomoogoo001oo`0000=oo`03F]Ioogoo
00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001oogoo009oo`03F]Ioogoo
00Moo`06F]Ioogoo001oo`0017oo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01goo00IJ
eWooOol007oo0003Ool00e[FOomoo`07Ool01e[FOomoogoo001oo`0000=oo`03F]Ioogoo00Moo`05
F]IoogooOol00004Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06
Ool00e[FOomoo`07Ool00e[FOomoo`07Ool01E[FOomoogoo000017oo00MJeWooOomoo`00Ool00003
Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`0@Ool001aoo`03F]Ioogoo00Moo`03F]Io
ogoo008000Aoo`04F]IoogooOol20004Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`02
Ool00`00Oomoo`02Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`06Ool015[FOomoogoo
0P0017oo00=JeWooOol01goo00EJeWooOomoo`0000Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03
F]Ioogoo00Moo`05F]IoogooOol00004Ool00e[FOomoo`07Ool00e[FOomoo`02Ool00`00Oomoo`02
Ool00e[FOomoo`07Ool00e[FOomoo`020004Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomo
o`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`020004Ool00e[F
Oomoo`02Ool00`00Oomoo`02Ool00e[FOomoo`07Ool00e[FOomoo`07Ool00e[FOomoo`0@Ool001ao
o`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Io
o`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Mo
o`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Mo
o`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Io
o`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Mo
o`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Moo`03F]Ioogoo00Mo
o`03F]Ioogoo011oo`0077oooe[F9U[F4Woo003oOomDOol00001\
\>"],
  ImageRangeCache->{{{0, 338.75}, {46.125, 0}} -> {-0.0348282, -0.00346847, 
  0.00315766, 0.00315766}, {{11, 327.688}, {45, 
  34.8125}} -> {-1.95505, -0.140964, 0.103075, 0.103075}, {{11, 327.688}, {
  33.75, 23.5625}} -> {-1.95505, -1.30055, 0.103075, 0.103075}, {{11, 
  327.688}, {22.5, 12.3125}} -> {-1.95505, -2.46014, 0.103075, 0.103075}, {{
  11, 327.688}, {11.3125, 1.0625}} -> {-1.90578, -3.60489, 0.102784, 
  0.102784}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .17419 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.00414747 0.952381 [
[ 0 0 0 0 ]
[ 1 .17419 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .17419 L
0 .17419 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.00414747 0.97619 0.0348694 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
gsave
.06989 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.10061 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.13134 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.16206 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .19278 .01613 0 0 Mshowa
gsave
.2235 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.25422 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.28495 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.31567 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.34639 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .37711 .01613 0 0 Mshowa
gsave
.40783 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.43856 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.46928 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.5 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.53072 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.56144 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.59217 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .62289 .01613 0 0 Mshowa
gsave
.65361 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.68433 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.71505 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.74578 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .7765 .01613 0 0 Mshowa
gsave
.80722 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.83794 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.86866 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.89939 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .93011 .01613 0 0 Mshowa
gsave
.96083 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0379416 0.97619 0.0686636 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
.5 Mabswid
newpath
.06989 .01613 .01229 0 365.73 arc
s
.07358 .01982 m
.07358 .01982 .00184 0 365.73 arc
F
.06621 .01982 m
.06621 .01982 .00184 0 365.73 arc
F
.06621 .01121 m
.06743 .01367 L
.07235 .01367 L
.07358 .01121 L
s
newpath
.16206 .01613 .01229 0 365.73 arc
s
.16575 .01982 m
.16575 .01982 .00184 0 365.73 arc
F
.15837 .01982 m
.15837 .01982 .00184 0 365.73 arc
F
.15837 .01121 m
.1596 .01367 L
.16452 .01367 L
.16575 .01121 L
s
newpath
.19278 .01613 .01229 0 365.73 arc
s
.19647 .01982 m
.19647 .01982 .00184 0 365.73 arc
F
.18909 .01982 m
.18909 .01982 .00184 0 365.73 arc
F
.18909 .01367 m
.19032 .01121 L
.19524 .01121 L
.19647 .01367 L
s
newpath
.25422 .01613 .01229 0 365.73 arc
s
.25791 .01982 m
.25791 .01982 .00184 0 365.73 arc
F
.25054 .01982 m
.25054 .01982 .00184 0 365.73 arc
F
.25054 .01121 m
.25177 .01367 L
.25668 .01367 L
.25791 .01121 L
s
newpath
.28495 .01613 .01229 0 365.73 arc
s
.28863 .01982 m
.28863 .01982 .00184 0 365.73 arc
F
.28126 .01982 m
.28126 .01982 .00184 0 365.73 arc
F
.28126 .01121 m
.28249 .01367 L
.2874 .01367 L
.28863 .01121 L
s
newpath
.31567 .01613 .01229 0 365.73 arc
s
.31935 .01982 m
.31935 .01982 .00184 0 365.73 arc
F
.31198 .01982 m
.31198 .01982 .00184 0 365.73 arc
F
.31198 .01121 m
.31321 .01367 L
.31813 .01367 L
.31935 .01121 L
s
newpath
.37711 .01613 .01229 0 365.73 arc
s
.3808 .01982 m
.3808 .01982 .00184 0 365.73 arc
F
.37343 .01982 m
.37343 .01982 .00184 0 365.73 arc
F
.37343 .01367 m
.37465 .01121 L
.37957 .01121 L
.3808 .01367 L
s
newpath
.43856 .01613 .01229 0 365.73 arc
s
.44224 .01982 m
.44224 .01982 .00184 0 365.73 arc
F
.43487 .01982 m
.43487 .01982 .00184 0 365.73 arc
F
.43487 .01121 m
.4361 .01367 L
.44101 .01367 L
.44224 .01121 L
s
newpath
.46928 .01613 .01229 0 365.73 arc
s
.47296 .01982 m
.47296 .01982 .00184 0 365.73 arc
F
.46559 .01982 m
.46559 .01982 .00184 0 365.73 arc
F
.46559 .01121 m
.46682 .01367 L
.47174 .01367 L
.47296 .01121 L
s
newpath
.5 .01613 .01229 0 365.73 arc
s
.50369 .01982 m
.50369 .01982 .00184 0 365.73 arc
F
.49631 .01982 m
.49631 .01982 .00184 0 365.73 arc
F
.49631 .01121 m
.49754 .01367 L
.50246 .01367 L
.50369 .01121 L
s
newpath
.56144 .01613 .01229 0 365.73 arc
s
.56513 .01982 m
.56513 .01982 .00184 0 365.73 arc
F
.55776 .01982 m
.55776 .01982 .00184 0 365.73 arc
F
.55776 .01121 m
.55899 .01367 L
.5639 .01367 L
.56513 .01121 L
s
newpath
.62289 .01613 .01229 0 365.73 arc
s
.62657 .01982 m
.62657 .01982 .00184 0 365.73 arc
F
.6192 .01982 m
.6192 .01982 .00184 0 365.73 arc
F
.6192 .01367 m
.62043 .01121 L
.62535 .01121 L
.62657 .01367 L
s
newpath
.68433 .01613 .01229 0 365.73 arc
s
.68802 .01982 m
.68802 .01982 .00184 0 365.73 arc
F
.68065 .01982 m
.68065 .01982 .00184 0 365.73 arc
F
.68065 .01121 m
.68187 .01367 L
.68679 .01367 L
.68802 .01121 L
s
newpath
.71505 .01613 .01229 0 365.73 arc
s
.71874 .01982 m
.71874 .01982 .00184 0 365.73 arc
F
.71137 .01982 m
.71137 .01982 .00184 0 365.73 arc
F
.71137 .01121 m
.7126 .01367 L
.71751 .01367 L
.71874 .01121 L
s
newpath
.74578 .01613 .01229 0 365.73 arc
s
.74946 .01982 m
.74946 .01982 .00184 0 365.73 arc
F
.74209 .01982 m
.74209 .01982 .00184 0 365.73 arc
F
.74209 .01121 m
.74332 .01367 L
.74823 .01367 L
.74946 .01121 L
s
newpath
.7765 .01613 .01229 0 365.73 arc
s
.78018 .01982 m
.78018 .01982 .00184 0 365.73 arc
F
.77281 .01982 m
.77281 .01982 .00184 0 365.73 arc
F
.77281 .01367 m
.77404 .01121 L
.77896 .01121 L
.78018 .01367 L
s
newpath
.80722 .01613 .01229 0 365.73 arc
s
.81091 .01982 m
.81091 .01982 .00184 0 365.73 arc
F
.80353 .01982 m
.80353 .01982 .00184 0 365.73 arc
F
.80353 .01121 m
.80476 .01367 L
.80968 .01367 L
.81091 .01121 L
s
newpath
.83794 .01613 .01229 0 365.73 arc
s
.84163 .01982 m
.84163 .01982 .00184 0 365.73 arc
F
.83425 .01982 m
.83425 .01982 .00184 0 365.73 arc
F
.83425 .01121 m
.83548 .01367 L
.8404 .01367 L
.84163 .01121 L
s
newpath
.93011 .01613 .01229 0 365.73 arc
s
.93379 .01982 m
.93379 .01982 .00184 0 365.73 arc
F
.92642 .01982 m
.92642 .01982 .00184 0 365.73 arc
F
.92642 .01367 m
.92765 .01121 L
.93257 .01121 L
.93379 .01367 L
s
newpath
.96083 .01613 .01229 0 365.73 arc
s
.96452 .01982 m
.96452 .01982 .00184 0 365.73 arc
F
.95714 .01982 m
.95714 .01982 .00184 0 365.73 arc
F
.95714 .01121 m
.95837 .01367 L
.96329 .01367 L
.96452 .01121 L
s
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0717358 0.97619 0.102458 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
[(0)] .06989 .01613 0 0 Mshowa
gsave
.10061 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.13134 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .16206 .01613 0 0 Mshowa
[(1)] .19278 .01613 0 0 Mshowa
gsave
.2235 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .25422 .01613 0 0 Mshowa
[(0)] .28495 .01613 0 0 Mshowa
[(1)] .31567 .01613 0 0 Mshowa
gsave
.34639 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .37711 .01613 0 0 Mshowa
gsave
.40783 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .43856 .01613 0 0 Mshowa
[(1)] .46928 .01613 0 0 Mshowa
[(0)] .5 .01613 0 0 Mshowa
gsave
.53072 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .56144 .01613 0 0 Mshowa
gsave
.59217 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .62289 .01613 0 0 Mshowa
gsave
.65361 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .68433 .01613 0 0 Mshowa
[(1)] .71505 .01613 0 0 Mshowa
[(1)] .74578 .01613 0 0 Mshowa
[(0)] .7765 .01613 0 0 Mshowa
[(1)] .80722 .01613 0 0 Mshowa
[(1)] .83794 .01613 0 0 Mshowa
gsave
.86866 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.89939 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .93011 .01613 0 0 Mshowa
[(0)] .96083 .01613 0 0 Mshowa
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.10553 0.97619 0.136252 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
.5 Mabswid
.06068 .01613 m
.07911 .01613 L
s
.06989 .02535 m
.06989 .00691 L
s
.15284 .01613 m
.17127 .01613 L
s
.16206 .02535 m
.16206 .00691 L
s
.18626 .00961 m
.1993 .02265 L
s
.18626 .02265 m
.1993 .00961 L
s
.24501 .01613 m
.26344 .01613 L
s
.25422 .02535 m
.25422 .00691 L
s
.27843 .00961 m
.29146 .02265 L
s
.27843 .02265 m
.29146 .00961 L
s
.30645 .01613 m
.32488 .01613 L
s
.31567 .02535 m
.31567 .00691 L
s
.3679 .01613 m
.38633 .01613 L
s
.37711 .02535 m
.37711 .00691 L
s
.43204 .00961 m
.44507 .02265 L
s
.43204 .02265 m
.44507 .00961 L
s
.46276 .00961 m
.4758 .02265 L
s
.46276 .02265 m
.4758 .00961 L
s
.49078 .01613 m
.50922 .01613 L
s
.5 .02535 m
.5 .00691 L
s
.55493 .00961 m
.56796 .02265 L
s
.55493 .02265 m
.56796 .00961 L
s
.61637 .00961 m
.6294 .02265 L
s
.61637 .02265 m
.6294 .00961 L
s
.67781 .00961 m
.69085 .02265 L
s
.67781 .02265 m
.69085 .00961 L
s
.70584 .01613 m
.72427 .01613 L
s
.71505 .02535 m
.71505 .00691 L
s
.73656 .01613 m
.75499 .01613 L
s
.74578 .02535 m
.74578 .00691 L
s
.76728 .01613 m
.78571 .01613 L
s
.7765 .02535 m
.7765 .00691 L
s
.8007 .00961 m
.81374 .02265 L
s
.8007 .02265 m
.81374 .00961 L
s
.82873 .01613 m
.84716 .01613 L
s
.83794 .02535 m
.83794 .00691 L
s
.92359 .00961 m
.93662 .02265 L
s
.92359 .02265 m
.93662 .00961 L
s
.95161 .01613 m
.97005 .01613 L
s
.96083 .02535 m
.96083 .00691 L
s
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.139324 0.97619 0.170046 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
.5 Mabswid
.06338 .00961 m
.07641 .02265 L
s
.06338 .02265 m
.07641 .00961 L
s
.15554 .00961 m
.16858 .02265 L
s
.15554 .02265 m
.16858 .00961 L
s
.18626 .00961 m
.1993 .02265 L
s
.18626 .02265 m
.1993 .00961 L
s
.24771 .00961 m
.26074 .02265 L
s
.24771 .02265 m
.26074 .00961 L
s
.27573 .01613 m
.29416 .01613 L
s
.28495 .02535 m
.28495 .00691 L
s
.30915 .00961 m
.32219 .02265 L
s
.30915 .02265 m
.32219 .00961 L
s
.3679 .01613 m
.38633 .01613 L
s
.37711 .02535 m
.37711 .00691 L
s
.42934 .01613 m
.44777 .01613 L
s
.43856 .02535 m
.43856 .00691 L
s
.46006 .01613 m
.47849 .01613 L
s
.46928 .02535 m
.46928 .00691 L
s
.49348 .00961 m
.50652 .02265 L
s
.49348 .02265 m
.50652 .00961 L
s
.55223 .01613 m
.57066 .01613 L
s
.56144 .02535 m
.56144 .00691 L
s
.61637 .00961 m
.6294 .02265 L
s
.61637 .02265 m
.6294 .00961 L
s
.67512 .01613 m
.69355 .01613 L
s
.68433 .02535 m
.68433 .00691 L
s
.70854 .00961 m
.72157 .02265 L
s
.70854 .02265 m
.72157 .00961 L
s
.73926 .00961 m
.75229 .02265 L
s
.73926 .02265 m
.75229 .00961 L
s
.76728 .01613 m
.78571 .01613 L
s
.7765 .02535 m
.7765 .00691 L
s
.798 .01613 m
.81644 .01613 L
s
.80722 .02535 m
.80722 .00691 L
s
.83142 .00961 m
.84446 .02265 L
s
.83142 .02265 m
.84446 .00961 L
s
.92359 .00961 m
.93662 .02265 L
s
.92359 .02265 m
.93662 .00961 L
s
.95431 .00961 m
.96735 .02265 L
s
.95431 .02265 m
.96735 .00961 L
s
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{303.188, 52.75},
  ImageMargins->{{34, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004_0000=1000`40O003h00Oogoo<7oo000IOooo
F]H6F]HAOol001Uoo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Eo
o`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Io
o`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Eo
o`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Io
o`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Io
o`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Io
o`03F]Ioogoo00Eoo`03F]Ioogoo00moo`006Goo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWoo
Ool01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWoo
Ool01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWoo
Ool01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWoo
Ool01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWoo
Ool01Goo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWoo
Ool01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol03goo000IOol00e[FOomoo`05Ool0
0e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOol000040002Ool00e[FOomoo`06
Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool01E[FOomoogoo
000017oo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWoo
Ool01Woo00=JeWooOol01Goo00=JeWooOol01Woo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Eoo`03
F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`05F]IoogooOol00004Ool00e[FOomoo`06
Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOol000040002Ool00e[FOomo
o`05Ool00e[FOomoo`0?Ool001Uoo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03
F]Ioogoo00Eoo`05F]IoogooOol00004Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06
Ool00e[FOomoo`06Ool00e[FOomoo`05Ool01U[FOomoo`00Ool000=oo`03F]Ioogoo00Ioo`03F]Io
ogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Io
ogoo00Ioo`06F]Ioogoo001oo`000goo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo
00=JeWooOol01Goo00IJeWooOol007oo0003Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomo
o`06Ool00e[FOomoo`06Ool01E[FOomoogoo000017oo00=JeWooOol01Goo00=JeWooOol03goo000I
Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool01E[FOomoogoo
000017oo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWoo
Ool01Goo00IJeWooOol007oo0003Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool0
0e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool01U[FOomoo`00Ool0
00=oo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`06F]Ioogoo
001oo`000goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo00EJ
eWooOomoo`0000Aoo`03F]Ioogoo00Eoo`03F]Ioogoo00moo`006Goo00=JeWooOol01Goo00=JeWoo
Ool01Woo00=JeWooOol01Woo00=JeWooOol01Goo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Ioo`03
F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`06F]Ioogoo001oo`00
0goo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol0
1Woo00=JeWooOol01Goo00=JeWooOol01Woo00IJeWooOol007oo0003Ool00e[FOomoo`05Ool00e[F
Oomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool01U[FOomoo`00Ool000=oo`03F]Ioogoo00Io
o`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`05F]IoogooOol00004Ool00e[FOomo
o`05Ool00e[FOomoo`0?Ool001Uoo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03
F]Ioogoo00Eoo`05F]IoogooOol00004Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06
Ool00e[FOomoo`06Ool00e[FOomoo`05Ool01U[FOomoo`00Ool000=oo`03F]Ioogoo00Ioo`03F]Io
ogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Io
ogoo00Ioo`06F]Ioogoo001oo`000goo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo
00=JeWooOol01Goo00IJeWooOol007oo0003Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomo
o`06Ool00e[FOomoo`06Ool01E[FOomoogoo000017oo00=JeWooOol01Goo00=JeWooOol03goo000I
Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`02
0004Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomo
o`05Ool01E[FOomoogoo000017oo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=J
eWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00EJeWooOomoo`0000Ao
o`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`05F]IoogooOol0
0004Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomo
o`020004Ool00e[FOomoo`05Ool00e[FOomoo`0?Ool001UooomJePIJeQ5oo`006Goooe[F1U[F4Goo
000IOol015[FOomoogoo0P000Woo00=JeWooOol01Woo00=JeWooOol01Woo00AJeWooOomoo`80009o
o`04F]IoogooOol20003Ool00e[FOomoo`06Ool015[FOomoogoo0P000Woo00AJeWooOomoo`8000=o
o`04F]IoogooOol20003Ool00e[FOomoo`05Ool00e[FOomoo`02Ool01@00OomoogooF]H027oo00AJ
eWooOomoo`8000=oo`04F]IoogooOol20002Ool015[FOomoogoo0P000goo00=JeWooOol01Woo00AJ
eWooOomoo`80009oo`03F]Ioogoo00Ioo`04F]IoogooOol20003Ool00e[FOomoo`05Ool015[FOomo
ogoo0P000goo00AJeWooOomoo`8000=oo`04F]IoogooOol20002Ool015[FOomoogoo0P000goo00AJ
eWooOomoo`8000=oo`04F]IoogooOol20002Ool00e[FOomoo`06Ool00e[FOomoo`06Ool015[FOomo
ogoo0P000goo00AJeWooOomoo`80009oo`03F]Ioogoo00moo`006Goo00AJeWoo0000009oo`800003
F]Ioogoo00Ioo`03F]Ioogoo00Ioo`04F]Ioo`000002Ool2000015[FOol000000Woo0P0000=ooe[F
Ool01goo00AJeWoo0000009oo`800004F]Ioo`000002Ool200000gooF]Ioo`020002Ool200000goo
F]Ioo`06Ool00e[FOomoo`0200001Goo000007ooF]H027oo00AJeWoo0000009oo`800003OomJeWoo
0080009oo`800004F]Ioo`000002Ool200000gooF]Ioo`07Ool015[FOol000000Woo0P0000=JeWoo
Ool01Woo00AJeWoo0000009oo`800003OomJeWoo00Ioo`04F]Ioo`000002Ool200000gooF]Ioo`02
0002Ool200000gooF]Ioo`020002Ool2000015[FOol000000Woo0P0000=ooe[FOol00P000Woo0P00
00=ooe[FOol00P000Woo0P0000=JeWooOol01Woo00=JeWooOol01Woo00AJeWoo0000009oo`800003
OomJeWoo0080009oo`800003F]Ioogoo00moo`006Goo00UJeP00Ool007oo001oo`00F]H027oo00=J
eWooOol01Woo00YJeP00Ool007oo001oo`00F]H0009oo`800004Ool007ooF]H8Ool02U[F001oo`00
Ool007oo001JeP000Woo00l007oo000007ooF]H007oo001oo`00Ool007ooF]H01goo00AJeWoo001o
o`@00003OomJeWoo00Moo`0CF]H007oo001oo`00Ool007ooF]H007oo001oo`00Ool005[F00000Woo
00H007oo000007ooF]H8Ool02E[F001oo`00Ool007oo001JeP08Ool015[F001oogoo0P0000Aoo`00
OomJePMoo`06F]H007ooOol007oo0P0001Aooe[F001oo`00Ool007oo001ooe[F001oo`00Ool007oo
001JeP000Woo0P0000Moo`00OomJeP00Ool00002Ool200002WooF]H007oo001oo`00Ool005[F27oo
00=JeWooOol01Woo00AJeP00Oomoo`80000<Ool007ooF]H007oo001oo`00Ool005[F4Goo000IOol0
0e[FOomoo`0300000goo001JeP08Ool00e[FOomoo`06Ool00e[F001oo`03000017oo001JeP000Woo
00H007oo001oo`00F]H8Ool00e[F001oo`030002Ool015[F001oogoo0P000Woo00<005[F00000Woo
0P000Woo00<005[FOol01Woo00YJeP00Oomoo`00Ool007oo001JePQoo`03F]H007oo00<0009oo`04
001JeWooOol3000017oo001JeP000Woo0P000Woo00<005[FOol01goo00=JeWooOol00`0000=oo`00
F]H027oo00IJeP00Ool007oo0002Ool00`00F]Ioo`06Ool015[F001oogoo0P000Woo00<005[F0000
0Woo0P000Woo00@005[F001oo`<0009oo`0;F]H007ooOol007oo001oo`00F]H00002Ool20002Ool0
1000F]H007oo0`000Woo00=JeWooOol01Woo00=JeWooOol01Woo00IJeP00Ool007oo0002Ool01000
F]Ioogoo0`0000=oo`00F]H04Goo000IOol00e[FOomoo`04Ool00`00F]Ioo`07Ool00e[FOomoo`06
Ool00e[F001oo`04Ool00`00F]H00006Ool00`00F]Ioo`07Ool00e[F001oo`05Ool00e[F001oo`05
Ool00`00F]H00006Ool00`00F]Ioo`06Ool00e[F001oo`05Ool00`00F]Ioo`07Ool00e[F001oo`05
Ool00`00F]Ioo`05Ool00`00F]H00006Ool00`00F]Ioo`07Ool00e[FOomoo`04Ool00`00F]Ioo`07
Ool00e[F001oo`05Ool00`00F]Ioo`06Ool00e[F001oo`05Ool00`00F]H00006Ool00`00F]H00006
Ool00e[F001oo`05Ool00`00F]H00006Ool00`00F]H00006Ool00e[FOomoo`06Ool00e[FOomoo`06
Ool00e[F001oo`05Ool00`00F]Ioo`05Ool00`00F]Ioo`0@Ool001Uoo`03F]H0000000@00003OomJ
eWoo00Moo`03F]Ioogoo00Ioo`05F]H00000001oo`0200000gooF]Ioo`0600000gooF]Ioo`07Ool0
0e[F0000000500000e[FOol0000500000gooF]Ioo`0600000gooF]Ioo`06Ool00e[FOol000050000
0gooF]Ioo`07Ool00e[FOol0000500000gooF]H0000500000gooF]Ioo`0600000gooF]Ioo`07Ool0
0e[F0000000400000gooF]Ioo`07Ool00e[FOol0000500000gooF]Ioo`06Ool00e[FOol000050000
0gooF]Ioo`0600000gooF]Ioo`0600000e[FOol0000500000gooF]Ioo`0600000gooF]Ioo`060000
0e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOol0000500000gooF]H0000500000gooF]Ioo`0@Ool0
01Uoo`03F]H0000000=oo`03001ooe[F00Qoo`03F]Ioogoo00Ioo`03F]H0000000=oo`04001ooe[F
Ool20003Ool00`00OomJeP08Ool00e[F00000003Ool01000OomJeWoo0P000goo00D007ooF]Ioo`00
00=oo`800003OomJeWoo00Ioo`04F]Ioo`000003Ool00`00OomJeP08Ool00e[FOol00003Ool20000
17ooF]H000000goo00D007ooF]Ioo`0000=oo`800003OomJeWoo00Moo`03F]H0000000=oo`03001o
oe[F00Qoo`03F]Ioo`0000=oo`800003OomJeWoo00Ioo`04F]Ioo`000003Ool01@00OomJeWoo0000
0goo0P0000Aooe[FOol000=oo`800004F]Ioo`000003Ool01@00OomJeWoo00000goo0P0000Aooe[F
Ool000=oo`800003F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioo`0000=oo`800004OomJeP000003
Ool00`00OomJeP0AOol001Uoo`03F]Ioogoo00<0009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03
F]Ioogoo00<0009oo`04F]IoogooOol30002Ool00e[FOomoo`06Ool00e[FOomoo`030002Ool015[F
Oomoogoo0`000Woo00=JeWooOol00`000goo00=JeWooOol01Goo00AJeWooOomoo`<0009oo`03F]Io
ogoo00Ioo`03F]Ioogoo00<000=oo`03F]Ioogoo00<0009oo`03F]Ioogoo00<000=oo`03F]Ioogoo
00Ioo`03F]Ioogoo00<0009oo`03F]Ioogoo00Ioo`03F]Ioogoo00<000=oo`03F]Ioogoo00Eoo`04
F]IoogooOol30002Ool00e[FOomoo`030003Ool00e[FOomoo`030002Ool015[FOomoogoo0`000Woo
00=JeWooOol00`000goo00=JeWooOol00`000Woo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWoo
Ool00`000goo00=JeWooOol00`000Woo00=JeWooOol03goo000IOoooF]H6F]HAOol001UooomJePIJ
eQ5oo`006Goo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=J
eWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo00=J
eWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=J
eWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo00=J
eWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=J
eWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Woo00=J
eWooOol01Goo00=JeWooOol03goo000IOol00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06
Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06
Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`06
Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06
Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05
Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06
Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`0?Ool001Uoo`04F]Ioogoo0004Ool00e[F
Oomoo`06Ool00e[FOomoo`06Ool00e[F000000030002Ool00e[FOol000040002Ool00e[FOomoo`06
Ool01E[FOomoogoo00000goo00EJeWooOomoo`0000Aoo`03F]Ioo`0000@0009oo`03F]Ioogoo00Eo
o`05F]IoogooOol00004Ool00e[FOomoo`06Ool01E[FOomoogoo000017oo00=JeP0000000`000Woo
00EJeWooOomoo`0000Aoo`03F]Ioogoo00Ioo`04F]Ioogoo0004Ool00e[FOomoo`06Ool01E[FOomo
ogoo000017oo00=JeWooOol01Goo00=JeWoo000010000Woo00=JeWoo000010000Woo00=JeWoo0000
100000=ooe[FOol00Woo00<007ooOol00Woo00=JeWoo000010000Woo00=JeWoo0000100000=ooe[F
Ool01goo00=JeWooOol01Woo00=JeWoo000010000Woo00AJeWooOol000Aoo`03F]Ioogoo00moo`00
6Goo00EJeWoo001oo`0000=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`04F]Ioogoo0004Ool01E[F
Oomoogoo000017oo00=JeWooOol01Woo00IJeWooOol007oo0002Ool01U[FOomoo`00Ool000=oo`05
F]IoogooOol00004Ool00e[FOomoo`05Ool01U[FOomoo`00Ool000=oo`03F]Ioogoo00Ioo`06F]Io
ogoo001oo`000goo00AJeWooOol000Aoo`06F]Ioogoo001oo`000goo00=JeWooOol01Woo00EJeWoo
001oo`0000=oo`03F]Ioogoo00Ioo`06F]Ioogoo001oo`000goo00=JeWooOol01Goo00EJeWooOomo
o`0000Aoo`05F]IoogooOol00004Ool01E[FOomoogoo00000goo00IJeWooOol007oo0003Ool01E[F
Oomoogoo000017oo00EJeWooOomoo`0000=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`05F]Ioogoo
Ool00004Ool01E[FOol007oo00000goo00=JeWooOol03goo000IOol01E[FOol007oo00000goo00=J
eWooOol01Woo00=JeWooOol01Woo00AJeWooOol000Aoo`05F]IoogooOol00004Ool00e[FOomoo`06
Ool01U[FOomoo`00Ool0009oo`06F]Ioogoo001oo`000goo00EJeWooOomoo`0000Aoo`03F]Ioogoo
00Eoo`06F]Ioogoo001oo`000goo00=JeWooOol01Woo00IJeWooOol007oo0003Ool015[FOomoo`00
17oo00IJeWooOol007oo0003Ool00e[FOomoo`06Ool01E[FOol007oo00000goo00=JeWooOol01Woo
00IJeWooOol007oo0003Ool00e[FOomoo`05Ool01E[FOomoogoo000017oo00EJeWooOomoo`0000Ao
o`05F]IoogooOol00003Ool01U[FOomoo`00Ool000=oo`05F]IoogooOol00004Ool01E[FOomoogoo
00000goo00=JeWooOol01Woo00=JeWooOol01Woo00EJeWooOomoo`0000Aoo`05F]Ioo`00Ool00003
Ool00e[FOomoo`0?Ool001Uoo`05F]Ioo`00Ool00003Ool00e[FOomoo`06Ool00e[FOomoo`06Ool0
15[FOomoo`0017oo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Ioo`06F]Ioogoo001oo`000Woo00IJ
eWooOol007oo0003Ool01E[FOomoogoo000017oo00=JeWooOol01Goo00IJeWooOol007oo0003Ool0
0e[FOomoo`06Ool01U[FOomoo`00Ool000=oo`04F]Ioogoo0004Ool01U[FOomoo`00Ool000=oo`03
F]Ioogoo00Ioo`05F]Ioo`00Ool00003Ool00e[FOomoo`06Ool01U[FOomoo`00Ool000=oo`03F]Io
ogoo00Eoo`05F]IoogooOol00004Ool01E[FOomoogoo000017oo00EJeWooOomoo`0000=oo`06F]Io
ogoo001oo`000goo00EJeWooOomoo`0000Aoo`05F]IoogooOol00003Ool00e[FOomoo`06Ool00e[F
Oomoo`06Ool01E[FOomoogoo000017oo00EJeWoo001oo`0000=oo`03F]Ioogoo00moo`006Goo00EJ
eWoo001oo`0000=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`04F]Ioogoo0004Ool01E[FOomoogoo
000017oo00=JeWooOol01Woo00IJeWooOol007oo0002Ool01U[FOomoo`00Ool000=oo`05F]Ioogoo
Ool00004Ool00e[FOomoo`05Ool01U[FOomoo`00Ool000=oo`03F]Ioogoo00Ioo`06F]Ioogoo001o
o`000goo00AJeWooOol000Aoo`06F]Ioogoo001oo`000goo00=JeWooOol01Woo00EJeWoo001oo`00
00=oo`03F]Ioogoo00Ioo`06F]Ioogoo001oo`000goo00=JeWooOol01Goo00EJeWooOomoo`0000Ao
o`05F]IoogooOol00004Ool01E[FOomoogoo00000goo00IJeWooOol007oo0003Ool01E[FOomoogoo
000017oo00EJeWooOomoo`0000=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`05F]IoogooOol00004
Ool01E[FOol007oo00000goo00=JeWooOol03goo000IOol015[FOomoo`0017oo00=JeWooOol01Woo
00=JeWooOol01Woo00AJeWoo000000Aoo`03F]Ioogoo008000Aoo`03F]Ioogoo00Ioo`05F]Ioogoo
Ool00003Ool01E[FOomoogoo000017oo00=JeWooOol00P0017oo00=JeWooOol01Goo00EJeWooOomo
o`0000Aoo`03F]Ioogoo00Ioo`05F]IoogooOol00004Ool015[FOol0000017oo00EJeWooOomoo`00
00Aoo`03F]Ioogoo00Ioo`04F]Ioogoo0004Ool00e[FOomoo`06Ool01E[FOomoogoo000017oo00=J
eWooOol01Goo00=JeWooOol00P0017oo00=JeWooOol00P0017oo00=JeWooOol00P000goo00EJeWoo
Oomoo`0000Aoo`03F]Ioogoo008000Aoo`03F]Ioogoo008000=oo`03F]Ioogoo00Ioo`03F]Ioogoo
00Ioo`03F]Ioogoo008000Aoo`04F]Ioogoo0004Ool00e[FOomoo`0?Ool001UooomJePIJeQ5oo`00
6Goooe[F1U[F4Goo000IOol00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomo
o`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomo
o`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomo
o`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomo
o`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomo
o`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomo
o`06Ool00e[FOomoo`05Ool00e[FOomoo`0?Ool001Uoo`05F]IoogooOomJeP03Ool00e[FOomoo`06
Ool00e[FOomoo`06Ool01E[FOomoogooF]H00goo00=JeWooOol01Woo00=JeWooOol01Woo00EJeWoo
Oomooe[F00=oo`03F]Ioogoo00Ioo`05F]IoogooOomJeP04Ool00e[FOomoo`05Ool00e[FOomoo`02
Ool01E[FOomoogooF]H027oo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol00Woo00EJeWoo
Oomooe[F00Qoo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03
F]Ioogoo00Ioo`05F]IoogooOomJeP04Ool01E[FOomoogooF]H00goo00=JeWooOol00Woo00EJeWoo
Oomooe[F00Qoo`05F]IoogooOomJeP03Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`06
Ool01E[FOomoogooF]H00goo00=JeWooOol03goo000IOol01E[FOomoogooF]H00goo00=JeWooOol0
1Woo00=JeWooOol01Woo00EJeWooOomooe[F00=oo`04F]IoogooF]H3Ool00e[FOomJeP08Ool01E[F
OomoogooF]H00goo00AJeWooOomJeP=oo`03F]Iooe[F00=oo`03F]Ioogoo009oo`03F]Ioogoo00Eo
o`03F]Ioogoo009oo`05F]IoogooOomJeP08Ool00e[FOomJeP03Ool01U[FOomooe[FOomJeP=oo`03
F]Iooe[F00Aoo`05F]IoogooOomJeP08Ool00e[FOomJeP03Ool00e[FOomJeP08Ool00e[FOomJeP03
Ool015[FOomooe[F1goo00AJeWooOomJeP=oo`03F]Iooe[F00=oo`03F]Ioogoo009oo`05F]Ioogoo
OomJeP03Ool00e[FOomoo`02Ool01E[FOomoogooF]H00Woo00AJeWooOomJeP9oo`05F]IoogooOomJ
eP03Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomJeP03Ool015[FOomooe[F0goo00EJeWoo
Oomooe[F015oo`006Goo00EJeWooOomooe[F00=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`05F]Io
ogooOomJeP03Ool01e[FOomoogooF]Iooe[F009oo`03F]Ioogoo00Ioo`05F]IoogooOomJeP03Ool0
1e[FOomoogooF]Iooe[F009oo`05F]IoogooOomJeP04Ool00e[FOomoo`05Ool00e[FOomoo`02Ool0
1E[FOomoogooF]H027oo00IJeWooOomJeWooF]H3Ool01U[FOomooe[FOomJeP9oo`03F]Ioogoo009o
o`05F]IoogooOomJeP08Ool01U[FOomooe[FOomJeP9oo`03F]Ioogoo00Ioo`06F]IoogooF]Iooe[F
0goo00=JeWooOol01Goo00MJeWooOomooe[FOomJeP02Ool01E[FOomoogooF]H017oo00EJeWooOomo
oe[F00=oo`03F]Ioogoo009oo`05F]IoogooOomJeP03Ool2F]H3Ool01E[FOomoogooF]H00goo00=J
eWooOol01Woo00=JeWooOol01Woo00IJeWooOomJeWooF]H3Ool01E[FOomoogooF]H00goo00=JeWoo
Ool03goo000IOol9F]H8Ool00e[FOomoo`06Ool9F]H4Ool01E[FOomoogooF]H027oo2E[F17oo00IJ
eWooOomooe[FOol6F]H00gooF]Ioo`06Ool00e[FOomJeP05F]H00gooF]Ioo`07Ool01E[FOomoogoo
F]H017oo00EJeWooOomooe[F00=oo`03F]Iooe[F00EJeP03OomJeWoo00Moo`05F]IoogooOomJeP03
Ool00e[FOomoo`06Ool01E[FOomoogooF]H017oo00=JeWooOol01Goo00=JeWooOol00Woo00IJeWoo
Oomooe[FOol6F]H00gooF]IJeP07F]H00gooF]IJeP04F]H00gooF]Ioo`02Ool00e[FOomoo`02Ool0
0e[FOomJeP06F]H8Ool00e[FOomoo`06Ool01E[FOomoogooF]H017oo2E[F4Goo000IOol01E[FOomo
ogooF]H00goo00=JeWooOol01Woo00=JeWooOol01Woo00EJeWooOomooe[F00=oo`07F]IoogooOomJ
eWooF]H00Woo00=JeWooOol01Woo00EJeWooOomooe[F00=oo`07F]IoogooOomJeWooF]H00Woo00EJ
eWooOomooe[F00Aoo`03F]Ioogoo00Eoo`03F]Ioogoo009oo`05F]IoogooOomJeP08Ool01U[FOomo
oe[FOomJeP=oo`06F]IoogooF]Iooe[F0Woo00=JeWooOol00Woo00EJeWooOomooe[F00Qoo`06F]Io
ogooF]Iooe[F0Woo00=JeWooOol01Woo00IJeWooOomJeWooF]H3Ool00e[FOomoo`05Ool01e[FOomo
ogooF]Iooe[F009oo`05F]IoogooOomJeP04Ool01E[FOomoogooF]H00goo00=JeWooOol00Woo00EJ
eWooOomooe[F00=oo`9JeP=oo`05F]IoogooOomJeP03Ool00e[FOomoo`06Ool00e[FOomoo`06Ool0
1U[FOomooe[FOomJeP=oo`05F]IoogooOomJeP03Ool00e[FOomoo`0?Ool001Uoo`05F]IoogooOomJ
eP03Ool00e[FOomoo`06Ool00e[FOomoo`06Ool01E[FOomoogooF]H00goo00AJeWooOomJeP=oo`03
F]Iooe[F00Qoo`05F]IoogooOomJeP03Ool015[FOomooe[F0goo00=JeWooF]H00goo00=JeWooOol0
0Woo00=JeWooOol01Goo00=JeWooOol00Woo00EJeWooOomooe[F00Qoo`03F]Iooe[F00=oo`06F]Io
ogooF]Iooe[F0goo00=JeWooF]H017oo00EJeWooOomooe[F00Qoo`03F]Iooe[F00=oo`03F]Iooe[F
00Qoo`03F]Iooe[F00=oo`04F]IoogooF]H7Ool015[FOomooe[F0goo00=JeWooF]H00goo00=JeWoo
Ool00Woo00EJeWooOomooe[F00=oo`03F]Ioogoo009oo`05F]IoogooOomJeP02Ool015[FOomooe[F
0Woo00EJeWooOomooe[F00=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Iooe[F00=oo`04F]Io
ogooF]H3Ool01E[FOomoogooF]H04Goo000IOol00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomo
o`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomo
o`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomo
o`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomo
o`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomo
o`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomo
o`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`0?Ool001UooomJePIJeQ5oo`006Goo
oe[F1U[F4Goo000IOol00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05
Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06
Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05
Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06
Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06
Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`06
Ool00e[FOomoo`05Ool00e[FOomoo`0?Ool001Uoo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Io
ogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Io
ogoo009oo`05F]IoogooOomJeP08Ool00e[FOomoo`05Ool00e[FOomoo`02Ool01E[FOomoogooF]H0
27oo00EJeWooOomooe[F00Aoo`05F]IoogooOomJeP03Ool00e[FOomoo`06Ool00e[FOomoo`06Ool0
1E[FOomoogooF]H00goo00=JeWooOol01Woo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol0
0Woo00EJeWooOomooe[F00Qoo`03F]Ioogoo00Eoo`03F]Ioogoo009oo`05F]IoogooOomJeP03Ool0
0e[FOomoo`02Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`06Ool0
0e[FOomoo`05Ool00e[FOomoo`0?Ool001Uoo`03F]Iooe[F00=oo`03F]Iooe[F00Qoo`03F]Ioogoo
00Ioo`03F]Iooe[F00=oo`03F]Iooe[F009oo`07F]IoogooOomJeWooF]H027oo00=JeWooF]H00goo
00=JeWooF]H017oo00EJeWooOomooe[F009oo`04F]IoogooF]H2Ool00e[FOomoo`05Ool00e[FOomo
o`02Ool01E[FOomoogooF]H027oo00EJeWooOomooe[F00Aoo`05F]IoogooOomJeP03Ool015[FOomo
oe[F0Woo00AJeWooOomJePQoo`05F]IoogooOomJeP03Ool00e[FOomoo`06Ool00e[FOomJeP03Ool0
15[FOomooe[F1goo00=JeWooOol00Woo00EJeWooOomooe[F009oo`04F]IoogooF]H2Ool00e[FOomJ
eP03Ool00e[FOomJeP04Ool01E[FOomoogooF]H00goo00=JeWooOol00Woo00=JeWooF]H00goo00=J
eWooF]H027oo00=JeWooOol01Woo00=JeWooF]H00goo00IJeWooOomJeWooF]H3Ool00e[FOomJeP0A
Ool001Uoo`06F]IoogooF]Iooe[F0Woo00=JeWooOol01Woo00=JeWooOol01Woo00IJeWooOomJeWoo
F]H2Ool01e[FOomoogooF]Iooe[F009oo`03F]Ioogoo00Ioo`06F]IoogooF]Iooe[F0Woo00=JeWoo
Ool00Woo00EJeWooOomooe[F00=oo`9JeP=oo`03F]Ioogoo00Eoo`03F]Ioogoo009oo`05F]Ioogoo
OomJeP08Ool01E[FOomoogooF]H017oo00EJeWooOomooe[F00=oo`04F]IoogooOol2F]H3Ool00e[F
Oomoo`06Ool01E[FOomoogooF]H00goo00=JeWooOol01Woo00IJeWooOomJeWooF]H3Ool00e[FOomo
o`05Ool00e[FOomoo`02Ool01E[FOomoogooF]H00goo0U[F0goo00IJeWooOomJeWooF]H2Ool00e[F
Oomoo`02Ool01E[FOomoogooF]H00goo00=JeWooOol00Woo00IJeWooOomJeWooF]H2Ool00e[FOomo
o`06Ool00e[FOomoo`06Ool01U[FOomooe[FOomJeP=oo`06F]IoogooF]Iooe[F0Woo00=JeWooOol0
3goo000IOol01E[FOomoogooF]H00goo00=JeWooOol01Woo00=JeWooOol01Woo00EJeWooOomooe[F
00=oo`03F]Ioogoo009oo`05F]IoogooOomJeP08Ool01E[FOomoogooF]H00goo00=JeWooF]H01E[F
00=ooe[FOol00Woo00=JeWooOol00Woo00=JeWooOol01Goo00=JeWooF]H01E[F00=ooe[FOol01goo
00=JeWooF]H01E[F00=ooe[FF]H01E[F00=ooe[FOol00Woo00=JeWooOol00Woo00=JeWooOol01Woo
2E[F27oo00EJeWooOomooe[F00Aoo`03F]Ioogoo00Eoo`03F]Iooe[F00EJeP03OomJeWoo009oo`03
F]Ioogoo009oo`05F]IoogooOomJeP03Ool00e[FOomJeP05F]H00gooF]Ioo`06F]H00gooF]Ioo`02
Ool01E[FOomoogooF]H027oo00=JeWooOol01Woo00EJeWooOomooe[F00Aoo`05F]IoogooOomJeP03
Ool00e[FOomoo`0?Ool001Uoo`06F]IoogooF]Iooe[F0Woo00=JeWooOol01Woo00=JeWooOol01Woo
00IJeWooOomJeWooF]H2Ool01e[FOomoogooF]Iooe[F009oo`03F]Ioogoo00Ioo`06F]IoogooF]Io
oe[F0Woo00=JeWooOol00Woo00EJeWooOomooe[F00=oo`9JeP=oo`03F]Ioogoo00Eoo`03F]Ioogoo
009oo`05F]IoogooOomJeP08Ool01E[FOomoogooF]H017oo00EJeWooOomooe[F00=oo`04F]Ioogoo
Ool2F]H3Ool00e[FOomoo`06Ool01E[FOomoogooF]H00goo00=JeWooOol01Woo00IJeWooOomJeWoo
F]H3Ool00e[FOomoo`05Ool00e[FOomoo`02Ool01E[FOomoogooF]H00goo0U[F0goo00IJeWooOomJ
eWooF]H2Ool00e[FOomoo`02Ool01E[FOomoogooF]H00goo00=JeWooOol00Woo00IJeWooOomJeWoo
F]H2Ool00e[FOomoo`06Ool00e[FOomoo`06Ool01U[FOomooe[FOomJeP=oo`06F]IoogooF]Iooe[F
0Woo00=JeWooOol03goo000IOol00e[FOomJeP03Ool00e[FOomJeP08Ool00e[FOomoo`06Ool00e[F
OomJeP03Ool00e[FOomJeP02Ool01e[FOomoogooF]Iooe[F00Qoo`03F]Iooe[F00=oo`03F]Iooe[F
00Aoo`05F]IoogooOomJeP02Ool015[FOomooe[F0Woo00=JeWooOol01Goo00=JeWooOol00Woo00EJ
eWooOomooe[F00Qoo`05F]IoogooOomJeP04Ool01E[FOomoogooF]H00goo00AJeWooOomJeP9oo`04
F]IoogooF]H8Ool01E[FOomoogooF]H00goo00=JeWooOol01Woo00=JeWooF]H00goo00AJeWooOomJ
ePMoo`03F]Ioogoo009oo`05F]IoogooOomJeP02Ool015[FOomooe[F0Woo00=JeWooF]H00goo00=J
eWooF]H017oo00EJeWooOomooe[F00=oo`03F]Ioogoo009oo`03F]Iooe[F00=oo`03F]Iooe[F00Qo
o`03F]Ioogoo00Ioo`03F]Iooe[F00=oo`06F]IoogooF]Iooe[F0goo00=JeWooF]H04Goo000IOol0
0e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool0
0e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool0
0e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool0
0e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool0
0e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool0
0e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`05Ool0
0e[FOomoo`0?Ool001UooomJePIJeQ5oo`00ogoo<7oo0000\
\>"],
  ImageRangeCache->{{{0, 302.188}, {51.75, 0}} -> {-0.0340091, -0.00435484, 
  0.00353429, 0.00353429}, {{9.5625, 292.563}, {50.5, 
  41.375}} -> {-1.88375, -0.168851, 0.115076, 0.115076}, {{9.5625, 292.563}, {
  40.4375, 31.3125}} -> {-1.88375, -1.32681, 0.115076, 0.115076}, {{9.5625, 
  292.563}, {30.4375, 21.3125}} -> {-1.88375, -2.47757, 0.115076, 
  0.115076}, {{9.5625, 292.563}, {20.375, 11.25}} -> {-1.88375, -3.63553, 
  0.115076, 0.115076}, {{9.5625, 292.563}, {10.3125, 
  1.1875}} -> {-1.88375, -4.79348, 0.115076, 0.115076}}],

Cell[OutputFormData[
"\<\
{{{1}, {0}, {0}, {0}, {1}}, {{1}, {0}, {0}, {0}, {1}}}\
\>", 
"\<\
{{{1}, {0}, {0}, {0}, {1}}, {{1}, {0}, {0}, {0}, {1}}}\
\>"], "Output",
  Evaluatable->False]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
The following code simulates quantum key distribution in the \
presence of eavesdropping.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["\<\
SeedRandom[123456];
DistributeKeyUsingQKD[0.75, 5, Eavesdropping->True]\
\>", "Input",
  AspectRatioFixed->True],

Cell["\<\
evePresentQ = True
{2, 3, 5, 6, 10, 11, 12, 16, 18, 19, 20, 24, 27, 28, 29}
Channel is insecure! Do not exchange a key.\
\>", "Print",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .10323 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.00245776 0.952381 [
[ 0 0 0 0 ]
[ 1 .10323 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .10323 L
0 .10323 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.00245776 0.97619 0.0331797 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
0 g
.06338 .02265 m
.07641 .00961 L
s
.0914 .01613 m
.10983 .01613 L
s
.12482 .02265 m
.13785 .00961 L
s
.15554 .02265 m
.16858 .00961 L
s
.18626 .02265 m
.1993 .00961 L
s
.21699 .00961 m
.23002 .02265 L
s
.24771 .00961 m
.26074 .02265 L
s
.27573 .01613 m
.29416 .01613 L
s
.30915 .00961 m
.32219 .02265 L
s
.33987 .02265 m
.35291 .00961 L
s
.37711 .00691 m
.37711 .02535 L
s
.40783 .00691 m
.40783 .02535 L
s
.43856 .00691 m
.43856 .02535 L
s
.46928 .00691 m
.46928 .02535 L
s
.49348 .02265 m
.50652 .00961 L
s
.5242 .00961 m
.53724 .02265 L
s
.56144 .00691 m
.56144 .02535 L
s
.59217 .00691 m
.59217 .02535 L
s
.61637 .00961 m
.6294 .02265 L
s
.64439 .01613 m
.66283 .01613 L
s
.68433 .00691 m
.68433 .02535 L
s
.70854 .00961 m
.72157 .02265 L
s
.73926 .00961 m
.75229 .02265 L
s
.7765 .00691 m
.7765 .02535 L
s
.80722 .00691 m
.80722 .02535 L
s
.83142 .00961 m
.84446 .02265 L
s
.86215 .02265 m
.87518 .00961 L
s
.89939 .00691 m
.89939 .02535 L
s
.92359 .02265 m
.93662 .00961 L
s
.95431 .02265 m
.96735 .00961 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0362519 0.97619 0.0669739 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
.5 Mabswid
.06338 .00961 m
.07641 .02265 L
s
.06338 .02265 m
.07641 .00961 L
s
.0914 .01613 m
.10983 .01613 L
s
.10061 .02535 m
.10061 .00691 L
s
.12482 .00961 m
.13785 .02265 L
s
.12482 .02265 m
.13785 .00961 L
s
.15554 .00961 m
.16858 .02265 L
s
.15554 .02265 m
.16858 .00961 L
s
.18626 .00961 m
.1993 .02265 L
s
.18626 .02265 m
.1993 .00961 L
s
.21699 .00961 m
.23002 .02265 L
s
.21699 .02265 m
.23002 .00961 L
s
.24771 .00961 m
.26074 .02265 L
s
.24771 .02265 m
.26074 .00961 L
s
.27573 .01613 m
.29416 .01613 L
s
.28495 .02535 m
.28495 .00691 L
s
.30915 .00961 m
.32219 .02265 L
s
.30915 .02265 m
.32219 .00961 L
s
.33987 .00961 m
.35291 .02265 L
s
.33987 .02265 m
.35291 .00961 L
s
.3679 .01613 m
.38633 .01613 L
s
.37711 .02535 m
.37711 .00691 L
s
.39862 .01613 m
.41705 .01613 L
s
.40783 .02535 m
.40783 .00691 L
s
.42934 .01613 m
.44777 .01613 L
s
.43856 .02535 m
.43856 .00691 L
s
.46006 .01613 m
.47849 .01613 L
s
.46928 .02535 m
.46928 .00691 L
s
.49348 .00961 m
.50652 .02265 L
s
.49348 .02265 m
.50652 .00961 L
s
.5242 .00961 m
.53724 .02265 L
s
.5242 .02265 m
.53724 .00961 L
s
.55223 .01613 m
.57066 .01613 L
s
.56144 .02535 m
.56144 .00691 L
s
.58295 .01613 m
.60138 .01613 L
s
.59217 .02535 m
.59217 .00691 L
s
.61637 .00961 m
.6294 .02265 L
s
.61637 .02265 m
.6294 .00961 L
s
.64439 .01613 m
.66283 .01613 L
s
.65361 .02535 m
.65361 .00691 L
s
.67512 .01613 m
.69355 .01613 L
s
.68433 .02535 m
.68433 .00691 L
s
.70854 .00961 m
.72157 .02265 L
s
.70854 .02265 m
.72157 .00961 L
s
.73926 .00961 m
.75229 .02265 L
s
.73926 .02265 m
.75229 .00961 L
s
.76728 .01613 m
.78571 .01613 L
s
.7765 .02535 m
.7765 .00691 L
s
.798 .01613 m
.81644 .01613 L
s
.80722 .02535 m
.80722 .00691 L
s
.83142 .00961 m
.84446 .02265 L
s
.83142 .02265 m
.84446 .00961 L
s
.86215 .00961 m
.87518 .02265 L
s
.86215 .02265 m
.87518 .00961 L
s
.89017 .01613 m
.9086 .01613 L
s
.89939 .02535 m
.89939 .00691 L
s
.92359 .00961 m
.93662 .02265 L
s
.92359 .02265 m
.93662 .00961 L
s
.95431 .00961 m
.96735 .02265 L
s
.95431 .02265 m
.96735 .00961 L
s
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0700461 0.97619 0.100768 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
[(1)] .06989 .01613 0 0 Mshowa
[(1)] .10061 .01613 0 0 Mshowa
[(1)] .13134 .01613 0 0 Mshowa
[(1)] .16206 .01613 0 0 Mshowa
[(1)] .19278 .01613 0 0 Mshowa
[(0)] .2235 .01613 0 0 Mshowa
[(0)] .25422 .01613 0 0 Mshowa
[(1)] .28495 .01613 0 0 Mshowa
[(0)] .31567 .01613 0 0 Mshowa
[(1)] .34639 .01613 0 0 Mshowa
[(0)] .37711 .01613 0 0 Mshowa
[(0)] .40783 .01613 0 0 Mshowa
[(0)] .43856 .01613 0 0 Mshowa
[(0)] .46928 .01613 0 0 Mshowa
[(1)] .5 .01613 0 0 Mshowa
[(0)] .53072 .01613 0 0 Mshowa
[(0)] .56144 .01613 0 0 Mshowa
[(0)] .59217 .01613 0 0 Mshowa
[(0)] .62289 .01613 0 0 Mshowa
[(1)] .65361 .01613 0 0 Mshowa
[(0)] .68433 .01613 0 0 Mshowa
[(0)] .71505 .01613 0 0 Mshowa
[(0)] .74578 .01613 0 0 Mshowa
[(0)] .7765 .01613 0 0 Mshowa
[(0)] .80722 .01613 0 0 Mshowa
[(0)] .83794 .01613 0 0 Mshowa
[(1)] .86866 .01613 0 0 Mshowa
[(0)] .89939 .01613 0 0 Mshowa
[(1)] .93011 .01613 0 0 Mshowa
[(1)] .96083 .01613 0 0 Mshowa
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{384.188, 39.625},
  ImageMargins->{{34, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40006000009a000`40O003h00OogooPGoo000POooo
F]I;F]HFOol0021oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Moo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo01Aoo`0087oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol00Woo00<007ooOol00goo00=JeWoo
Ool00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol027oo00=JeWooOol01goo00=JeWooOol00goo00<007ooOol00Woo00=JeWoo
Ool00goo00<007ooOol00Woo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol00goo00<007oo
Ool00Woo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol00goo00<007ooOol00Woo00=JeWoo
Ool00goo00<007ooOol00Woo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol057oo000POol00e[FOomoo`05Ool0
1000Oomooe[F2Woo00=JeWooOol01Goo00@007ooOomJePMoo`04001oogooF]H7Ool01000Oomooe[F
0Woo00<007ooOol01Goo00AJeWooOol000Moo`03F]Ioogoo00Qoo`04F]Ioogoo0007Ool00e[FOomo
o`05Ool01000Oomooe[F17oo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWoo
Ool00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol017oo00D007oo
Oomooe[F009oo`03001oogoo00Aoo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03
001oogoo009oo`05F]IoogooOol00006Ool00e[FOomoo`08Ool00e[FOomoo`03Ool00`00Oomoo`02
Ool015[FOomoo`001goo00AJeWooOol000Moo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo
00=oo`03001oogoo009oo`04F]Ioogoo0007Ool00e[FOomoo`05Ool01000Oomooe[F17oo00<007oo
Ool00goo00=JeWooOol01Goo00@007ooOomJePMoo`04001oogooF]HFOol0021oo`03F]Ioogoo00Ao
o`05001oogooOomJeP0:Ool00e[FOomoo`04Ool01@00OomoogooF]H01Woo00D007ooOomooe[F00Io
o`05001oogooOomJeP03Ool00`00Oomoo`04Ool01E[FOomoogoo00001Woo00=JeWooOol027oo00EJ
eWooOomoo`0000Ioo`03F]Ioogoo00Aoo`05001oogooOomJeP04Ool00`00Oomoo`03Ool00e[FOomo
o`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomo
o`03Ool00e[FOomoo`03Ool00`00Oomoo`02Ool01E[FOomoogoo00001Goo00=JeWooOol00goo00<0
07ooOol00Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol027oo00=JeWooOol00goo00<007ooOol00Woo00EJeWooOomoo`0000Ioo`05F]IoogooOol0
0006Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool01E[FOomo
ogoo00001Woo00=JeWooOol017oo00D007ooOomooe[F00Aoo`03001oogoo00=oo`03F]Ioogoo00Ao
o`05001oogooOomJeP06Ool01@00OomoogooF]H05Woo000POol00e[FOomoo`03Ool00`00Oomoo`02
Ool00e[FOomoo`08Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02
Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool00e[FOomoo`08Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`03
Ool00`00Oomoo`02Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02
Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`08
Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02
Ool00e[FOomoo`0DOol0021oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioo`0000L00003OomJ
eWoo00=oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001o
ogoo00=oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Io
o`0000L00003OomJeWoo00Aoo`03001oogoo009oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Io
ogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001o
ogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Io
ogoo009oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001o
ogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00L00003OomJeWoo00Aoo`03001o
ogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Io
ogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001o
ogoo009oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Io
ogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo01Aoo`0087oo
00EJeWooOomoo`0000Ioo`03F]Ioogoo00Qoo`05F]IoogooOol00006Ool01E[FOomoogoo00001Woo
00EJeWooOomoo`0000Ioo`03F]Ioogoo00Aoo`05001oogooOomJeP06Ool01@00OomoogooF]H02Woo
00=JeWooOol017oo00D007ooOomooe[F00=oo`03001oogoo00Aoo`03F]Ioogoo009oo`03001oogoo
00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo
009oo`03001oogoo00=oo`05F]IoogooOol00006Ool00e[FOomoo`03Ool01@00OomoogooF]H01Goo
00<007ooOol00Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol017oo00D007ooOomooe[F
00Yoo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00Aoo`05001oogooOomJeP06Ool01@00
OomoogooF]H01Goo00<007ooOol00Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol017oo
00D007ooOomooe[F00=oo`03001oogoo00Aoo`03F]Ioogoo009oo`03001oogoo00=oo`05F]Ioogoo
Ool00006Ool01E[FOomoogoo00001Woo00=JeWooOol057oo000POol015[FOomoo`001goo00=JeWoo
Ool027oo00AJeWooOol000Moo`04F]Ioogoo0007Ool015[FOomoo`001goo00=JeWooOol01Goo00@0
07ooOomJePMoo`04001oogooF]H:Ool00e[FOomoo`05Ool01000Oomooe[F0Woo00<007ooOol01Goo
00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo
00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00AJeWooOol000Moo`03F]Ioogoo00Ao
o`04001oogooF]H5Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`05
Ool01000Oomooe[F2Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol01Goo00@007ooOomJ
ePMoo`04001oogooF]H5Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomo
o`05Ool01000Oomooe[F0Woo00<007ooOol01Goo00=JeWooOol00Woo00<007ooOol00goo00AJeWoo
Ool000Moo`04F]Ioogoo0007Ool00e[FOomoo`0DOol0021oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03001oogoo00=o
o`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009o
o`03001oogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Moo`03F]Ioogoo00=oo`03001oogoo009o
o`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00=o
o`03001oogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03001oogoo009o
o`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo009o
o`03001oogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo01Aoo`0087oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool01goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool057oo000POoooF]I;F]HFOol00?mooh5oo`0087oooe[FBe[F5Woo000POol00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`07Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`0DOol0
021oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo
00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03F]Ioogoo
00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo
009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo
00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Moo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo
00=oo`03F]Ioogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo
00=oo`03F]Ioogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo
009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo
009oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo01Aoo`0087oo00AJ
eWooOomJePAoo`04F]IoogooF]H5Ool00e[FOomoo`02Ool015[FOomooe[F17oo00AJeWooOomJeP9o
o`03F]Ioogoo009oo`04F]IoogooF]H2Ool00e[FOomoo`02Ool015[FOomooe[F0Woo00=JeWooOol0
0Woo00AJeWooOomJeP9oo`03F]Ioogoo009oo`04F]IoogooF]H4Ool00e[FOomoo`03Ool015[FOomo
oe[F17oo00AJeWooOomJeP9oo`03F]Ioogoo009oo`04F]IoogooF]H4Ool00e[FOomoo`03Ool00e[F
Oomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[F
Oomoo`03Ool015[FOomooe[F0goo00EJeWooOomooe[F009oo`05F]IoogooOomJeP02Ool00e[FOomo
o`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool01E[FOomoogooF]H00goo00AJ
eWooOomJePEoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`04F]IoogooF]H4Ool0
15[FOomooe[F0Woo00=JeWooOol00Woo00AJeWooOomJePEoo`03F]Ioogoo009oo`03F]Ioogoo00=o
o`03F]Ioogoo009oo`04F]IoogooF]H4Ool015[FOomooe[F0Woo00=JeWooOol00Woo00AJeWooOomJ
ePAoo`03F]Ioogoo00=oo`04F]IoogooF]H4Ool015[FOomooe[F0Woo00=JeWooOol00Woo00AJeWoo
OomJeQIoo`0087oo00EJeWooOomooe[F009oo`05F]IoogooOomJeP05Ool00e[FOomoo`02Ool01E[F
OomoogooF]H00Woo00EJeWooOomooe[F00=oo`04F]IoogooF]H3Ool01E[FOomoogooF]H00Woo00EJ
eWooOomooe[F00=oo`04F]IoogooF]H3Ool01E[FOomoogooF]H00Woo00EJeWooOomooe[F00Aoo`03
F]Ioogoo00=oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H00goo00AJeWooOomJeP=oo`03F]Io
ogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Io
ogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`07F]IoogooOomJeWooF]H017oo00MJeWooOomo
oe[FOomJeP03Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool0
0e[FOomoo`02Ool00e[FOomJeP03Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool0
0e[FOomoo`02Ool01E[FOomoogooF]H00Woo00EJeWooOomooe[F00=oo`04F]IoogooF]H3Ool00e[F
Oomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool01E[FOomoogooF]H00Woo
00EJeWooOomooe[F00=oo`04F]IoogooF]H3Ool00e[FOomoo`02Ool00e[FOomoo`03Ool01E[FOomo
ogooF]H00Woo00EJeWooOomooe[F00=oo`04F]IoogooF]H3Ool00e[FOomoo`0DOol0021oo`03F]Io
ogoo009oo`9JePAoo`03F]Iooe[F00MJeP03OomJeWoo00=oo`9JePAoo`03F]Ioogoo009oo`9JePAo
o`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Io
oe[F00MJeP03OomJeWoo00=oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Iooe[F00MJeP03OomJ
eWoo00QJeP03OomJeWoo00MJeP9oo`03F]Iooe[F00IJeP9oo`03F]Ioogoo009oo`03F]Ioogoo00=o
o`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00MJeP04OomJeWooOol7F]H00gooF]Ioo`04
Ool00e[FOomoo`02Ool00e[FOomoo`07F]H00gooF]Ioo`08F]H00gooF]Ioo`03Ool2F]H4Ool00e[F
Oomoo`02Ool2F]H4Ool00e[FOomJeP07F]H00gooF]Ioo`08F]H00gooF]Ioo`03Ool2F]H4Ool00e[F
Oomoo`02Ool2F]H4Ool00e[FOomJeP07F]H00gooF]Ioo`03Ool2F]H4Ool00e[FOomoo`02Ool2F]H4
Ool00e[FOomoo`0DOol0021oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo00=oo`03F]Ioogoo009o
o`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Io
ogoo009oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Io
ogoo009oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Io
ogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Io
ogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Io
ogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Io
ogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Io
ogoo009oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Io
ogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Io
ogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Io
ogoo01Aoo`0087oo00EJeWooOomooe[F009oo`05F]IoogooOomJeP05Ool00e[FOomoo`02Ool01E[F
OomoogooF]H00Woo00EJeWooOomooe[F00=oo`04F]IoogooF]H3Ool01E[FOomoogooF]H00Woo00EJ
eWooOomooe[F00=oo`04F]IoogooF]H3Ool01E[FOomoogooF]H00Woo00EJeWooOomooe[F00Aoo`03
F]Ioogoo00=oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H00goo00AJeWooOomJeP=oo`03F]Io
ogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Io
ogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`07F]IoogooOomJeWooF]H017oo00MJeWooOomo
oe[FOomJeP03Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool0
0e[FOomoo`02Ool00e[FOomJeP03Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool0
0e[FOomoo`02Ool01E[FOomoogooF]H00Woo00EJeWooOomooe[F00=oo`04F]IoogooF]H3Ool00e[F
Oomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool01E[FOomoogooF]H00Woo
00EJeWooOomooe[F00=oo`04F]IoogooF]H3Ool00e[FOomoo`02Ool00e[FOomoo`03Ool01E[FOomo
ogooF]H00Woo00EJeWooOomooe[F00=oo`04F]IoogooF]H3Ool00e[FOomoo`0DOol0021oo`04F]Io
ogooF]H4Ool015[FOomooe[F1Goo00=JeWooOol00Woo00AJeWooOomJePAoo`04F]IoogooF]H2Ool0
0e[FOomoo`02Ool015[FOomooe[F0Woo00=JeWooOol00Woo00AJeWooOomJeP9oo`03F]Ioogoo009o
o`04F]IoogooF]H2Ool00e[FOomoo`02Ool015[FOomooe[F17oo00=JeWooOol00goo00AJeWooOomJ
ePAoo`04F]IoogooF]H2Ool00e[FOomoo`02Ool015[FOomooe[F17oo00=JeWooOol00goo00=JeWoo
Ool00Woo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo00=JeWoo
Ool00goo00AJeWooOomJeP=oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H00Woo00=JeWooOol0
0goo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo00EJeWooOomooe[F00=oo`04F]Io
ogooF]H5Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool015[FOomooe[F17oo00AJ
eWooOomJeP9oo`03F]Ioogoo009oo`04F]IoogooF]H5Ool00e[FOomoo`02Ool00e[FOomoo`03Ool0
0e[FOomoo`02Ool015[FOomooe[F17oo00AJeWooOomJeP9oo`03F]Ioogoo009oo`04F]IoogooF]H4
Ool00e[FOomoo`03Ool015[FOomooe[F17oo00AJeWooOomJeP9oo`03F]Ioogoo009oo`04F]Ioogoo
F]HFOol0021oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Moo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo01Aoo`0087oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol01goo00=JeWooOol027oo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol057oo000POoooF]I;F]HFOol0021ooomJ
eT]JeQIoo`0087oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol01goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol057oo000POol00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`07Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`0DOol0021oo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Moo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo01Aoo`00
87oo00=JeWooOol01@000goo00=JeWooOol01@000goo00=JeWooOol01@000goo00=JeWooOol01@00
0goo00=JeWooOol01@000goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol01@000goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol01@000goo
00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo
00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol01@000goo00=JeWooOol0
0Woo00<007ooOol00Woo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol0
0goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol01@000goo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWoo
Ool00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol01@000goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol01@000goo
00=JeWooOol01@000goo00=JeWooOol057oo000POol00e[FOomoo`02Ool00`00Oomoo`03Ool00e[F
Oomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00
Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]Ioogoo
Ool007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[F
Oomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo
00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001oogoo
00=oo`07F]IoogooOol007oo00000goo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`00
00Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00
Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00
Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo
00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[F
Oomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`0DOol0021oo`03F]Ioogoo009oo`03001oogoo00=o
o`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009o
o`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`07F]IoogooOol007oo000017oo00MJ
eWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Ao
o`03F]Ioogoo009oo`03001oogoo00=oo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool0
0004Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<0
07ooOol00goo00MJeWooOomoo`00Ool00003Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol0
07oo000017oo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomo
ogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomo
ogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool00e[FOomo
o`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001oogoo00=o
o`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo01Aoo`0087oo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWoo
Ool00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004
Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001oogoo00=oo`07F]IoogooOol007oo
000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo
001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000=oo`07F]IoogooOol007oo000017oo00MJeWoo
Oomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001oogoo00=oo`07
F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07
F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`03
F]Ioogoo009oo`03001oogoo00=oo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol057oo000POol00e[FOomoo`02Ool0
0`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool0
0e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001o
o`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomo
o`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]Io
ogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`03F]Io
ogoo009oo`03001oogoo00=oo`07F]IoogooOol007oo00000goo00MJeWooOomoo`00Ool00004Ool0
1e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol0
0goo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo0000
17oo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo0000
17oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool0
0`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`0DOol0021oo`04F]Ioogoo
Ool20005Ool015[FOomoogoo0P001Goo00AJeWooOomoo`8000Eoo`04F]IoogooOol20005Ool015[F
Oomoogoo0P001Goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo
00AJeWooOomoo`8000Eoo`03F]Ioogoo009oo`03001oogoo00=oo`04F]IoogooOol20005Ool00e[F
Oomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00
Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool015[FOomoogoo0P001Goo00=JeWooOol00Woo
00<007ooOol00Woo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo
00=JeWooOol00Woo00<007ooOol00goo00AJeWooOomoo`8000Eoo`03F]Ioogoo009oo`03001oogoo
00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo
009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo
00=oo`04F]IoogooOol20005Ool00e[FOomoo`02Ool00`00Oomoo`03Ool015[FOomoogoo0P001Goo
00AJeWooOomoo`8000Eoo`03F]Ioogoo01Aoo`0087oo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol01goo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol057oo000POoooF]I;F]HFOol0
0?mooh5oo`00\
\>"],
  ImageRangeCache->{{{0, 383.188}, {38.625, 0}} -> {-0.0376658, -0.00258119, 
  0.00280628, 0.00280628}, {{13.375, 369.75}, {37.6875, 
  26.1875}} -> {-1.9968, -0.110786, 0.0913372, 0.0913372}, {{13.375, 
  369.75}, {25.0625, 13.5625}} -> {-1.9968, -1.26392, 0.0913372, 
  0.0913372}, {{13.375, 369.75}, {12.375, 0.875}} -> {-1.9968, -2.42276, 
  0.0913372, 0.0913372}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .10323 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.00245776 0.952381 [
[ 0 0 0 0 ]
[ 1 .10323 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .10323 L
0 .10323 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.00245776 0.97619 0.0331797 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
[(0)] .06989 .01613 0 0 Mshowa
[(1)] .10061 .01613 0 0 Mshowa
[(1)] .13134 .01613 0 0 Mshowa
[(1)] .16206 .01613 0 0 Mshowa
[(1)] .19278 .01613 0 0 Mshowa
[(0)] .2235 .01613 0 0 Mshowa
[(0)] .25422 .01613 0 0 Mshowa
[(1)] .28495 .01613 0 0 Mshowa
[(0)] .31567 .01613 0 0 Mshowa
[(0)] .34639 .01613 0 0 Mshowa
[(0)] .37711 .01613 0 0 Mshowa
[(0)] .40783 .01613 0 0 Mshowa
[(1)] .43856 .01613 0 0 Mshowa
[(0)] .46928 .01613 0 0 Mshowa
[(0)] .5 .01613 0 0 Mshowa
[(0)] .53072 .01613 0 0 Mshowa
[(0)] .56144 .01613 0 0 Mshowa
[(0)] .59217 .01613 0 0 Mshowa
[(1)] .62289 .01613 0 0 Mshowa
[(1)] .65361 .01613 0 0 Mshowa
[(1)] .68433 .01613 0 0 Mshowa
[(0)] .71505 .01613 0 0 Mshowa
[(1)] .74578 .01613 0 0 Mshowa
[(1)] .7765 .01613 0 0 Mshowa
[(0)] .80722 .01613 0 0 Mshowa
[(0)] .83794 .01613 0 0 Mshowa
[(1)] .86866 .01613 0 0 Mshowa
[(0)] .89939 .01613 0 0 Mshowa
[(1)] .93011 .01613 0 0 Mshowa
[(0)] .96083 .01613 0 0 Mshowa
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0362519 0.97619 0.0669739 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
.5 Mabswid
.06068 .01613 m
.07911 .01613 L
s
.06989 .02535 m
.06989 .00691 L
s
.0914 .01613 m
.10983 .01613 L
s
.10061 .02535 m
.10061 .00691 L
s
.12482 .00961 m
.13785 .02265 L
s
.12482 .02265 m
.13785 .00961 L
s
.15284 .01613 m
.17127 .01613 L
s
.16206 .02535 m
.16206 .00691 L
s
.18626 .00961 m
.1993 .02265 L
s
.18626 .02265 m
.1993 .00961 L
s
.21429 .01613 m
.23272 .01613 L
s
.2235 .02535 m
.2235 .00691 L
s
.24501 .01613 m
.26344 .01613 L
s
.25422 .02535 m
.25422 .00691 L
s
.27843 .00961 m
.29146 .02265 L
s
.27843 .02265 m
.29146 .00961 L
s
.30915 .00961 m
.32219 .02265 L
s
.30915 .02265 m
.32219 .00961 L
s
.33717 .01613 m
.35561 .01613 L
s
.34639 .02535 m
.34639 .00691 L
s
.3679 .01613 m
.38633 .01613 L
s
.37711 .02535 m
.37711 .00691 L
s
.39862 .01613 m
.41705 .01613 L
s
.40783 .02535 m
.40783 .00691 L
s
.43204 .00961 m
.44507 .02265 L
s
.43204 .02265 m
.44507 .00961 L
s
.46006 .01613 m
.47849 .01613 L
s
.46928 .02535 m
.46928 .00691 L
s
.49078 .01613 m
.50922 .01613 L
s
.5 .02535 m
.5 .00691 L
s
.5242 .00961 m
.53724 .02265 L
s
.5242 .02265 m
.53724 .00961 L
s
.55223 .01613 m
.57066 .01613 L
s
.56144 .02535 m
.56144 .00691 L
s
.58565 .00961 m
.59868 .02265 L
s
.58565 .02265 m
.59868 .00961 L
s
.61367 .01613 m
.6321 .01613 L
s
.62289 .02535 m
.62289 .00691 L
s
.64709 .00961 m
.66013 .02265 L
s
.64709 .02265 m
.66013 .00961 L
s
.67781 .00961 m
.69085 .02265 L
s
.67781 .02265 m
.69085 .00961 L
s
.70854 .00961 m
.72157 .02265 L
s
.70854 .02265 m
.72157 .00961 L
s
.73656 .01613 m
.75499 .01613 L
s
.74578 .02535 m
.74578 .00691 L
s
.76998 .00961 m
.78301 .02265 L
s
.76998 .02265 m
.78301 .00961 L
s
.8007 .00961 m
.81374 .02265 L
s
.8007 .02265 m
.81374 .00961 L
s
.82873 .01613 m
.84716 .01613 L
s
.83794 .02535 m
.83794 .00691 L
s
.86215 .00961 m
.87518 .02265 L
s
.86215 .02265 m
.87518 .00961 L
s
.89017 .01613 m
.9086 .01613 L
s
.89939 .02535 m
.89939 .00691 L
s
.92359 .00961 m
.93662 .02265 L
s
.92359 .02265 m
.93662 .00961 L
s
.95161 .01613 m
.97005 .01613 L
s
.96083 .02535 m
.96083 .00691 L
s
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0700461 0.97619 0.100768 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
0 g
.06338 .02265 m
.07641 .00961 L
s
.0914 .01613 m
.10983 .01613 L
s
.12482 .02265 m
.13785 .00961 L
s
.15554 .02265 m
.16858 .00961 L
s
.18626 .02265 m
.1993 .00961 L
s
.21699 .00961 m
.23002 .02265 L
s
.24771 .00961 m
.26074 .02265 L
s
.27573 .01613 m
.29416 .01613 L
s
.30915 .00961 m
.32219 .02265 L
s
.33987 .02265 m
.35291 .00961 L
s
.37711 .00691 m
.37711 .02535 L
s
.40783 .00691 m
.40783 .02535 L
s
.43856 .00691 m
.43856 .02535 L
s
.46928 .00691 m
.46928 .02535 L
s
.49348 .02265 m
.50652 .00961 L
s
.5242 .00961 m
.53724 .02265 L
s
.56144 .00691 m
.56144 .02535 L
s
.59217 .00691 m
.59217 .02535 L
s
.61637 .00961 m
.6294 .02265 L
s
.64439 .01613 m
.66283 .01613 L
s
.68433 .00691 m
.68433 .02535 L
s
.70854 .00961 m
.72157 .02265 L
s
.73926 .00961 m
.75229 .02265 L
s
.7765 .00691 m
.7765 .02535 L
s
.80722 .00691 m
.80722 .02535 L
s
.83142 .00961 m
.84446 .02265 L
s
.86215 .02265 m
.87518 .00961 L
s
.89939 .00691 m
.89939 .02535 L
s
.92359 .02265 m
.93662 .00961 L
s
.95431 .02265 m
.96735 .00961 L
s
MathSubEnd
P
% End of sub-graphic
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{384.188, 39.625},
  ImageMargins->{{34, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40006000009a000`40O003h00OogooPGoo000POooo
F]I;F]HFOol0021oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Moo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo01Aoo`0087oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol01goo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol057oo000POol00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`07Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`0DOol0
021oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00D000=oo`03F]Ioogoo00D000=oo`03
F]Ioogoo00D000=oo`03F]Ioogoo00D000=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo
009oo`03001oogoo00=oo`03F]Ioogoo00D000=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Io
ogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001o
ogoo00=oo`03F]Ioogoo00D000=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03
001oogoo00=oo`03F]Ioogoo009oo`03001oogoo009oo`03F]Ioogoo009oo`03001oogoo00=oo`03
F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00D000=oo`03F]Ioogoo00D000=oo`03F]Ioogoo
00D000=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00D000=oo`03F]Ioogoo00D000=o
o`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00D0
00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00D000=oo`03F]Ioogoo009oo`03001o
ogoo00=oo`03F]Ioogoo01Aoo`0087oo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00
Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[F
Oomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo
00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`00
00Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00
Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00
Ool00003Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo
00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo
00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00
Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo
00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[F
Oomoogoo001oo`0000Aoo`03F]Ioogoo01Aoo`0087oo00MJeWooOomoo`00Ool00004Ool00e[FOomo
o`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomo
o`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol0
07oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool01e[FOomo
ogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool00e[FOomo
o`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJ
eWooOomoo`00Ool00003Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=J
eWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<0
07ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomo
o`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=J
eWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomo
o`03Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo01Aoo`0087oo00MJeWooOomoo`00Ool00004
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07
F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004
Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo
000017oo00MJeWooOomoo`00Ool00003Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo
000017oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWoo
Ool00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo
000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo01Aoo`0087oo00MJeWooOomo
o`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool0
0e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001o
o`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomo
o`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomo
o`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]Io
ogooOol007oo000017oo00MJeWooOomoo`00Ool00003Ool01e[FOomoogoo001oo`0000Aoo`07F]Io
ogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol0
0goo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool0
0`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]Io
ogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool0
0e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo01Aoo`0087oo
00=JeWooOol00Woo00<007ooOol00goo00AJeWooOomoo`8000Eoo`04F]IoogooOol20005Ool015[F
Oomoogoo0P001Goo00AJeWooOomoo`8000Eoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo
009oo`03001oogoo00=oo`04F]IoogooOol20005Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[F
Oomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00
Oomoo`03Ool015[FOomoogoo0P001Goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo
00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00Woo00=JeWooOol00Woo00<007ooOol00goo
00=JeWooOol00Woo00<007ooOol00goo00AJeWooOomoo`8000Eoo`04F]IoogooOol20005Ool015[F
Oomoogoo0P001Goo00=JeWooOol00Woo00<007ooOol00goo00AJeWooOomoo`8000Eoo`04F]Ioogoo
Ool20005Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool015[F
Oomoogoo0P001Goo00=JeWooOol00Woo00<007ooOol00goo00AJeWooOomoo`8000Eoo`03F]Ioogoo
009oo`03001oogoo00=oo`03F]Ioogoo01Aoo`0087oo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol01goo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol057oo000POoooF]I;F]HFOol0
0?mooh5oo`0087oooe[FBe[F5Woo000POol00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`07Ool00e[FOomoo`08Ool00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`0DOol0021oo`03F]Ioogoo00=oo`03F]Io
ogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Io
ogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Io
ogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Io
ogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Io
ogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00Moo`03F]Io
ogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Io
ogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Io
ogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Io
ogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Io
ogoo01Aoo`0087oo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo
00AJeWooOomJePAoo`04F]IoogooF]H5Ool00e[FOomoo`02Ool015[FOomooe[F17oo00AJeWooOomJ
ePEoo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`04F]IoogooF]H4Ool015[FOomo
oe[F0Woo00=JeWooOol00Woo00AJeWooOomJePAoo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Io
ogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`04F]IoogooF]H3Ool01E[FOomoogooF]H017oo
00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00goo00AJeWooOomJeP=oo`04F]IoogooF]H5
Ool00e[FOomoo`02Ool01E[FOomoogooF]H00goo00AJeWooOomJePEoo`03F]Ioogoo009oo`04F]Io
ogooF]H4Ool015[FOomooe[F0Woo00=JeWooOol00Woo00AJeWooOomJeP9oo`03F]Ioogoo009oo`04
F]IoogooF]H5Ool00e[FOomoo`02Ool015[FOomooe[F17oo00AJeWooOomJeP9oo`03F]Ioogoo009o
o`04F]IoogooF]H4Ool00e[FOomoo`03Ool015[FOomooe[F17oo00AJeWooOomJePAoo`03F]Ioogoo
00=oo`04F]IoogooF]H4Ool015[FOomooe[F17oo00=JeWooOol00goo00=JeWooOol057oo000POol0
0e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool01E[FOomoogooF]H0
0Woo00EJeWooOomooe[F00Eoo`03F]Ioogoo009oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H0
1Goo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooOol00goo00EJeWooOomooe[F009oo`05F]Io
ogooOomJeP03Ool015[FOomooe[F0goo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo
00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00goo00MJeWooOomooe[FOomJeP04Ool00e[F
Oomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool01e[FOomoogooF]Iooe[F
00=oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Iooe[F00=oo`03F]Ioogoo
00=oo`03F]Ioogoo009oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H00goo00AJeWooOomJeP=o
o`05F]IoogooOomJeP02Ool01E[FOomoogooF]H01Goo00=JeWooOol00Woo00EJeWooOomooe[F009o
o`05F]IoogooOomJeP03Ool015[FOomooe[F0goo00=JeWooOol00Woo00=JeWooOol00goo00EJeWoo
Oomooe[F009oo`05F]IoogooOomJeP04Ool00e[FOomoo`03Ool01E[FOomoogooF]H00Woo00EJeWoo
Oomooe[F00Aoo`03F]Ioogoo00=oo`03F]Ioogoo01Aoo`0087oo00=JeWooOol01e[F00=ooe[FOol0
25[F00=ooe[FOol00goo0U[F17oo00=JeWooF]H01e[F00=ooe[FOol00goo0U[F17oo00=JeWooF]H0
1e[F00=ooe[FOol025[F00=ooe[FOol00goo0U[F17oo00=JeWooOol00Woo0U[F17oo00=JeWooF]H0
1e[F00=ooe[FOol025[F00=ooe[FOol025[F00=ooe[FOol00goo00=JeWooOol00goo00=JeWooF]H0
1U[F0Woo00=JeWooF]H01U[F0Woo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooOol01e[F00=o
oe[FOol017oo00=JeWooOol00Woo00=JeWooOol01e[F00=ooe[FOol00goo0U[F17oo00=JeWooOol0
0Woo0U[F17oo00=JeWooOol00Woo0U[F17oo00=JeWooF]H01e[F00=ooe[FOol00goo0U[F17oo00=J
eWooOol00Woo0U[F17oo00=JeWooF]H01e[F00=ooe[FOol00goo0U[F17oo00=JeWooF]H01e[F00=o
oe[FOol00goo0U[F17oo00=JeWooF]H01e[F00=ooe[FOol05Goo000POol00e[FOomoo`03Ool00e[F
Oomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`02Ool2F]H4Ool00e[FOomoo`03
Ool00e[FOomoo`02Ool00e[FOomoo`02Ool2F]H4Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[F
Oomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool2F]H4Ool00e[FOomoo`02Ool2F]H4Ool00e[F
Oomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[F
Oomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[F
Oomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[F
Oomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[F
Oomoo`02Ool2F]H4Ool00e[FOomoo`02Ool2F]H4Ool00e[FOomoo`02Ool2F]H4Ool00e[FOomoo`03
Ool00e[FOomoo`02Ool00e[FOomoo`02Ool2F]H4Ool00e[FOomoo`02Ool2F]H4Ool00e[FOomoo`02
Ool00e[FOomoo`03Ool00e[FOomoo`02Ool2F]H4Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[F
Oomoo`02Ool2F]H4Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`0DOol0021oo`03F]Io
ogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`05F]IoogooOomJeP02Ool0
1E[FOomoogooF]H01Goo00=JeWooOol00Woo00EJeWooOomooe[F009oo`05F]IoogooOomJeP05Ool0
0e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`03Ool01E[FOomoogooF]H00Woo00EJeWooOomo
oe[F00=oo`04F]IoogooF]H3Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[F
Oomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool01e[FOomoogooF]Iooe[F00Aoo`03F]Ioogoo
009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`07F]IoogooOomJeWooF]H00goo
00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooF]H00goo00=JeWooOol00goo
00=JeWooOol00Woo00EJeWooOomooe[F009oo`05F]IoogooOomJeP03Ool015[FOomooe[F0goo00EJ
eWooOomooe[F009oo`05F]IoogooOomJeP05Ool00e[FOomoo`02Ool01E[FOomoogooF]H00Woo00EJ
eWooOomooe[F00=oo`04F]IoogooF]H3Ool00e[FOomoo`02Ool00e[FOomoo`03Ool01E[FOomoogoo
F]H00Woo00EJeWooOomooe[F00Aoo`03F]Ioogoo00=oo`05F]IoogooOomJeP02Ool01E[FOomoogoo
F]H017oo00=JeWooOol00goo00=JeWooOol057oo000POol00e[FOomoo`03Ool00e[FOomoo`02Ool0
0e[FOomoo`03Ool00e[FOomoo`02Ool015[FOomooe[F17oo00AJeWooOomJePEoo`03F]Ioogoo009o
o`04F]IoogooF]H4Ool015[FOomooe[F1Goo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooOol0
0goo00AJeWooOomJePAoo`04F]IoogooF]H2Ool00e[FOomoo`02Ool015[FOomooe[F17oo00=JeWoo
Ool00goo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00goo00AJeWoo
OomJeP=oo`05F]IoogooOomJeP04Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool0
15[FOomooe[F0goo00AJeWooOomJePEoo`03F]Ioogoo009oo`05F]IoogooOomJeP03Ool015[FOomo
oe[F1Goo00=JeWooOol00Woo00AJeWooOomJePAoo`04F]IoogooF]H2Ool00e[FOomoo`02Ool015[F
Oomooe[F0Woo00=JeWooOol00Woo00AJeWooOomJePEoo`03F]Ioogoo009oo`04F]IoogooF]H4Ool0
15[FOomooe[F0Woo00=JeWooOol00Woo00AJeWooOomJePAoo`03F]Ioogoo00=oo`04F]IoogooF]H4
Ool015[FOomooe[F17oo00=JeWooOol00goo00AJeWooOomJePAoo`04F]IoogooF]H4Ool00e[FOomo
o`03Ool00e[FOomoo`0DOol0021oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Moo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo01Aoo`0087oo00=JeWooOol027oo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol01goo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol0
27oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol057oo000POoooF]I;
F]HFOol0021ooomJeT]JeQIoo`0087oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol01goo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol057oo000POol00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`07Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`0DOol0021oo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03
001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Moo`03
F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00Qoo`03
F]Ioogoo00Qoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03
F]Ioogoo01Aoo`0087oo00=JeWooOol01Goo00@007ooOomJePYoo`03F]Ioogoo00Eoo`04001oogoo
F]H7Ool01000Oomooe[F1goo00@007ooOomJeP9oo`03001oogoo00Eoo`04F]Ioogoo0007Ool00e[F
Oomoo`08Ool015[FOomoo`001goo00=JeWooOol01Goo00@007ooOomJePAoo`03001oogoo00=oo`03
F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03
001oogoo00=oo`03F]Ioogoo00Aoo`05001oogooOomJeP02Ool00`00Oomoo`04Ool00e[FOomoo`03
Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool01E[FOomoogoo00001Woo00=JeWoo
Ool027oo00=JeWooOol00goo00<007ooOol00Woo00AJeWooOol000Moo`04F]Ioogoo0007Ool00e[F
Oomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool015[FOomoo`001goo00=J
eWooOol01Goo00@007ooOomJePAoo`03001oogoo00=oo`03F]Ioogoo00Eoo`04001oogooF]H7Ool0
1000Oomooe[F5Woo000POol00e[FOomoo`04Ool01@00OomoogooF]H02Woo00=JeWooOol017oo00D0
07ooOomooe[F00Ioo`05001oogooOomJeP06Ool01@00OomoogooF]H00goo00<007ooOol017oo00EJ
eWooOomoo`0000Ioo`03F]Ioogoo00Qoo`05F]IoogooOol00006Ool00e[FOomoo`04Ool01@00Oomo
ogooF]H017oo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<0
07ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00goo00<007ooOol00Woo00EJ
eWooOomoo`0000Eoo`03F]Ioogoo00=oo`03001oogoo009oo`03F]Ioogoo00=oo`03001oogoo009o
o`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03001oogoo009o
o`05F]IoogooOol00006Ool01E[FOomoogoo00001Woo00=JeWooOol00goo00<007ooOol00Woo00=J
eWooOol00goo00<007ooOol00Woo00EJeWooOomoo`0000Ioo`03F]Ioogoo00Aoo`05001oogooOomJ
eP04Ool00`00Oomoo`03Ool00e[FOomoo`04Ool01@00OomoogooF]H01Woo00D007ooOomooe[F01Io
o`0087oo00=JeWooOol00Woo0P0017oo00=JeWoo00001`0000=ooe[FOol00goo0P0017oo00=JeWoo
Ool00Woo0P0017oo00=JeWooOol00Woo0P0017oo00=JeWooOol00Woo0P0017oo00=JeWooOol00Woo
0P0017oo00=JeWoo00001`0000=ooe[FOol00goo0P0017oo00=JeWooOol00Woo0P0017oo00=JeWoo
Ool00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWoo
Ool00Woo00<007ooOol00Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00goo00<007oo
Ool00Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol01`0000=ooe[FOol017oo00<007oo
Ool00Woo00=JeWooOol00Woo0P0017oo00=JeWooOol00Woo0P0017oo00=JeWooOol00goo00<007oo
Ool00Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00Woo0P0017oo00=JeWooOol00Woo
0P0017oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo0P0017oo00=JeWooOol00Woo
0P0017oo00=JeWooOol057oo000POol01E[FOomoogoo00001Woo00=JeWooOol027oo00EJeWooOomo
o`0000Ioo`05F]IoogooOol00006Ool01E[FOomoogoo00001Woo00=JeWooOol017oo00D007ooOomo
oe[F00Ioo`05001oogooOomJeP0:Ool00e[FOomoo`04Ool01@00OomoogooF]H00goo00<007ooOol0
17oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol0
0Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00EJeWooOomoo`0000Ioo`03F]Io
ogoo00=oo`05001oogooOomJeP05Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool0
0e[FOomoo`04Ool01@00OomoogooF]H02Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol0
17oo00D007ooOomooe[F00Ioo`05001oogooOomJeP05Ool00`00Oomoo`02Ool00e[FOomoo`03Ool0
0`00Oomoo`02Ool00e[FOomoo`04Ool01@00OomoogooF]H00goo00<007ooOol017oo00=JeWooOol0
0Woo00<007ooOol00goo00EJeWooOomoo`0000Ioo`05F]IoogooOol00006Ool00e[FOomoo`0DOol0
021oo`04F]Ioogoo0007Ool00e[FOomoo`08Ool015[FOomoo`001goo00AJeWooOol000Moo`04F]Io
ogoo0007Ool00e[FOomoo`05Ool01000Oomooe[F1goo00@007ooOomJePYoo`03F]Ioogoo00Eoo`04
001oogooF]H2Ool00`00Oomoo`05Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool0
0`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool0
15[FOomoo`001goo00=JeWooOol017oo00@007ooOomJePEoo`03001oogoo009oo`03F]Ioogoo00=o
o`03001oogoo009oo`03F]Ioogoo00Eoo`04001oogooF]H:Ool00e[FOomoo`03Ool00`00Oomoo`02
Ool00e[FOomoo`05Ool01000Oomooe[F1goo00@007ooOomJePEoo`03001oogoo009oo`03F]Ioogoo
00=oo`03001oogoo009oo`03F]Ioogoo00Eoo`04001oogooF]H2Ool00`00Oomoo`05Ool00e[FOomo
o`02Ool00`00Oomoo`03Ool015[FOomoo`001goo00AJeWooOol000Moo`03F]Ioogoo01Aoo`0087oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo
00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol027oo00=JeWooOol01goo
00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol027oo
00=JeWooOol027oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol027oo00=JeWooOol027oo
00=JeWooOol057oo000POol00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`07Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomo
o`08Ool00e[FOomoo`08Ool00e[FOomoo`0DOol0021ooomJeT]JeQIoo`00ogooPGoo0000\
\>"],
  ImageRangeCache->{{{0, 383.188}, {38.625, 0}} -> {-0.0376658, -0.00258119, 
  0.00280628, 0.00280628}, {{13.375, 369.75}, {37.6875, 
  26.1875}} -> {-1.9968, -0.110786, 0.0913372, 0.0913372}, {{13.375, 
  369.75}, {25.0625, 13.5625}} -> {-1.9968, -1.26392, 0.0913372, 
  0.0913372}, {{13.375, 369.75}, {12.375, 0.875}} -> {-1.9968, -2.42276, 
  0.0913372, 0.0913372}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .10323 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.00245776 0.952381 [
[ 0 0 0 0 ]
[ 1 .10323 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .10323 L
0 .10323 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.00245776 0.97619 0.0331797 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
[(0)] .06989 .01613 0 0 Mshowa
[(1)] .10061 .01613 0 0 Mshowa
[(1)] .13134 .01613 0 0 Mshowa
[(1)] .16206 .01613 0 0 Mshowa
[(1)] .19278 .01613 0 0 Mshowa
[(1)] .2235 .01613 0 0 Mshowa
[(0)] .25422 .01613 0 0 Mshowa
[(1)] .28495 .01613 0 0 Mshowa
[(1)] .31567 .01613 0 0 Mshowa
[(1)] .34639 .01613 0 0 Mshowa
[(0)] .37711 .01613 0 0 Mshowa
[(0)] .40783 .01613 0 0 Mshowa
[(1)] .43856 .01613 0 0 Mshowa
[(0)] .46928 .01613 0 0 Mshowa
[(0)] .5 .01613 0 0 Mshowa
[(0)] .53072 .01613 0 0 Mshowa
[(0)] .56144 .01613 0 0 Mshowa
[(0)] .59217 .01613 0 0 Mshowa
[(1)] .62289 .01613 0 0 Mshowa
[(1)] .65361 .01613 0 0 Mshowa
[(1)] .68433 .01613 0 0 Mshowa
[(1)] .71505 .01613 0 0 Mshowa
[(1)] .74578 .01613 0 0 Mshowa
[(0)] .7765 .01613 0 0 Mshowa
[(0)] .80722 .01613 0 0 Mshowa
[(0)] .83794 .01613 0 0 Mshowa
[(1)] .86866 .01613 0 0 Mshowa
[(0)] .89939 .01613 0 0 Mshowa
[(1)] .93011 .01613 0 0 Mshowa
[(0)] .96083 .01613 0 0 Mshowa
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0362519 0.97619 0.0669739 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
.5 Mabswid
.06068 .01613 m
.07911 .01613 L
s
.06989 .02535 m
.06989 .00691 L
s
.0914 .01613 m
.10983 .01613 L
s
.10061 .02535 m
.10061 .00691 L
s
.12482 .00961 m
.13785 .02265 L
s
.12482 .02265 m
.13785 .00961 L
s
.15284 .01613 m
.17127 .01613 L
s
.16206 .02535 m
.16206 .00691 L
s
.18626 .00961 m
.1993 .02265 L
s
.18626 .02265 m
.1993 .00961 L
s
.21699 .00961 m
.23002 .02265 L
s
.21699 .02265 m
.23002 .00961 L
s
.24501 .01613 m
.26344 .01613 L
s
.25422 .02535 m
.25422 .00691 L
s
.27843 .00961 m
.29146 .02265 L
s
.27843 .02265 m
.29146 .00961 L
s
.30645 .01613 m
.32488 .01613 L
s
.31567 .02535 m
.31567 .00691 L
s
.33987 .00961 m
.35291 .02265 L
s
.33987 .02265 m
.35291 .00961 L
s
.3679 .01613 m
.38633 .01613 L
s
.37711 .02535 m
.37711 .00691 L
s
.39862 .01613 m
.41705 .01613 L
s
.40783 .02535 m
.40783 .00691 L
s
.43204 .00961 m
.44507 .02265 L
s
.43204 .02265 m
.44507 .00961 L
s
.46276 .00961 m
.4758 .02265 L
s
.46276 .02265 m
.4758 .00961 L
s
.49078 .01613 m
.50922 .01613 L
s
.5 .02535 m
.5 .00691 L
s
.5242 .00961 m
.53724 .02265 L
s
.5242 .02265 m
.53724 .00961 L
s
.55493 .00961 m
.56796 .02265 L
s
.55493 .02265 m
.56796 .00961 L
s
.58295 .01613 m
.60138 .01613 L
s
.59217 .02535 m
.59217 .00691 L
s
.61637 .00961 m
.6294 .02265 L
s
.61637 .02265 m
.6294 .00961 L
s
.64439 .01613 m
.66283 .01613 L
s
.65361 .02535 m
.65361 .00691 L
s
.67781 .00961 m
.69085 .02265 L
s
.67781 .02265 m
.69085 .00961 L
s
.70584 .01613 m
.72427 .01613 L
s
.71505 .02535 m
.71505 .00691 L
s
.73656 .01613 m
.75499 .01613 L
s
.74578 .02535 m
.74578 .00691 L
s
.76728 .01613 m
.78571 .01613 L
s
.7765 .02535 m
.7765 .00691 L
s
.8007 .00961 m
.81374 .02265 L
s
.8007 .02265 m
.81374 .00961 L
s
.82873 .01613 m
.84716 .01613 L
s
.83794 .02535 m
.83794 .00691 L
s
.86215 .00961 m
.87518 .02265 L
s
.86215 .02265 m
.87518 .00961 L
s
.89017 .01613 m
.9086 .01613 L
s
.89939 .02535 m
.89939 .00691 L
s
.92359 .00961 m
.93662 .02265 L
s
.92359 .02265 m
.93662 .00961 L
s
.95161 .01613 m
.97005 .01613 L
s
.96083 .02535 m
.96083 .00691 L
s
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0700461 0.97619 0.100768 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
0 g
.06989 .00691 m
.06989 .02535 L
s
.0914 .01613 m
.10983 .01613 L
s
.12482 .02265 m
.13785 .00961 L
s
.15284 .01613 m
.17127 .01613 L
s
.18626 .02265 m
.1993 .00961 L
s
.2235 .00691 m
.2235 .02535 L
s
.25422 .00691 m
.25422 .02535 L
s
.27843 .02265 m
.29146 .00961 L
s
.30915 .00961 m
.32219 .02265 L
s
.34639 .00691 m
.34639 .02535 L
s
.37711 .00691 m
.37711 .02535 L
s
.40783 .00691 m
.40783 .02535 L
s
.43204 .02265 m
.44507 .00961 L
s
.46928 .00691 m
.46928 .02535 L
s
.5 .00691 m
.5 .02535 L
s
.5242 .00961 m
.53724 .02265 L
s
.56144 .00691 m
.56144 .02535 L
s
.58565 .00961 m
.59868 .02265 L
s
.61367 .01613 m
.6321 .01613 L
s
.64709 .02265 m
.66013 .00961 L
s
.67781 .02265 m
.69085 .00961 L
s
.70854 .00961 m
.72157 .02265 L
s
.73656 .01613 m
.75499 .01613 L
s
.76998 .02265 m
.78301 .00961 L
s
.8007 .00961 m
.81374 .02265 L
s
.83794 .00691 m
.83794 .02535 L
s
.86215 .02265 m
.87518 .00961 L
s
.89939 .00691 m
.89939 .02535 L
s
.92359 .02265 m
.93662 .00961 L
s
.96083 .00691 m
.96083 .02535 L
s
MathSubEnd
P
% End of sub-graphic
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{384.188, 39.625},
  ImageMargins->{{34, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40006000009a000`40O003h00OogooPGoo000POooo
F]I;F]HFOol0021oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Moo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo01Aoo`0087oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol01goo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol057oo000POol00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`07Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`0DOol0
021oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00D000=oo`03F]Ioogoo00D000=oo`03
F]Ioogoo00D000=oo`03F]Ioogoo00D000=oo`03F]Ioogoo00D000=oo`03F]Ioogoo009oo`03001o
ogoo00=oo`03F]Ioogoo00D000=oo`03F]Ioogoo00D000=oo`03F]Ioogoo00D000=oo`03F]Ioogoo
009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00D000=oo`03F]Io
ogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001o
ogoo009oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Io
ogoo00D000=oo`03F]Ioogoo00D000=oo`03F]Ioogoo00D000=oo`03F]Ioogoo00D000=oo`03F]Io
ogoo00D000=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03
F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00D000=oo`03F]Ioogoo009oo`03001oogoo00=o
o`03F]Ioogoo00D000=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo01Aoo`0087oo00MJ
eWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomo
o`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomo
o`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001oogoo00=o
o`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`07F]IoogooOol0
07oo000017oo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomo
ogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00003Ool01e[FOomo
ogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<0
07ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool01e[FOomo
ogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJ
eWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Ao
o`03F]Ioogoo01Aoo`0087oo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`03
F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03
001oogoo00=oo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWoo
Oomoo`00Ool00003Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWoo
Ool00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00MJeWoo
Oomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWoo
Ool00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03
Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo01Aoo`0087oo00MJeWooOomoo`00Ool00004Ool0
0e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool0
0`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool0
1e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001o
ogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`07F]IoogooOol007oo000017oo00MJeWooOomo
o`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`07F]Io
ogooOol007oo000017oo00MJeWooOomoo`00Ool00003Ool01e[FOomoogoo001oo`0000Aoo`07F]Io
ogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol0
0goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol0
0Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool01e[FOomoogoo001oo`0000Aoo`07F]Io
ogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool0
0e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo01Aoo`0087oo
00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00
Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[F
Oomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`0000Aoo`03F]Ioogoo009oo`03001oogoo
00=oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001oogoo00=oo`07F]Ioogoo
Ool007oo000017oo00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[F
Oomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00MJeWooOomoo`00Ool00003Ool01e[F
Oomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo
00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo
00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00MJeWooOomoo`00Ool00004Ool01e[F
Oomoogoo001oo`0000Aoo`07F]IoogooOol007oo000017oo00=JeWooOol00Woo00<007ooOol00goo
00MJeWooOomoo`00Ool00004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01e[FOomoogoo001oo`00
00Aoo`03F]Ioogoo01Aoo`0087oo00=JeWooOol00Woo00<007ooOol00goo00AJeWooOomoo`8000Eo
o`04F]IoogooOol20005Ool015[FOomoogoo0P001Goo00AJeWooOomoo`8000Eoo`04F]IoogooOol2
0005Ool00e[FOomoo`02Ool00`00Oomoo`03Ool015[FOomoogoo0P001Goo00AJeWooOomoo`8000Eo
o`04F]IoogooOol20005Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomo
o`03Ool015[FOomoogoo0P001Goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<0
07ooOol00goo00=JeWooOol00Woo00<007ooOol00Woo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol00Woo00<007ooOol00goo00AJeWooOomoo`8000Eoo`04F]IoogooOol20005Ool015[FOomo
ogoo0P001Goo00AJeWooOomoo`8000Eoo`04F]IoogooOol20005Ool00e[FOomoo`02Ool00`00Oomo
o`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool015[FOomo
ogoo0P001Goo00=JeWooOol00Woo00<007ooOol00goo00AJeWooOomoo`8000Eoo`03F]Ioogoo009o
o`03001oogoo00=oo`03F]Ioogoo01Aoo`0087oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol01goo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol057oo000POoooF]I;F]HFOol00?mo
oh5oo`0087oooe[FBe[F5Woo000POol00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`07Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`0DOol0021oo`03F]Ioogoo00=oo`03F]Ioogoo
009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo
009oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo
00Qoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03F]Ioogoo
00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo
009oo`03F]Ioogoo00=oo`03F]Ioogoo00Moo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo
009oo`03F]Ioogoo00Qoo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00Qoo`03F]Ioogoo
00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo
009oo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo
009oo`03F]Ioogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo
01Aoo`0087oo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo00AJ
eWooOomJePAoo`04F]IoogooF]H5Ool00e[FOomoo`02Ool015[FOomooe[F17oo00AJeWooOomJeP9o
o`03F]Ioogoo009oo`04F]IoogooF]H4Ool00e[FOomoo`03Ool015[FOomooe[F17oo00AJeWooOomJ
ePAoo`03F]Ioogoo00=oo`04F]IoogooF]H4Ool015[FOomooe[F17oo00=JeWooOol00goo00=JeWoo
Ool00Woo00=JeWooOol00goo00AJeWooOomJeP=oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H0
0goo00=JeWooOol00Woo00=JeWooOol00goo00AJeWooOomJeP=oo`04F]IoogooF]H3Ool01E[FOomo
ogooF]H00Woo00=JeWooOol00goo00=JeWooOol00Woo00EJeWooOomooe[F00=oo`04F]IoogooF]H5
Ool00e[FOomoo`02Ool015[FOomooe[F17oo00AJeWooOomJePEoo`03F]Ioogoo009oo`03F]Ioogoo
00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`04F]IoogooF]H4Ool015[FOomo
oe[F17oo00=JeWooOol00goo00AJeWooOomJePAoo`04F]IoogooF]H4Ool00e[FOomoo`03Ool015[F
Oomooe[F17oo00AJeWooOomJePAoo`03F]Ioogoo00=oo`03F]Ioogoo01Aoo`0087oo00=JeWooOol0
0goo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo00EJeWooOomooe[F009oo`05F]Io
ogooOomJeP05Ool00e[FOomoo`02Ool01E[FOomoogooF]H00Woo00EJeWooOomooe[F00=oo`04F]Io
ogooF]H3Ool00e[FOomoo`02Ool00e[FOomoo`03Ool01E[FOomoogooF]H00Woo00EJeWooOomooe[F
00Aoo`03F]Ioogoo00=oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H017oo00=JeWooOol00goo
00=JeWooOol00Woo00=JeWooOol00goo00MJeWooOomooe[FOomJeP04Ool01e[FOomoogooF]Iooe[F
00Aoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`07F]IoogooOomJeWooF]H00goo00=JeWooOol00Woo
00=JeWooF]H00goo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooF]H00goo
00=JeWooOol00goo00=JeWooOol00Woo00EJeWooOomooe[F009oo`05F]IoogooOomJeP05Ool00e[F
Oomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool01E[F
OomoogooF]H00Woo00EJeWooOomooe[F00Aoo`03F]Ioogoo00=oo`05F]IoogooOomJeP02Ool01E[F
OomoogooF]H017oo00=JeWooOol00goo00EJeWooOomooe[F009oo`05F]IoogooOomJeP04Ool00e[F
Oomoo`03Ool00e[FOomoo`0DOol0021oo`03F]Ioogoo00MJeP03OomJeWoo00QJeP03OomJeWoo00=o
o`9JePAoo`03F]Iooe[F00MJeP03OomJeWoo00=oo`9JePAoo`03F]Ioogoo009oo`9JePAoo`03F]Io
oe[F00MJeP03OomJeWoo00=oo`9JePAoo`03F]Iooe[F00MJeP03OomJeWoo00=oo`9JePAoo`03F]Io
oe[F00MJeP03OomJeWoo00QJeP03OomJeWoo00=oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Io
ogoo00=oo`03F]Iooe[F00IJeP9oo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03
F]Ioogoo009oo`03F]Ioogoo00MJeP03OomJeWoo00Aoo`03F]Ioogoo009oo`03F]Ioogoo00MJeP03
OomJeWoo00=oo`9JePAoo`03F]Iooe[F00MJeP03OomJeWoo00QJeP03OomJeWoo00QJeP03OomJeWoo
00=oo`9JePAoo`03F]Iooe[F00MJeP03OomJeWoo00=oo`9JePAoo`03F]Iooe[F00MJeP03OomJeWoo
00=oo`9JePAoo`03F]Iooe[F00MJeP03OomJeWoo01Eoo`0087oo00=JeWooOol00goo00=JeWooOol0
0Woo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00Woo0U[F17oo00=JeWooOol00goo00=J
eWooOol00Woo00=JeWooOol00Woo0U[F17oo00=JeWooOol00Woo0U[F17oo00=JeWooOol00Woo00=J
eWooOol00goo00=JeWooOol00Woo0U[F17oo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol0
0Woo0U[F17oo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00goo00=J
eWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo00=J
eWooOol00goo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo00=J
eWooOol00goo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00goo00=J
eWooOol00Woo00=JeWooOol00Woo0U[F17oo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol0
0goo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00Woo0U[F17oo00=J
eWooOol00Woo00=JeWooOol00goo00=JeWooOol00Woo0U[F17oo00=JeWooOol00Woo00=JeWooOol0
0goo00=JeWooOol00Woo0U[F17oo00=JeWooOol00Woo00=JeWooOol00goo00=JeWooOol057oo000P
Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool00e[FOomoo`02Ool01E[FOomoogoo
F]H00Woo00EJeWooOomooe[F00Eoo`03F]Ioogoo009oo`05F]IoogooOomJeP02Ool01E[FOomoogoo
F]H00goo00AJeWooOomJeP=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`05F]IoogooOomJeP02Ool0
1E[FOomoogooF]H017oo00=JeWooOol00goo00EJeWooOomooe[F009oo`05F]IoogooOomJeP04Ool0
0e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`03Ool01e[FOomoogooF]Iooe[F00Aoo`07F]Io
ogooOomJeWooF]H017oo00=JeWooOol00Woo00=JeWooOol00goo00MJeWooOomooe[FOomJeP03Ool0
0e[FOomoo`02Ool00e[FOomJeP03Ool00e[FOomoo`03Ool00e[FOomoo`02Ool00e[FOomoo`02Ool0
0e[FOomJeP03Ool00e[FOomoo`03Ool00e[FOomoo`02Ool01E[FOomoogooF]H00Woo00EJeWooOomo
oe[F00Eoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Io
ogoo009oo`05F]IoogooOomJeP02Ool01E[FOomoogooF]H017oo00=JeWooOol00goo00EJeWooOomo
oe[F009oo`05F]IoogooOomJeP04Ool00e[FOomoo`03Ool01E[FOomoogooF]H00Woo00EJeWooOomo
oe[F00Aoo`03F]Ioogoo00=oo`03F]Ioogoo01Aoo`0087oo00=JeWooOol00goo00=JeWooOol00Woo
00=JeWooOol00goo00=JeWooOol00Woo00AJeWooOomJePAoo`04F]IoogooF]H5Ool00e[FOomoo`02
Ool015[FOomooe[F17oo00AJeWooOomJeP9oo`03F]Ioogoo009oo`04F]IoogooF]H4Ool00e[FOomo
o`03Ool015[FOomooe[F17oo00AJeWooOomJePAoo`03F]Ioogoo00=oo`04F]IoogooF]H4Ool015[F
Oomooe[F17oo00=JeWooOol00goo00=JeWooOol00Woo00=JeWooOol00goo00AJeWooOomJeP=oo`05
F]IoogooOomJeP02Ool01E[FOomoogooF]H00goo00=JeWooOol00Woo00=JeWooOol00goo00AJeWoo
OomJeP=oo`04F]IoogooF]H3Ool01E[FOomoogooF]H00Woo00=JeWooOol00goo00=JeWooOol00Woo
00EJeWooOomooe[F00=oo`04F]IoogooF]H5Ool00e[FOomoo`02Ool015[FOomooe[F17oo00AJeWoo
OomJePEoo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Ioogoo009oo`03F]Ioogoo00=oo`03F]Io
ogoo009oo`04F]IoogooF]H4Ool015[FOomooe[F17oo00=JeWooOol00goo00AJeWooOomJePAoo`04
F]IoogooF]H4Ool00e[FOomoo`03Ool015[FOomooe[F17oo00AJeWooOomJePAoo`03F]Ioogoo00=o
o`03F]Ioogoo01Aoo`0087oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol01goo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol057oo000POol00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`07Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool0
0e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`0DOol0021ooomJeT]JeQIo
o`0087oooe[FBe[F5Woo000POol00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[F
Oomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[F
Oomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[F
Oomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`07Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[F
Oomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[F
Oomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[F
Oomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`0DOol0021oo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Moo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qo
o`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo00Qoo`03F]Ioogoo01Aoo`0087oo00=JeWoo
Ool00goo00<007ooOol00Woo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00Woo00<007ooOol00goo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol027oo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol01goo00=JeWooOol00goo00<007oo
Ool00Woo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWoo
Ool027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol00Woo00<007oo
Ool00goo00=JeWooOol027oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol027oo00=JeWoo
Ool00Woo00<007ooOol00goo00=JeWooOol057oo000POol00e[FOomoo`03Ool00`00Oomoo`02Ool0
0e[FOomoo`08Ool00e[FOomoo`05Ool01000Oomooe[F2Woo00=JeWooOol01Goo00@007ooOomJePEo
o`03001oogoo009oo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00Eoo`04001oogooF]H2
Ool00`00Oomoo`05Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`04Ool01@00OomoogooF]H017oo00<007oo
Ool00goo00=JeWooOol00Woo00<007ooOol00goo00AJeWooOol000Ioo`03F]Ioogoo00=oo`03001o
ogoo009oo`05F]IoogooOol00006Ool00e[FOomoo`08Ool00e[FOomoo`05Ool01000Oomooe[F1goo
00@007ooOomJeP9oo`03001oogoo00Eoo`03F]Ioogoo00Qoo`03F]Ioogoo00Eoo`04001oogooF]H2
Ool00`00Oomoo`05Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`05Ool01000Oomooe[F
17oo00<007ooOol00goo00=JeWooOol01Goo00@007ooOomJePAoo`03001oogoo00=oo`03F]Ioogoo
01Aoo`0087oo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol027oo00=JeWooOol017oo00D0
07ooOomooe[F00Yoo`03F]Ioogoo00Aoo`05001oogooOomJeP05Ool00`00Oomoo`02Ool00e[FOomo
o`02Ool00`00Oomoo`03Ool00e[FOomoo`04Ool01@00OomoogooF]H00goo00<007ooOol017oo00=J
eWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<0
07ooOol00goo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol00Woo00<007ooOol00goo00EJeWooOomoo`0000Eoo`03F]Ioogoo00=oo`03001oogoo009o
o`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00Qoo`03F]Ioogoo00Aoo`05001oogooOomJ
eP06Ool01@00OomoogooF]H00goo00<007ooOol017oo00=JeWooOol027oo00=JeWooOol017oo00D0
07ooOomooe[F00=oo`03001oogoo00Aoo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo00Ao
o`05001oogooOomJeP04Ool00`00Oomoo`03Ool00e[FOomoo`04Ool01@00OomoogooF]H017oo00<0
07ooOol00goo00=JeWooOol057oo000POol00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOol00007
00000gooF]Ioo`03Ool20004Ool00e[FOol0000700000gooF]Ioo`03Ool20004Ool00e[FOomoo`03
Ool00`00Oomoo`02Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool20004Ool00e[F
Oomoo`02Ool20004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03
Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02
Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`02
Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`07
00000gooF]Ioo`03Ool20004Ool00e[FOomoo`02Ool20004Ool00e[FOomoo`02Ool20004Ool00e[F
Ool0000700000gooF]Ioo`03Ool20004Ool00e[FOomoo`02Ool20004Ool00e[FOomoo`02Ool00`00
Oomoo`03Ool00e[FOomoo`02Ool20004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`02
Ool20004Ool00e[FOomoo`02Ool00`00Oomoo`03Ool00e[FOomoo`0DOol0021oo`03F]Ioogoo00=o
o`03001oogoo009oo`03F]Ioogoo00Qoo`05F]IoogooOol00006Ool00e[FOomoo`08Ool01E[FOomo
ogoo00001Woo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol00Woo00<007ooOol00goo00EJ
eWooOomoo`0000Ioo`03F]Ioogoo00Aoo`05001oogooOomJeP04Ool00`00Oomoo`03Ool00e[FOomo
o`02Ool00`00Oomoo`03Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01E[FOomoogoo00001Woo00=J
eWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00goo00D0
07ooOomooe[F00Eoo`03001oogoo009oo`03F]Ioogoo00Aoo`05001oogooOomJeP0:Ool01E[FOomo
ogoo00001Woo00EJeWooOomoo`0000Ioo`03F]Ioogoo00Aoo`05001oogooOomJeP0:Ool01E[FOomo
ogoo00001Woo00=JeWooOol017oo00D007ooOomooe[F00Aoo`03001oogoo00=oo`05F]IoogooOol0
0006Ool00e[FOomoo`02Ool00`00Oomoo`03Ool01E[FOomoogoo00001Woo00=JeWooOol00Woo00<0
07ooOol00goo00=JeWooOol057oo000POol00e[FOomoo`03Ool00`00Oomoo`02Ool00e[FOomoo`08
Ool015[FOomoo`001goo00=JeWooOol027oo00AJeWooOol000Moo`03F]Ioogoo00=oo`03001oogoo
009oo`03F]Ioogoo009oo`03001oogoo00=oo`04F]Ioogoo0007Ool00e[FOomoo`05Ool01000Oomo
oe[F17oo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007oo
Ool00goo00AJeWooOol000Moo`03F]Ioogoo009oo`03001oogoo00=oo`03F]Ioogoo009oo`03001o
ogoo00=oo`03F]Ioogoo00Aoo`04001oogooF]H5Ool00`00Oomoo`02Ool00e[FOomoo`05Ool01000
Oomooe[F2Woo00AJeWooOol000Moo`04F]Ioogoo0007Ool00e[FOomoo`05Ool01000Oomooe[F2Woo
00AJeWooOol000Moo`03F]Ioogoo00Eoo`04001oogooF]H4Ool00`00Oomoo`03Ool015[FOomoo`00
1goo00=JeWooOol00Woo00<007ooOol00goo00AJeWooOol000Moo`03F]Ioogoo009oo`03001oogoo
00=oo`03F]Ioogoo01Aoo`0087oo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol00goo00<007ooOol00Woo00=J
eWooOol00Woo00<007ooOol00goo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol00Woo00<0
07ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol027oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol00Woo00<007ooOol00goo00=J
eWooOol01goo00=JeWooOol00goo00<007ooOol00Woo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=JeWooOol027oo00=J
eWooOol027oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol027oo00=JeWooOol00Woo00<0
07ooOol00goo00=JeWooOol027oo00=JeWooOol00Woo00<007ooOol00goo00=JeWooOol057oo000P
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08
Ool00e[FOomoo`07Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08
Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08Ool00e[FOomoo`08
Ool00e[FOomoo`0DOol0021ooomJeT]JeQIoo`00ogooPGoo0000\
\>"],
  ImageRangeCache->{{{0, 383.188}, {38.625, 0}} -> {-0.0376658, -0.00258119, 
  0.00280628, 0.00280628}, {{13.375, 369.75}, {37.6875, 
  26.1875}} -> {-1.9968, -0.110786, 0.0913372, 0.0913372}, {{13.375, 
  369.75}, {25.0625, 13.5625}} -> {-1.9968, -1.26392, 0.0913372, 
  0.0913372}, {{13.375, 369.75}, {12.375, 0.875}} -> {-1.9968, -2.42276, 
  0.0913372, 0.0913372}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .13871 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.00330261 0.952381 [
[ 0 0 0 0 ]
[ 1 .13871 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .13871 L
0 .13871 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.00330261 0.97619 0.0340246 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
gsave
.06989 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.10061 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .13134 .01613 0 0 Mshowa
gsave
.16206 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .19278 .01613 0 0 Mshowa
[(1)] .2235 .01613 0 0 Mshowa
gsave
.25422 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.28495 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.31567 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .34639 .01613 0 0 Mshowa
[(0)] .37711 .01613 0 0 Mshowa
[(0)] .40783 .01613 0 0 Mshowa
gsave
.43856 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.46928 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.5 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .53072 .01613 0 0 Mshowa
gsave
.56144 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.59217 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.62289 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .65361 .01613 0 0 Mshowa
gsave
.68433 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.71505 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.74578 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .7765 .01613 0 0 Mshowa
gsave
.80722 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.83794 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.86866 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .89939 .01613 0 0 Mshowa
gsave
.93011 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.96083 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0370968 0.97619 0.0678187 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
.5 Mabswid
.12482 .00961 m
.13785 .02265 L
s
.12482 .02265 m
.13785 .00961 L
s
.18626 .00961 m
.1993 .02265 L
s
.18626 .02265 m
.1993 .00961 L
s
.21699 .00961 m
.23002 .02265 L
s
.21699 .02265 m
.23002 .00961 L
s
.33987 .00961 m
.35291 .02265 L
s
.33987 .02265 m
.35291 .00961 L
s
.3679 .01613 m
.38633 .01613 L
s
.37711 .02535 m
.37711 .00691 L
s
.39862 .01613 m
.41705 .01613 L
s
.40783 .02535 m
.40783 .00691 L
s
.5242 .00961 m
.53724 .02265 L
s
.5242 .02265 m
.53724 .00961 L
s
.64439 .01613 m
.66283 .01613 L
s
.65361 .02535 m
.65361 .00691 L
s
.76728 .01613 m
.78571 .01613 L
s
.7765 .02535 m
.7765 .00691 L
s
.89017 .01613 m
.9086 .01613 L
s
.89939 .02535 m
.89939 .00691 L
s
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0708909 0.97619 0.101613 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
.5 Mabswid
.12482 .00961 m
.13785 .02265 L
s
.12482 .02265 m
.13785 .00961 L
s
.18626 .00961 m
.1993 .02265 L
s
.18626 .02265 m
.1993 .00961 L
s
.21699 .00961 m
.23002 .02265 L
s
.21699 .02265 m
.23002 .00961 L
s
.33987 .00961 m
.35291 .02265 L
s
.33987 .02265 m
.35291 .00961 L
s
.3679 .01613 m
.38633 .01613 L
s
.37711 .02535 m
.37711 .00691 L
s
.39862 .01613 m
.41705 .01613 L
s
.40783 .02535 m
.40783 .00691 L
s
.5242 .00961 m
.53724 .02265 L
s
.5242 .02265 m
.53724 .00961 L
s
.64439 .01613 m
.66283 .01613 L
s
.65361 .02535 m
.65361 .00691 L
s
.76728 .01613 m
.78571 .01613 L
s
.7765 .02535 m
.7765 .00691 L
s
.89017 .01613 m
.9086 .01613 L
s
.89939 .02535 m
.89939 .00691 L
s
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.104685 0.97619 0.135407 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
gsave
.06989 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.10061 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .13134 .01613 0 0 Mshowa
gsave
.16206 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .19278 .01613 0 0 Mshowa
[(0)] .2235 .01613 0 0 Mshowa
gsave
.25422 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.28495 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.31567 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .34639 .01613 0 0 Mshowa
[(0)] .37711 .01613 0 0 Mshowa
[(0)] .40783 .01613 0 0 Mshowa
gsave
.43856 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.46928 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.5 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .53072 .01613 0 0 Mshowa
gsave
.56144 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.59217 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.62289 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .65361 .01613 0 0 Mshowa
gsave
.68433 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.71505 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.74578 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .7765 .01613 0 0 Mshowa
gsave
.80722 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.83794 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.86866 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .89939 .01613 0 0 Mshowa
gsave
.93011 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.96083 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{331.438, 45.9375},
  ImageMargins->{{34, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40005;0000;A000`40O003h00OogooC7oo000KOooo
F]HNF]HCOol001]oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Io
o`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Mo
o`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Mo
o`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Io
o`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Mo
o`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Mo
o`03F]Ioogoo00Ioo`03F]Ioogoo015oo`006goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWoo
Ool01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWoo
Ool01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWoo
Ool01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWoo
Ool01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWoo
Ool01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWoo
Ool01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol04Goo000KOol00e[FOomoo`07Ool0
0e[FOomoo`06Ool00e[FOomoo`050002Ool00e[FOomoo`06Ool00e[FOomoo`050002Ool00e[FOol0
00040002Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOol000040002Ool0
1E[FOomoogoo00001Goo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Io
ogoo00Ioo`03F]Ioogoo009oo`03001oogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Io
ogoo00Ioo`03F]Ioogoo00D0009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`05
F]IoogooOol00005Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool01E[FOomoogoo
000017oo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol04Goo000KOol00e[FOomoo`07Ool0
0e[FOomoo`06Ool00e[FOomoo`02Ool00`00Oomoo`02Ool00e[FOomoo`06Ool00e[FOomoo`02Ool0
0`00Oomoo`02Ool01E[FOomoogoo000017oo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol0
1goo00EJeWooOomoo`0000Aoo`06F]Ioogoo001oo`0017oo00IJeWooOol007oo0003Ool00e[FOomo
o`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool01e[FOomoogoo001oo`0000=oo`03F]Ioogoo00Io
o`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001oogoo009oo`03F]Ioogoo00Io
o`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`06F]Ioogoo001oo`0017oo00=JeWooOol01Woo00=JeWoo
Ool01goo00=JeWooOol01Woo00IJeWooOol007oo0003Ool00e[FOomoo`07Ool00e[FOomoo`06Ool0
0e[FOomoo`0AOol001]oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001oogoo
009oo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001oogoo009oo`05F]IoogooOol00004Ool00e[F
Oomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool01E[FOomoogoo000017oo00IJeWooOol007oo
0004Ool01U[FOomoo`00Ool000=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`07
F]IoogooOol007oo00000goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWoo
Ool00Woo00<007ooOol00Woo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00IJeWoo
Ool007oo0004Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool01U[FOomoo`00Ool0
00=oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo015oo`006goo00=JeWooOol01goo00=J
eWooOol01Woo00=JeWooOol00Woo00<007ooOol00Woo00=JeWooOol01Woo00=JeWooOol00Woo00<0
07ooOol00Woo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Mo
o`05F]IoogooOol00004Ool01U[FOomoo`00Ool000Aoo`06F]Ioogoo001oo`000goo00=JeWooOol0
1Woo00=JeWooOol01goo00=JeWooOol01Woo00MJeWooOomoo`00Ool00003Ool00e[FOomoo`06Ool0
0e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`02Ool00`00Oomoo`02Ool00e[FOomoo`06Ool0
0e[FOomoo`07Ool00e[FOomoo`06Ool01U[FOomoo`00Ool000Aoo`03F]Ioogoo00Ioo`03F]Ioogoo
00Moo`03F]Ioogoo00Ioo`06F]Ioogoo001oo`000goo00=JeWooOol01goo00=JeWooOol01Woo00=J
eWooOol04Goo000KOol00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`02Ool00`00Oomoo`02
Ool00e[FOomoo`06Ool00e[FOomoo`02Ool00`00Oomoo`02Ool01E[FOomoogoo000017oo00=JeWoo
Ool01goo00=JeWooOol01Woo00=JeWooOol01goo00EJeWooOomoo`0000Aoo`06F]Ioogoo001oo`00
17oo00IJeWooOol007oo0003Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool01e[F
Oomoogoo001oo`0000=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo
009oo`03001oogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`06F]Ioogoo
001oo`0017oo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00IJeWooOol007oo0003
Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001]oo`03F]Ioogoo00Moo`03F]Io
ogoo00Ioo`04F]IoogooOol20004Ool00e[FOomoo`06Ool015[FOomoogoo0P0017oo00=JeWooOol0
0P0017oo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol00P0017oo00EJ
eWooOomoo`0000Eoo`05F]IoogooOol00004Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool00e[FOomoo`02Ool00`00Oomoo`02Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool015[FOomoogoo0P0017oo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00EJ
eWooOomoo`0000Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`05F]IoogooOol0
0004Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001]ooomJeQiJeQ=oo`00ogoo
C7oo000KOoooF]HNF]HCOol001]oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03
F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03
F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03
F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03
F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03
F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03
F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo015oo`006goo00=JeWooOol01goo00=JeWooOol0
1Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol0
1goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol00Woo00=JeWooOol0
0Woo00EJeWooOomooe[F00Aoo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Io
ogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03F]Io
ogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03F]Io
ogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`05F]Io
ogooOomJeP09Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001]oo`03F]Ioogoo00Moo`03F]Ioogoo
00Ioo`04F]IoogooF]H3Ool015[FOomooe[F27oo00AJeWooOomJeP=oo`04F]IoogooF]H2Ool01e[F
OomoogooF]Iooe[F00Uoo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Iooe[F00=oo`04F]Ioogoo
F]H4Ool00e[FOomoo`02Ool01E[FOomoogooF]H017oo00=JeWooOol01Woo00=JeWooOol01goo00=J
eWooOol01Woo00AJeWooOomJeP=oo`04F]IoogooF]H8Ool00e[FOomoo`07Ool00e[FOomoo`06Ool0
0e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool0
0e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool0
0e[FOomoo`02Ool01E[FOomoogooF]H02Goo00=JeWooOol01Woo00=JeWooOol04Goo000KOol00e[F
Oomoo`07Ool00e[FOomoo`06Ool01e[FOomoogooF]Iooe[F00=oo`03F]Ioogoo00Ioo`07F]Ioogoo
OomJeWooF]H00goo00MJeWooOomooe[FOomJeP02Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[F
Oomoo`07Ool01U[FOomooe[FOomJeP=oo`03F]Ioogoo009oo`03F]Ioogoo009oo`05F]IoogooOomJ
eP04Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool01e[FOomoogooF]Iooe[F00=o
o`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03F]Ioogoo009o
o`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03F]Ioogoo009o
o`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`05F]IoogooOomJ
eP09Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001]oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03
F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03F]Ioogoo009oo`03
F]Ioogoo009oo`05F]IoogooOomJeP09Ool00e[FOomoo`06Ool00e[FOomoo`07Ool01E[FOomoogoo
F]H017oo00=JeWooF]H01U[F00=ooe[FF]H01U[F00=ooe[FOol01goo00=JeWooOol01goo00=JeWoo
Ool01Woo00=JeWooOol00Woo00=JeWooOol00Woo00=JeWooOol01Woo00=JeWooOol01goo00=JeWoo
Ool01Woo00=JeWooF]H01U[F00=ooe[FOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWoo
F]H01U[F00=ooe[FOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooF]H01e[F2Goo00=J
eWooOol01Woo00=JeWooOol04Goo000KOol00e[FOomoo`07Ool00e[FOomoo`06Ool01e[FOomoogoo
F]Iooe[F00=oo`03F]Ioogoo00Ioo`07F]IoogooOomJeWooF]H00goo00MJeWooOomooe[FOomJeP02
Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool01U[FOomooe[FOomJeP=oo`03F]Io
ogoo009oo`03F]Ioogoo009oo`05F]IoogooOomJeP04Ool00e[FOomoo`06Ool00e[FOomoo`07Ool0
0e[FOomoo`06Ool01e[FOomoogooF]Iooe[F00=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Io
ogoo00Ioo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Io
ogoo00Ioo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Io
ogoo00Ioo`03F]Ioogoo009oo`05F]IoogooOomJeP09Ool00e[FOomoo`06Ool00e[FOomoo`0AOol0
01]oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`04F]IoogooF]H3Ool015[FOomooe[F27oo00AJeWoo
OomJeP=oo`04F]IoogooF]H2Ool01e[FOomoogooF]Iooe[F00Uoo`03F]Ioogoo00Ioo`03F]Ioogoo
00Moo`03F]Iooe[F00=oo`04F]IoogooF]H4Ool00e[FOomoo`02Ool01E[FOomoogooF]H017oo00=J
eWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00AJeWooOomJeP=oo`04F]IoogooF]H8Ool0
0e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`06Ool0
0e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`06Ool0
0e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`02Ool01E[FOomoogooF]H02Goo00=JeWooOol0
1Woo00=JeWooOol04Goo000KOol00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[F
Oomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[F
Oomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[F
Oomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[F
Oomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[F
Oomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[F
Oomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001]ooomJeQiJeQ=oo`00ogooC7oo000KOooo
F]HNF]HCOol001]oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Io
o`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Mo
o`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Mo
o`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Io
o`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Mo
o`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Mo
o`03F]Ioogoo00Ioo`03F]Ioogoo015oo`006goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWoo
Ool01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00=JeWoo
Ool01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol00Woo00=JeWooOol00Woo00EJeWoo
Oomooe[F00Aoo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03
F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03F]Ioogoo009oo`03
F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03F]Ioogoo009oo`03
F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`05F]IoogooOomJeP09
Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001]oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`04F]Io
ogooF]H3Ool015[FOomooe[F27oo00AJeWooOomJeP=oo`04F]IoogooF]H2Ool01e[FOomoogooF]Io
oe[F00Uoo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Iooe[F00=oo`04F]IoogooF]H4Ool00e[F
Oomoo`02Ool01E[FOomoogooF]H017oo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo
00AJeWooOomJeP=oo`04F]IoogooF]H8Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`02
Ool00e[FOomoo`02Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`02
Ool00e[FOomoo`02Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`02
Ool01E[FOomoogooF]H02Goo00=JeWooOol01Woo00=JeWooOol04Goo000KOol00e[FOomoo`07Ool0
0e[FOomoo`06Ool01e[FOomoogooF]Iooe[F00=oo`03F]Ioogoo00Ioo`07F]IoogooOomJeWooF]H0
0goo00MJeWooOomooe[FOomJeP02Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool0
1U[FOomooe[FOomJeP=oo`03F]Ioogoo009oo`03F]Ioogoo009oo`05F]IoogooOomJeP04Ool00e[F
Oomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool01e[FOomoogooF]Iooe[F00=oo`03F]Ioogoo
00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo
00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo
00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`05F]IoogooOomJeP09Ool00e[F
Oomoo`06Ool00e[FOomoo`0AOol001]oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009o
o`03F]Ioogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo009o
o`05F]IoogooOomJeP09Ool00e[FOomoo`06Ool00e[FOomoo`07Ool01E[FOomoogooF]H017oo00=J
eWooF]H01U[F00=ooe[FF]H01U[F00=ooe[FOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=J
eWooOol00Woo00=JeWooOol00Woo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00=J
eWooF]H01U[F00=ooe[FOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooF]H01U[F00=o
oe[FOol01goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooF]H01e[F2Goo00=JeWooOol01Woo
00=JeWooOol04Goo000KOol00e[FOomoo`07Ool00e[FOomoo`06Ool01e[FOomoogooF]Iooe[F00=o
o`03F]Ioogoo00Ioo`07F]IoogooOomJeWooF]H00goo00MJeWooOomooe[FOomJeP02Ool00e[FOomo
o`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool01U[FOomooe[FOomJeP=oo`03F]Ioogoo009oo`03
F]Ioogoo009oo`05F]IoogooOomJeP04Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06
Ool01e[FOomoogooF]Iooe[F00=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03
F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03
F]Ioogoo009oo`03F]Ioogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03
F]Ioogoo009oo`05F]IoogooOomJeP09Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001]oo`03F]Io
ogoo00Moo`03F]Ioogoo00Ioo`04F]IoogooF]H3Ool015[FOomooe[F27oo00AJeWooOomJeP=oo`04
F]IoogooF]H2Ool01e[FOomoogooF]Iooe[F00Uoo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Io
oe[F00=oo`04F]IoogooF]H4Ool00e[FOomoo`02Ool01E[FOomoogooF]H017oo00=JeWooOol01Woo
00=JeWooOol01goo00=JeWooOol01Woo00AJeWooOomJeP=oo`04F]IoogooF]H8Ool00e[FOomoo`07
Ool00e[FOomoo`06Ool00e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`06Ool00e[FOomoo`07
Ool00e[FOomoo`06Ool00e[FOomoo`02Ool00e[FOomoo`02Ool00e[FOomoo`06Ool00e[FOomoo`07
Ool00e[FOomoo`06Ool00e[FOomoo`02Ool01E[FOomoogooF]H02Goo00=JeWooOol01Woo00=JeWoo
Ool04Goo000KOol00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool0
0e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool0
0e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`07Ool0
0e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool0
0e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool0
0e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`06Ool00e[FOomoo`07Ool0
0e[FOomoo`06Ool00e[FOomoo`0AOol001]ooomJeQiJeQ=oo`00ogooC7oo000KOoooF]HNF]HCOol0
01]oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo
00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo
00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo
00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo
00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo
00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo
00Ioo`03F]Ioogoo015oo`006goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01@000Woo
00=JeWooOol01Woo00=JeWooOol01@000Woo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Moo`03F]Io
ogoo00Ioo`03F]Ioogoo00Moo`03F]Ioo`0000@0009oo`05F]IoogooOol00005Ool01E[FOomoogoo
000017oo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol00Woo00<007oo
Ool00Woo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01@000Woo00=J
eWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00EJeWooOomoo`0000Eoo`03F]Ioogoo00Io
o`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`05F]IoogooOol00004Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool00e[FOomoo`0AOol001]oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03
001oogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001oogoo009oo`06F]Ioogoo001oo`00
0goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00EJeWooOomoo`0000Aoo`06F]Io
ogoo001oo`0017oo00IJeWooOol007oo0003Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool01e[FOomoogoo001oo`0000=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Io
o`03F]Ioogoo009oo`03001oogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Io
o`06F]Ioogoo001oo`0017oo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00IJeWoo
Ool007oo0003Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001]oo`03F]Ioogoo
00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001oogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo
009oo`03001oogoo009oo`06F]Ioogoo001oo`000goo00=JeWooOol01goo00=JeWooOol01Woo00=J
eWooOol01goo00EJeWooOomoo`0000Aoo`06F]Ioogoo001oo`0017oo00IJeWooOol007oo0003Ool0
0e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool01e[FOomoogoo001oo`0000=oo`03F]Io
ogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001oogoo009oo`03F]Io
ogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`06F]Ioogoo001oo`0017oo00=JeWooOol01Woo
00=JeWooOol01goo00=JeWooOol01Woo00IJeWooOol007oo0003Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool00e[FOomoo`0AOol001]oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03
001oogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001oogoo009oo`06F]Ioogoo001oo`00
0goo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol01goo00EJeWooOomoo`0000Aoo`06F]Io
ogoo001oo`0017oo00IJeWooOol007oo0003Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool01e[FOomoogoo001oo`0000=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Io
o`03F]Ioogoo009oo`03001oogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Io
o`06F]Ioogoo001oo`0017oo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00IJeWoo
Ool007oo0003Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001]oo`03F]Ioogoo
00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001oogoo009oo`03F]Ioogoo00Ioo`03F]Ioogoo
009oo`03001oogoo009oo`06F]Ioogoo001oo`000goo00=JeWooOol01goo00=JeWooOol01Woo00=J
eWooOol01goo00EJeWooOomoo`0000Aoo`06F]Ioogoo001oo`0017oo00IJeWooOol007oo0003Ool0
0e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool01e[FOomoogoo001oo`0000=oo`03F]Io
ogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`03F]Ioogoo009oo`03001oogoo009oo`03F]Io
ogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`06F]Ioogoo001oo`0017oo00=JeWooOol01Woo
00=JeWooOol01goo00=JeWooOol01Woo00IJeWooOol007oo0003Ool00e[FOomoo`07Ool00e[FOomo
o`06Ool00e[FOomoo`0AOol001]oo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`04F]IoogooOol20004
Ool00e[FOomoo`06Ool015[FOomoogoo0P0017oo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Moo`03
F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo008000Aoo`05F]IoogooOol00005Ool01E[FOomo
ogoo000017oo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00=JeWooOol00Woo00<0
07ooOol00Woo00=JeWooOol01Woo00=JeWooOol01goo00=JeWooOol01Woo00AJeWooOomoo`8000Ao
o`03F]Ioogoo00Ioo`03F]Ioogoo00Moo`03F]Ioogoo00Ioo`05F]IoogooOol00005Ool00e[FOomo
o`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool01E[FOomoogoo000017oo00=JeWooOol01goo00=J
eWooOol01Woo00=JeWooOol04Goo000KOol00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07
Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06
Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`06
Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07
Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06
Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`06
Ool00e[FOomoo`07Ool00e[FOomoo`06Ool00e[FOomoo`0AOol001]ooomJeQiJeQ=oo`00ogooC7oo
0000\
\>"],
  ImageRangeCache->{{{0, 330.438}, {44.9375, 0}} -> {-0.0354906, -0.00346847, 
  0.0032411, 0.0032411}, {{10.9375, 319.438}, {43.8125, 
  33.875}} -> {-1.95499, -0.143881, 0.105668, 0.105668}, {{10.9375, 
  319.438}, {32.875, 22.9375}} -> {-1.95499, -1.29962, 0.105668, 0.105668}, {{
  10.9375, 319.438}, {21.9375, 12}} -> {-1.95499, -2.45536, 0.105668, 
  0.105668}, {{10.9375, 319.438}, {11, 1.0625}} -> {-1.95499, -3.6111, 
  0.105668, 0.105668}}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .17419 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.00414747 0.952381 [
[ 0 0 0 0 ]
[ 1 .17419 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .17419 L
0 .17419 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.00414747 0.97619 0.0348694 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
gsave
.06989 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .10061 .01613 0 0 Mshowa
gsave
.13134 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.16206 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.19278 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.2235 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.25422 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.28495 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.31567 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.34639 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.37711 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.40783 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.43856 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.46928 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.5 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.53072 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.56144 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .59217 .01613 0 0 Mshowa
[(1)] .62289 .01613 0 0 Mshowa
gsave
.65361 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.68433 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.71505 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.74578 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.7765 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.80722 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.83794 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .86866 .01613 0 0 Mshowa
gsave
.89939 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .93011 .01613 0 0 Mshowa
gsave
.96083 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0379416 0.97619 0.0686636 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
.5 Mabswid
newpath
.06989 .01613 .01229 0 365.73 arc
s
.07358 .01982 m
.07358 .01982 .00184 0 365.73 arc
F
.06621 .01982 m
.06621 .01982 .00184 0 365.73 arc
F
.06621 .01121 m
.06743 .01367 L
.07235 .01367 L
.07358 .01121 L
s
newpath
.10061 .01613 .01229 0 365.73 arc
s
.1043 .01982 m
.1043 .01982 .00184 0 365.73 arc
F
.09693 .01982 m
.09693 .01982 .00184 0 365.73 arc
F
.09693 .01367 m
.09816 .01121 L
.10307 .01121 L
.1043 .01367 L
s
newpath
.16206 .01613 .01229 0 365.73 arc
s
.16575 .01982 m
.16575 .01982 .00184 0 365.73 arc
F
.15837 .01982 m
.15837 .01982 .00184 0 365.73 arc
F
.15837 .01121 m
.1596 .01367 L
.16452 .01367 L
.16575 .01121 L
s
newpath
.25422 .01613 .01229 0 365.73 arc
s
.25791 .01982 m
.25791 .01982 .00184 0 365.73 arc
F
.25054 .01982 m
.25054 .01982 .00184 0 365.73 arc
F
.25054 .01121 m
.25177 .01367 L
.25668 .01367 L
.25791 .01121 L
s
newpath
.28495 .01613 .01229 0 365.73 arc
s
.28863 .01982 m
.28863 .01982 .00184 0 365.73 arc
F
.28126 .01982 m
.28126 .01982 .00184 0 365.73 arc
F
.28126 .01121 m
.28249 .01367 L
.2874 .01367 L
.28863 .01121 L
s
newpath
.31567 .01613 .01229 0 365.73 arc
s
.31935 .01982 m
.31935 .01982 .00184 0 365.73 arc
F
.31198 .01982 m
.31198 .01982 .00184 0 365.73 arc
F
.31198 .01121 m
.31321 .01367 L
.31813 .01367 L
.31935 .01121 L
s
newpath
.43856 .01613 .01229 0 365.73 arc
s
.44224 .01982 m
.44224 .01982 .00184 0 365.73 arc
F
.43487 .01982 m
.43487 .01982 .00184 0 365.73 arc
F
.43487 .01121 m
.4361 .01367 L
.44101 .01367 L
.44224 .01121 L
s
newpath
.46928 .01613 .01229 0 365.73 arc
s
.47296 .01982 m
.47296 .01982 .00184 0 365.73 arc
F
.46559 .01982 m
.46559 .01982 .00184 0 365.73 arc
F
.46559 .01121 m
.46682 .01367 L
.47174 .01367 L
.47296 .01121 L
s
newpath
.5 .01613 .01229 0 365.73 arc
s
.50369 .01982 m
.50369 .01982 .00184 0 365.73 arc
F
.49631 .01982 m
.49631 .01982 .00184 0 365.73 arc
F
.49631 .01121 m
.49754 .01367 L
.50246 .01367 L
.50369 .01121 L
s
newpath
.56144 .01613 .01229 0 365.73 arc
s
.56513 .01982 m
.56513 .01982 .00184 0 365.73 arc
F
.55776 .01982 m
.55776 .01982 .00184 0 365.73 arc
F
.55776 .01121 m
.55899 .01367 L
.5639 .01367 L
.56513 .01121 L
s
newpath
.59217 .01613 .01229 0 365.73 arc
s
.59585 .01982 m
.59585 .01982 .00184 0 365.73 arc
F
.58848 .01982 m
.58848 .01982 .00184 0 365.73 arc
F
.58848 .01367 m
.58971 .01121 L
.59462 .01121 L
.59585 .01367 L
s
newpath
.62289 .01613 .01229 0 365.73 arc
s
.62657 .01982 m
.62657 .01982 .00184 0 365.73 arc
F
.6192 .01982 m
.6192 .01982 .00184 0 365.73 arc
F
.6192 .01367 m
.62043 .01121 L
.62535 .01121 L
.62657 .01367 L
s
newpath
.68433 .01613 .01229 0 365.73 arc
s
.68802 .01982 m
.68802 .01982 .00184 0 365.73 arc
F
.68065 .01982 m
.68065 .01982 .00184 0 365.73 arc
F
.68065 .01121 m
.68187 .01367 L
.68679 .01367 L
.68802 .01121 L
s
newpath
.71505 .01613 .01229 0 365.73 arc
s
.71874 .01982 m
.71874 .01982 .00184 0 365.73 arc
F
.71137 .01982 m
.71137 .01982 .00184 0 365.73 arc
F
.71137 .01121 m
.7126 .01367 L
.71751 .01367 L
.71874 .01121 L
s
newpath
.74578 .01613 .01229 0 365.73 arc
s
.74946 .01982 m
.74946 .01982 .00184 0 365.73 arc
F
.74209 .01982 m
.74209 .01982 .00184 0 365.73 arc
F
.74209 .01121 m
.74332 .01367 L
.74823 .01367 L
.74946 .01121 L
s
newpath
.80722 .01613 .01229 0 365.73 arc
s
.81091 .01982 m
.81091 .01982 .00184 0 365.73 arc
F
.80353 .01982 m
.80353 .01982 .00184 0 365.73 arc
F
.80353 .01121 m
.80476 .01367 L
.80968 .01367 L
.81091 .01121 L
s
newpath
.83794 .01613 .01229 0 365.73 arc
s
.84163 .01982 m
.84163 .01982 .00184 0 365.73 arc
F
.83425 .01982 m
.83425 .01982 .00184 0 365.73 arc
F
.83425 .01121 m
.83548 .01367 L
.8404 .01367 L
.84163 .01121 L
s
newpath
.86866 .01613 .01229 0 365.73 arc
s
.87235 .01982 m
.87235 .01982 .00184 0 365.73 arc
F
.86498 .01982 m
.86498 .01982 .00184 0 365.73 arc
F
.86498 .01367 m
.86621 .01121 L
.87112 .01121 L
.87235 .01367 L
s
newpath
.93011 .01613 .01229 0 365.73 arc
s
.93379 .01982 m
.93379 .01982 .00184 0 365.73 arc
F
.92642 .01982 m
.92642 .01982 .00184 0 365.73 arc
F
.92642 .01367 m
.92765 .01121 L
.93257 .01121 L
.93379 .01367 L
s
newpath
.96083 .01613 .01229 0 365.73 arc
s
.96452 .01982 m
.96452 .01982 .00184 0 365.73 arc
F
.95714 .01982 m
.95714 .01982 .00184 0 365.73 arc
F
.95714 .01121 m
.95837 .01367 L
.96329 .01367 L
.96452 .01121 L
s
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.0717358 0.97619 0.102458 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
0 g
[(0)] .06989 .01613 0 0 Mshowa
[(1)] .10061 .01613 0 0 Mshowa
gsave
.13134 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .16206 .01613 0 0 Mshowa
gsave
.19278 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.2235 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .25422 .01613 0 0 Mshowa
[(1)] .28495 .01613 0 0 Mshowa
[(1)] .31567 .01613 0 0 Mshowa
gsave
.34639 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.37711 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
gsave
.40783 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .43856 .01613 0 0 Mshowa
[(0)] .46928 .01613 0 0 Mshowa
[(0)] .5 .01613 0 0 Mshowa
gsave
.53072 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .56144 .01613 0 0 Mshowa
[(0)] .59217 .01613 0 0 Mshowa
[(1)] .62289 .01613 0 0 Mshowa
gsave
.65361 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .68433 .01613 0 0 Mshowa
[(1)] .71505 .01613 0 0 Mshowa
[(1)] .74578 .01613 0 0 Mshowa
gsave
.7765 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(0)] .80722 .01613 0 0 Mshowa
[(0)] .83794 .01613 0 0 Mshowa
[(1)] .86866 .01613 0 0 Mshowa
gsave
.89939 .01613 -66 -10.2813 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.562500 moveto
 457.000000 16.562500 lineto
 457.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.562500 lineto
closepath
clip newpath
63.000000 12.812500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
69.000000 12.812500 moveto
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
[(1)] .93011 .01613 0 0 Mshowa
[(0)] .96083 .01613 0 0 Mshowa
.7 g
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.10553 0.97619 0.136252 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
.5 Mabswid
.06068 .01613 m
.07911 .01613 L
s
.06989 .02535 m
.06989 .00691 L
s
.0914 .01613 m
.10983 .01613 L
s
.10061 .02535 m
.10061 .00691 L
s
.15284 .01613 m
.17127 .01613 L
s
.16206 .02535 m
.16206 .00691 L
s
.24501 .01613 m
.26344 .01613 L
s
.25422 .02535 m
.25422 .00691 L
s
.27843 .00961 m
.29146 .02265 L
s
.27843 .02265 m
.29146 .00961 L
s
.30645 .01613 m
.32488 .01613 L
s
.31567 .02535 m
.31567 .00691 L
s
.43204 .00961 m
.44507 .02265 L
s
.43204 .02265 m
.44507 .00961 L
s
.46276 .00961 m
.4758 .02265 L
s
.46276 .02265 m
.4758 .00961 L
s
.49078 .01613 m
.50922 .01613 L
s
.5 .02535 m
.5 .00691 L
s
.55493 .00961 m
.56796 .02265 L
s
.55493 .02265 m
.56796 .00961 L
s
.58295 .01613 m
.60138 .01613 L
s
.59217 .02535 m
.59217 .00691 L
s
.61637 .00961 m
.6294 .02265 L
s
.61637 .02265 m
.6294 .00961 L
s
.67781 .00961 m
.69085 .02265 L
s
.67781 .02265 m
.69085 .00961 L
s
.70584 .01613 m
.72427 .01613 L
s
.71505 .02535 m
.71505 .00691 L
s
.73656 .01613 m
.75499 .01613 L
s
.74578 .02535 m
.74578 .00691 L
s
.8007 .00961 m
.81374 .02265 L
s
.8007 .02265 m
.81374 .00961 L
s
.82873 .01613 m
.84716 .01613 L
s
.83794 .02535 m
.83794 .00691 L
s
.86215 .00961 m
.87518 .02265 L
s
.86215 .02265 m
.87518 .00961 L
s
.92359 .00961 m
.93662 .02265 L
s
.92359 .02265 m
.93662 .00961 L
s
.95161 .01613 m
.97005 .01613 L
s
.96083 .02535 m
.96083 .00691 L
s
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.0238095 0.139324 0.97619 0.170046 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.030722 0.000768049 0.030722 [
[ 0 0 0 0 ]
[ 1 .03226 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .03226 L
0 .03226 L
closepath
clip
newpath
.7 g
.5 Mabswid
.06338 .00961 m
.07641 .02265 L
s
.06338 .02265 m
.07641 .00961 L
s
.0914 .01613 m
.10983 .01613 L
s
.10061 .02535 m
.10061 .00691 L
s
.15554 .00961 m
.16858 .02265 L
s
.15554 .02265 m
.16858 .00961 L
s
.24771 .00961 m
.26074 .02265 L
s
.24771 .02265 m
.26074 .00961 L
s
.27573 .01613 m
.29416 .01613 L
s
.28495 .02535 m
.28495 .00691 L
s
.30915 .00961 m
.32219 .02265 L
s
.30915 .02265 m
.32219 .00961 L
s
.42934 .01613 m
.44777 .01613 L
s
.43856 .02535 m
.43856 .00691 L
s
.46006 .01613 m
.47849 .01613 L
s
.46928 .02535 m
.46928 .00691 L
s
.49348 .00961 m
.50652 .02265 L
s
.49348 .02265 m
.50652 .00961 L
s
.55223 .01613 m
.57066 .01613 L
s
.56144 .02535 m
.56144 .00691 L
s
.58295 .01613 m
.60138 .01613 L
s
.59217 .02535 m
.59217 .00691 L
s
.61637 .00961 m
.6294 .02265 L
s
.61637 .02265 m
.6294 .00961 L
s
.67512 .01613 m
.69355 .01613 L
s
.68433 .02535 m
.68433 .00691 L
s
.70854 .00961 m
.72157 .02265 L
s
.70854 .02265 m
.72157 .00961 L
s
.73926 .00961 m
.75229 .02265 L
s
.73926 .02265 m
.75229 .00961 L
s
.798 .01613 m
.81644 .01613 L
s
.80722 .02535 m
.80722 .00691 L
s
.83142 .00961 m
.84446 .02265 L
s
.83142 .02265 m
.84446 .00961 L
s
.86215 .00961 m
.87518 .02265 L
s
.86215 .02265 m
.87518 .00961 L
s
.92359 .00961 m
.93662 .02265 L
s
.92359 .02265 m
.93662 .00961 L
s
.95431 .00961 m
.96735 .02265 L
s
.95431 .02265 m
.96735 .00961 L
s
1 Mabswid
.05453 .00077 m
.08525 .00077 L
.08525 .03149 L
.05453 .03149 L
.05453 .00077 L
s
.08525 .00077 m
.11598 .00077 L
.11598 .03149 L
.08525 .03149 L
.08525 .00077 L
s
.11598 .00077 m
.1467 .00077 L
.1467 .03149 L
.11598 .03149 L
.11598 .00077 L
s
.1467 .00077 m
.17742 .00077 L
.17742 .03149 L
.1467 .03149 L
.1467 .00077 L
s
.17742 .00077 m
.20814 .00077 L
.20814 .03149 L
.17742 .03149 L
.17742 .00077 L
s
.20814 .00077 m
.23886 .00077 L
.23886 .03149 L
.20814 .03149 L
.20814 .00077 L
s
.23886 .00077 m
.26959 .00077 L
.26959 .03149 L
.23886 .03149 L
.23886 .00077 L
s
.26959 .00077 m
.30031 .00077 L
.30031 .03149 L
.26959 .03149 L
.26959 .00077 L
s
.30031 .00077 m
.33103 .00077 L
.33103 .03149 L
.30031 .03149 L
.30031 .00077 L
s
.33103 .00077 m
.36175 .00077 L
.36175 .03149 L
.33103 .03149 L
.33103 .00077 L
s
.36175 .00077 m
.39247 .00077 L
.39247 .03149 L
.36175 .03149 L
.36175 .00077 L
s
.39247 .00077 m
.4232 .00077 L
.4232 .03149 L
.39247 .03149 L
.39247 .00077 L
s
.4232 .00077 m
.45392 .00077 L
.45392 .03149 L
.4232 .03149 L
.4232 .00077 L
s
.45392 .00077 m
.48464 .00077 L
.48464 .03149 L
.45392 .03149 L
.45392 .00077 L
s
.48464 .00077 m
.51536 .00077 L
.51536 .03149 L
.48464 .03149 L
.48464 .00077 L
s
.51536 .00077 m
.54608 .00077 L
.54608 .03149 L
.51536 .03149 L
.51536 .00077 L
s
.54608 .00077 m
.5768 .00077 L
.5768 .03149 L
.54608 .03149 L
.54608 .00077 L
s
.5768 .00077 m
.60753 .00077 L
.60753 .03149 L
.5768 .03149 L
.5768 .00077 L
s
.60753 .00077 m
.63825 .00077 L
.63825 .03149 L
.60753 .03149 L
.60753 .00077 L
s
.63825 .00077 m
.66897 .00077 L
.66897 .03149 L
.63825 .03149 L
.63825 .00077 L
s
.66897 .00077 m
.69969 .00077 L
.69969 .03149 L
.66897 .03149 L
.66897 .00077 L
s
.69969 .00077 m
.73041 .00077 L
.73041 .03149 L
.69969 .03149 L
.69969 .00077 L
s
.73041 .00077 m
.76114 .00077 L
.76114 .03149 L
.73041 .03149 L
.73041 .00077 L
s
.76114 .00077 m
.79186 .00077 L
.79186 .03149 L
.76114 .03149 L
.76114 .00077 L
s
.79186 .00077 m
.82258 .00077 L
.82258 .03149 L
.79186 .03149 L
.79186 .00077 L
s
.82258 .00077 m
.8533 .00077 L
.8533 .03149 L
.82258 .03149 L
.82258 .00077 L
s
.8533 .00077 m
.88402 .00077 L
.88402 .03149 L
.8533 .03149 L
.8533 .00077 L
s
.88402 .00077 m
.91475 .00077 L
.91475 .03149 L
.88402 .03149 L
.88402 .00077 L
s
.91475 .00077 m
.94547 .00077 L
.94547 .03149 L
.91475 .03149 L
.91475 .00077 L
s
.94547 .00077 m
.97619 .00077 L
.97619 .03149 L
.94547 .03149 L
.94547 .00077 L
s
MathSubEnd
P
% End of sub-graphic
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{295.75, 51.5},
  ImageMargins->{{34, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004W0000<a000`40O003h00Oogoo:7oo000HOooo
F]H@Ool001Qoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03
F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03
F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03
F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03
F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03
F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03
F]Ioogoo00Ioo`03F]Ioogoo00ioo`0067oo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol0
1Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Goo00=JeWooOol0
1Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol0
1Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol0
1Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol0
1Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Goo00=JeWooOol0
1Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol03Woo000HOol00e[FOomoo`06Ool00e[F
000000030002Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool0
0e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool0
0e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool0
0e[FOomoo`05Ool01E[FOomoogoo00000goo00=JeWoo000010000Woo00=JeWooOol01Goo00=JeWoo
Ool01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWoo
Ool01Goo00=JeWoo0000100000=ooe[FOol01goo00=JeWoo0000100000=ooe[FOol01goo00=JeWoo
Ool03Woo000HOol00e[FOomoo`06Ool015[FOomoo`0017oo00=JeWooOol01Woo00=JeWooOol01Goo
00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo
00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo
00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00IJeWooOol007oo0002Ool01E[FOomo
ogoo000017oo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=J
eWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00EJeWooOomoo`0000=oo`03F]Ioogoo00Io
o`05F]IoogooOol00003Ool00e[FOomoo`06Ool00e[FOomoo`0>Ool001Qoo`03F]Ioogoo00Ioo`04
F]Ioogoo0004Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool0
0e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool0
0e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool0
0e[FOomoo`05Ool01U[FOomoo`00Ool0009oo`05F]IoogooOol00004Ool00e[FOomoo`05Ool00e[F
Oomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[F
Oomoo`05Ool01E[FOomoogoo00000goo00=JeWooOol01Woo00EJeWooOomoo`0000=oo`03F]Ioogoo
00Ioo`03F]Ioogoo00ioo`0067oo00=JeWooOol01Woo00AJeWooOol000Aoo`03F]Ioogoo00Ioo`03
F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03
F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03
F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`06F]Ioogoo001oo`00
0Woo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Io
ogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`05F]IoogooOol00003Ool0
0e[FOomoo`06Ool01E[FOomoogoo00000goo00=JeWooOol01Woo00=JeWooOol03Woo000HOol00e[F
Oomoo`06Ool015[FOomoo`0017oo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=J
eWooOol01Goo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=J
eWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=J
eWooOol01Woo00=JeWooOol01Goo00IJeWooOol007oo0002Ool01E[FOomoogoo000017oo00=JeWoo
Ool01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWoo
Ool01Woo00=JeWooOol01Goo00EJeWooOomoo`0000=oo`03F]Ioogoo00Ioo`05F]IoogooOol00003
Ool00e[FOomoo`06Ool00e[FOomoo`0>Ool001Qoo`03F]Ioogoo00Ioo`04F]Ioo`000004Ool00e[F
Oomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`05Ool00e[F
Oomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[F
Oomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool01E[F
Oomoogoo00000goo00=JeWooOol00P0017oo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol0
1Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol0
0P000goo00=JeWooOol01Woo00=JeWooOol00P000goo00=JeWooOol01Woo00=JeWooOol03Woo000H
OoooF]H@Ool001QooomJeQ1oo`0067oo00AJeWooOomoo`80009oo`04F]IoogooOol20003Ool00e[F
Oomoo`05Ool015[FOomoogoo0P000goo00=JeWooOol01Goo00=JeWooOol01Woo00AJeWooOomoo`80
009oo`04F]IoogooOol20003Ool015[FOomoogoo0P000Woo00=JeWooOol01Woo00=JeWooOol01Goo
00=JeWooOol01Woo00EJeWooOomoo`0000=oo`04F]IoogooOol20003Ool01E[FOomoogoo00000goo
00=JeWooOol01Woo00EJeWooOomoo`0000=oo`04F]IoogooOol20003Ool01E[FOomoogoo00000goo
00=JeWooOol01Goo00AJeWooOomoo`8000=oo`04F]IoogooOol20002Ool015[FOomoogoo0P000goo
00=JeWooOol01Goo00AJeWooOomoo`8000=oo`04F]IoogooOol20002Ool015[FOomoogoo0P000goo
00=JeWooOol01Goo00AJeWooOomoo`8000=oo`04F]IoogooOol20002Ool00e[FOomoo`0>Ool001Qo
o`04F]Ioo`000002Ool2000015[FOol000000Woo0P0000=ooe[FOol01Woo00AJeWoo0000009oo`80
0003OomJeWoo00Ioo`03F]Ioogoo00Ioo`04F]Ioo`000002Ool2000015[FOol000000Woo0P0000=o
oe[FOol00P000Woo0P0000=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00EJeWoo0000
07oo00800003OomJeWoo0080009oo`800003OomJeWoo00800005Ool00000OomJeP08Ool01E[FOol0
0000Ool00P0000=ooe[FOol00P000Woo0P0000=ooe[FOol00P0000Eoo`00001ooe[F00Moo`04F]Io
o`000002Ool200000gooF]Ioo`020002Ool2000015[FOol000000Woo0P0000=ooe[FOol01Woo00AJ
eWoo0000009oo`800003OomJeWoo0080009oo`800004F]Ioo`000002Ool200000gooF]Ioo`06Ool0
15[FOol000000Woo0P0000=ooe[FOol00P000Woo0P0000=JeWooOol03Woo000HOol02U[F001oo`00
Ool007oo001JeP000Woo0P0000Aoo`00OomJePMoo`06F]H007ooOol007oo0P0000=ooe[FOol01Woo
00=JeWooOol01Woo01YJeP00Ool007oo001oo`00F]H007oo001oo`00Ool007ooF]H007oo001oo`00
Ool005[F27oo00=JeWooOol01Goo00=JeWooOol01Woo00EJeP00Ool007oo0080000?OomJeP00Ool0
07oo001oo`00OomJeP00Ool007oo00800003OomJeWoo00Moo`05F]H007oo001oo`0200000gooF]H0
0002Ool200001Woo001ooe[F001oo`800004Ool007ooF]H7Ool01U[F001oogoo001oo`80000;OomJ
eP00Ool007oo001oo`00F]H00002Ool01P00Ool00000OomJePMoo`06F]H007ooOol007oo0P0000]o
oe[F001oo`00Ool007oo001JeP00009oo`800004Ool007ooF]H7Ool015[F001oogoo0P0000aoo`00
OomJeP00Ool007oo001oo`00F]H@Ool001Qoo`03F]H007oo00<00004Ool005[F0002Ool01P00Ool0
07oo001JePMoo`04F]H007ooOol20002Ool00`00F]Ioo`06Ool00e[FOomoo`06Ool00e[F001oo`03
000017oo001JeP000Woo0P000Woo00@005[F001oo`<00003Ool005[F00Qoo`03F]Ioogoo00Eoo`03
F]Ioogoo00Ioo`03F]Ioogoo00<00004Ool005[F0002Ool20002Ool01000F]Ioogoo0`0000=oo`00
F]H027oo00=JeWooOol00P000Woo00L005[F001oo`00Ool00002Ool02P00F]Ioogoo001oo`00Ool0
05[F1goo00AJeP00Oomoo`80009oo`03001JeP00009oo`80009oo`04F]H007ooOol20002Ool00`00
F]Ioo`06Ool015[F001oogoo0P000Woo00@005[F001oo`<0009oo`0:F]H007ooOol007oo001oo`00
F]H7Ool035[F001oogoo001oo`00Ool005[F001oo`<00003Ool005[F011oo`0067oo00=JeP00Ool0
1@0000=JeP00Ool0100000=oo`00F]H01goo00=JeP00Ool0100000=oo`00F]H01goo00=JeWooOol0
1Woo00EJeP00000007oo00<00003F]H007oo00@00003Ool005[F00D00003Ool005[F00Qoo`03F]Io
ogoo00Eoo`03F]Ioogoo00Ioo`03F]Ioo`0000<00005Ool005[F001oo`04000017oo001JeWoo1000
00=oo`00F]H027oo00=JeWoo00000`0000Eoo`00F]H007oo00@00004Ool005[FOol400000goo001J
eP07Ool00e[F001oo`0400001Goo001JeP00Ool0100000Aooe[F001oo`@00003Ool005[F00Moo`03
F]H007oo00@00005Ool005[F001oo`04000017ooF]H007oo100000=oo`00F]H01goo00=JeP00Ool0
100000Eoo`00F]H007oo00D00003F]Ioogoo00ioo`0067oo00=JeWoo000017oo00@005[FOol000Ao
o`03001ooe[F00Moo`03F]Ioo`0000Aoo`03001ooe[F00Moo`03F]Ioogoo00Ioo`03F]Ioo`0000=o
o`05001ooe[FOol00004Ool01000OomJeP0017oo00<007ooF]H027oo00=JeWooOol01Goo00=JeWoo
Ool01Woo00=JeP00Ool00goo00D007ooF]Ioo`0000Aoo`04001ooe[F0004Ool00`00OomJeP08Ool0
0e[F001oo`03Ool01@00OomJeWoo000017oo00@007ooF]H000Aoo`03001ooe[F00Moo`03F]Ioo`00
00Aoo`05001ooe[FOol00004Ool01000F]Ioo`0017oo00<007ooF]H01goo00=JeWoo000017oo00D0
07ooF]Ioo`0000Aoo`04001JeWoo0004Ool00`00OomJeP07Ool00e[FOol00004Ool01@00OomJeWoo
00000goo00<007ooF]H047oo000HOol00e[FOol00003Ool01000OomJeWoo0P000goo00<007ooF]H0
1goo00=JeWoo00000goo0P0000=ooe[FOol01Woo00=JeWooOol01Woo00=JeWoo00000goo00D007oo
F]Ioo`0000=oo`800004OomJeP000003Ool00`00OomJeP08Ool00e[FOomoo`05Ool00e[FOomoo`06
Ool00e[F00000003Ool01@00OomJeWoo00000goo0P0000Aooe[F000000=oo`03001ooe[F00Qoo`03
F]H0000000=oo`05001ooe[FOol00003Ool2000017ooF]H000000goo00<007ooF]H01goo00AJeWoo
000000=oo`05001ooe[FOol00003Ool2000015[FOol000000goo00<007ooF]H01goo00AJeWoo0000
00=oo`05001ooe[FOol00003Ool2000015[FOol000000goo00<007ooF]H01goo00=JeWoo00000goo
0P0000Aooe[FOol000=oo`03001ooe[F011oo`0067oo00=JeWooOol00`000Woo00AJeWooOomoo`<0
009oo`03F]Ioogoo00Eoo`03F]Ioogoo00<000=oo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`03F]Io
ogoo00<0009oo`03F]Ioogoo00<000=oo`03F]Ioogoo00<0009oo`03F]Ioogoo00Ioo`03F]Ioogoo
00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00<0009oo`03F]Ioogoo00<000=oo`03F]Ioogoo00<0009o
o`03F]Ioogoo00Ioo`03F]Ioogoo00<0009oo`03F]Ioogoo00<000=oo`03F]Ioogoo00<0009oo`03
F]Ioogoo00Eoo`04F]IoogooOol30002Ool00e[FOomoo`030002Ool015[FOomoogoo0`000Woo00=J
eWooOol01Goo00AJeWooOomoo`<0009oo`03F]Ioogoo00<0009oo`04F]IoogooOol30002Ool00e[F
Oomoo`05Ool00e[FOomoo`030003Ool00e[FOomoo`030002Ool00e[FOomoo`0>Ool001QooomJeQ1o
o`0067oooe[F47oo000HOol00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomo
o`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomo
o`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomo
o`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomo
o`05Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomo
o`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomo
o`06Ool00e[FOomoo`05Ool00e[FOomoo`0>Ool001Qoo`05F]IoogooOol00003Ool00e[FOol00004
0002Ool00e[FOomoo`05Ool00e[FOol000040002Ool00e[FOomoo`05Ool00e[FOomoo`06Ool015[F
Oomoo`0017oo00=JeWoo000010000Woo00=JeP0000000`000Woo00=JeWooOol01Woo00=JeWooOol0
1Goo00=JeWooOol01Woo00=JeP0000000`000Woo00EJeWooOomoo`0000Aoo`04F]Ioogoo0004Ool0
0e[FOomoo`06Ool015[FOomoo`0017oo00EJeWooOomoo`0000Aoo`03F]H0000000<0009oo`03F]Io
ogoo00Eoo`03F]Ioo`0000@0009oo`03F]Ioo`0000@00003OomJeWoo00D0009oo`03F]Ioogoo00Eo
o`05F]IoogooOol00004Ool01E[FOomoogoo00000goo00=JeWoo000010000Woo00=JeWooOol01Goo
00=JeWoo000010000Woo00EJeWooOomoo`0000=oo`03F]Ioogoo00ioo`0067oo00IJeWooOol007oo
0002Ool01E[FOomoogoo000017oo00=JeWooOol01Goo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Eo
o`03F]Ioogoo00Ioo`05F]Ioo`00Ool00003Ool01E[FOomoogoo000017oo00AJeWooOol000Aoo`03
F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`04F]Ioogoo0004Ool01U[FOomoo`00Ool0
00=oo`05F]Ioo`00Ool00003Ool00e[FOomoo`06Ool01E[FOol007oo00000goo00IJeWooOol007oo
0003Ool015[FOomoo`0017oo00=JeWooOol01Goo00EJeWooOomoo`0000Aoo`05F]IoogooOol00003
Ool01E[FOomoogoo000017oo00=JeWooOol01Goo00IJeWooOol007oo0003Ool01U[FOomoo`00Ool0
009oo`05F]IoogooOol00004Ool00e[FOomoo`05Ool01E[FOomoogoo000017oo00IJeWooOol007oo
0002Ool00e[FOomoo`0>Ool001Qoo`06F]Ioogoo001oo`000Woo00EJeWooOomoo`0000Aoo`03F]Io
ogoo00Eoo`05F]IoogooOol00004Ool00e[FOomoo`05Ool00e[FOomoo`06Ool01E[FOol007oo0000
0goo00EJeWooOomoo`0000Aoo`04F]Ioogoo0004Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[F
Oomoo`06Ool015[FOomoo`0017oo00IJeWooOol007oo0003Ool01E[FOol007oo00000goo00=JeWoo
Ool01Woo00EJeWoo001oo`0000=oo`06F]Ioogoo001oo`000goo00AJeWooOol000Aoo`03F]Ioogoo
00Eoo`05F]IoogooOol00004Ool01E[FOomoogoo00000goo00EJeWooOomoo`0000Aoo`03F]Ioogoo
00Eoo`06F]Ioogoo001oo`000goo00IJeWooOol007oo0002Ool01E[FOomoogoo000017oo00=JeWoo
Ool01Goo00EJeWooOomoo`0000Aoo`06F]Ioogoo001oo`000Woo00=JeWooOol03Woo000HOol01U[F
Oomoo`00Ool0009oo`05F]IoogooOol00004Ool00e[FOomoo`05Ool01E[FOomoogoo000017oo00=J
eWooOol01Goo00=JeWooOol01Woo00EJeWoo001oo`0000=oo`05F]IoogooOol00004Ool015[FOomo
o`0017oo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00AJeWooOol000Aoo`06F]Io
ogoo001oo`000goo00EJeWoo001oo`0000=oo`03F]Ioogoo00Ioo`05F]Ioo`00Ool00003Ool01U[F
Oomoo`00Ool000=oo`04F]Ioogoo0004Ool00e[FOomoo`05Ool01E[FOomoogoo000017oo00EJeWoo
Oomoo`0000=oo`05F]IoogooOol00004Ool00e[FOomoo`05Ool01U[FOomoo`00Ool000=oo`06F]Io
ogoo001oo`000Woo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Eoo`05F]IoogooOol00004Ool01U[F
Oomoo`00Ool0009oo`03F]Ioogoo00ioo`0067oo00IJeWooOol007oo0002Ool01E[FOomoogoo0000
17oo00=JeWooOol01Goo00EJeWooOomoo`0000Aoo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`05F]Io
o`00Ool00003Ool01E[FOomoogoo000017oo00AJeWooOol000Aoo`03F]Ioogoo00Ioo`03F]Ioogoo
00Eoo`03F]Ioogoo00Ioo`04F]Ioogoo0004Ool01U[FOomoo`00Ool000=oo`05F]Ioo`00Ool00003
Ool00e[FOomoo`06Ool01E[FOol007oo00000goo00IJeWooOol007oo0003Ool015[FOomoo`0017oo
00=JeWooOol01Goo00EJeWooOomoo`0000Aoo`05F]IoogooOol00003Ool01E[FOomoogoo000017oo
00=JeWooOol01Goo00IJeWooOol007oo0003Ool01U[FOomoo`00Ool0009oo`05F]IoogooOol00004
Ool00e[FOomoo`05Ool01E[FOomoogoo000017oo00IJeWooOol007oo0002Ool00e[FOomoo`0>Ool0
01Qoo`05F]IoogooOol00003Ool00e[FOomoo`020004Ool00e[FOomoo`05Ool00e[FOomoo`020004
Ool00e[FOomoo`05Ool00e[FOomoo`06Ool015[FOomoo`0017oo00=JeWooOol00P0017oo00AJeWoo
000000Aoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`04F]Ioo`000004Ool01E[F
Oomoogoo000017oo00AJeWooOol000Aoo`03F]Ioogoo00Ioo`04F]Ioogoo0004Ool01E[FOomoogoo
000017oo00AJeWoo000000Aoo`03F]Ioogoo00Eoo`03F]Ioogoo008000Aoo`03F]Ioogoo008000=o
o`03F]Ioogoo008000Aoo`03F]Ioogoo00Eoo`05F]IoogooOol00004Ool01E[FOomoogoo00000goo
00=JeWooOol00P0017oo00=JeWooOol01Goo00=JeWooOol00P0017oo00EJeWooOomoo`0000=oo`03
F]Ioogoo00ioo`0067oo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol0
1Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol0
1Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol0
1Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol0
1Goo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol0
1Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol0
1Woo00=JeWooOol01Goo00=JeWooOol03Woo000HOoooF]H@Ool001QooomJeQ1oo`0067oo00=JeWoo
Ool01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWoo
Ool01Goo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWoo
Ool01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWoo
Ool01Woo00=JeWooOol01Goo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWoo
Ool01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWoo
Ool01Goo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWoo
Ool03Woo000HOol00e[FOomoo`02Ool01E[FOomoogooF]H00goo00EJeWooOomooe[F00Qoo`05F]Io
ogooOomJeP03Ool00e[FOomoo`06Ool00e[FOomoo`05Ool01E[FOomoogooF]H00goo00AJeWooOomJ
eP9oo`04F]IoogooF]H3Ool01E[FOomoogooF]H027oo00=JeWooOol01Goo00=JeWooOol01Woo00=J
eWooF]H00goo00=JeWooF]H00Woo00AJeWooOomJeP9oo`05F]IoogooOomJeP03Ool00e[FOomoo`06
Ool00e[FOomJeP03Ool00e[FOomJeP03Ool01E[FOomoogooF]H00Woo00AJeWooOomJeP9oo`03F]Io
ogoo00Eoo`04F]IoogooF]H2Ool015[FOomooe[F0goo00EJeWooOomooe[F00=oo`03F]Ioogoo009o
o`03F]Ioogoo00Eoo`04F]IoogooF]H2Ool015[FOomooe[F0goo00MJeWooOomooe[FOomJeP03Ool0
0e[FOomJeP08Ool00e[FOomJeP03Ool00e[FOomJeP04Ool01E[FOomoogooF]H047oo000HOol00e[F
Oomoo`02Ool01E[FOomoogooF]H00goo00EJeWooOomooe[F00Qoo`05F]IoogooOomJeP03Ool00e[F
Oomoo`06Ool00e[FOomoo`05Ool01E[FOomoogooF]H00goo00AJeWooOomoo`9JeP=oo`05F]Ioogoo
OomJeP03Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool01U[FOomooe[FOomJeP9o
o`04F]IoogooOol2F]H3Ool01E[FOomoogooF]H00goo00=JeWooOol01Woo00IJeWooOomJeWooF]H2
Ool01E[FOomoogooF]H00goo00AJeWooOomoo`9JeP=oo`03F]Ioogoo00Eoo`04F]IoogooOol2F]H3
Ool01E[FOomoogooF]H00goo00EJeWooOomooe[F00Aoo`03F]Ioogoo00Eoo`04F]IoogooOol2F]H3
Ool01E[FOomoogooF]H00goo00IJeWooOomJeWooF]H2Ool00e[FOomoo`06Ool01U[FOomooe[FOomJ
eP9oo`03F]Ioogoo009oo`05F]IoogooOomJeP0@Ool001Qoo`03F]Iooe[F00EJeP03OomJeU[F00MJ
ePQoo`MJeP03OomJeWoo00Moo`03F]Ioogoo00Eoo`03F]Iooe[F00IJeP=oo`03F]Ioogoo009oo`03
F]Iooe[F00IJePQoo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`05F]IoogooOomJeP03Ool01E[FOomo
ogooF]H017oo1e[F00=ooe[FOol01goo00EJeWooOomooe[F00=oo`03F]Iooe[F00IJeP=oo`03F]Io
ogoo009oo`03F]Ioogoo00Eoo`05F]IoogooOomJeP04Ool9F]H00gooF]IJeP04F]H00gooF]Ioo`06
Ool01E[FOomoogooF]H017oo1e[F00=ooe[FOol00Woo00EJeWooOomooe[F00Qoo`05F]IoogooOomJ
eP03Ool00e[FOomJeP05F]H00gooF]Ioo`0?Ool001Qoo`03F]Ioogoo009oo`05F]IoogooOomJeP03
Ool01E[FOomoogooF]H027oo00EJeWooOomooe[F00=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`05
F]IoogooOomJeP03Ool015[FOomoogoo0U[F0goo00EJeWooOomooe[F00=oo`03F]Ioogoo00Ioo`03
F]Ioogoo00Eoo`03F]Ioogoo00Ioo`06F]IoogooF]Iooe[F0Woo00AJeWooOomoo`9JeP=oo`05F]Io
ogooOomJeP03Ool00e[FOomoo`06Ool01U[FOomooe[FOomJeP9oo`05F]IoogooOomJeP03Ool015[F
Oomoogoo0U[F0goo00=JeWooOol01Goo00AJeWooOomoo`9JeP=oo`05F]IoogooOomJeP03Ool01E[F
OomoogooF]H017oo00=JeWooOol01Goo00AJeWooOomoo`9JeP=oo`05F]IoogooOomJeP03Ool01U[F
Oomooe[FOomJeP9oo`03F]Ioogoo00Ioo`06F]IoogooF]Iooe[F0Woo00=JeWooOol00Woo00EJeWoo
Oomooe[F011oo`0067oo00=JeWooOol00Woo00EJeWooOomooe[F00=oo`05F]IoogooOomJeP08Ool0
1E[FOomoogooF]H00goo00=JeWooOol01Woo00=JeWooOol01Goo00EJeWooOomooe[F00=oo`04F]Io
ogooF]H2Ool015[FOomooe[F0goo00EJeWooOomooe[F00Qoo`03F]Ioogoo00Eoo`03F]Ioogoo00Io
o`03F]Iooe[F00=oo`03F]Iooe[F009oo`04F]IoogooF]H2Ool01E[FOomoogooF]H00goo00=JeWoo
Ool01Woo00=JeWooF]H00goo00=JeWooF]H00goo00EJeWooOomooe[F009oo`04F]IoogooF]H2Ool0
0e[FOomoo`05Ool015[FOomooe[F0Woo00AJeWooOomJeP=oo`05F]IoogooOomJeP03Ool00e[FOomo
o`02Ool00e[FOomoo`05Ool015[FOomooe[F0Woo00AJeWooOomJeP=oo`07F]IoogooOomJeWooF]H0
0goo00=JeWooF]H027oo00=JeWooF]H00goo00=JeWooF]H017oo00EJeWooOomooe[F011oo`0067oo
00=JeWooOol00Woo00EJeWooOomooe[F00=oo`05F]IoogooOomJeP08Ool01E[FOomoogooF]H00goo
00=JeWooOol01Woo00=JeWooOol01Goo00EJeWooOomooe[F00=oo`03F]Ioogoo00Ioo`05F]Ioogoo
OomJeP03Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[F
Oomoo`06Ool01E[FOomoogooF]H00goo00=JeWooOol01Woo00=JeWooOol01Goo00EJeWooOomooe[F
00=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`05F]IoogooOomJeP03Ool01E[F
OomoogooF]H017oo00=JeWooOol01Goo00=JeWooOol01Woo00EJeWooOomooe[F00=oo`03F]Ioogoo
00Eoo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo009oo`05F]IoogooOomJeP0@Ool001Qo
oomJeQ1oo`00ogoo:7oo000HOoooF]H@Ool001Qoo`03F]Ioogoo00Ioo`05F]IoogooOomJeP03Ool0
0e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`05Ool0
0e[FOomoo`02Ool01E[FOomoogooF]H01goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol0
1Woo00EJeWooOomooe[F00=oo`05F]IoogooOomJeP04Ool00e[FOomoo`05Ool00e[FOomoo`06Ool0
1E[FOomoogooF]H00goo00EJeWooOomooe[F00=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Io
ogoo009oo`05F]IoogooOomJeP07Ool00e[FOomoo`06Ool00e[FOomoo`05Ool01E[FOomoogooF]H0
17oo00=JeWooOol01Goo00=JeWooOol01Goo00=JeWooOol01Woo00=JeWooOol01Goo00=JeWooOol0
1Woo00=JeWooOol03Woo000HOol015[FOomooe[F0Woo00AJeWooOomJeP=oo`05F]IoogooOomJeP08
Ool00e[FOomJeP03Ool00e[FOomJeP08Ool00e[FOomoo`05Ool00e[FOomJeP03Ool00e[FOomJeP04
Ool01e[FOomoogooF]Iooe[F00=oo`03F]Iooe[F00Qoo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`05
F]IoogooOomJeP03Ool01E[FOomoogooF]H017oo00=JeWooF]H00goo00=JeWooF]H027oo00EJeWoo
Oomooe[F00=oo`05F]IoogooOomJeP03Ool015[FOomooe[F0Woo00AJeWooOomJePMoo`03F]Ioogoo
009oo`07F]IoogooOomJeWooF]H00goo00=JeWooF]H00Woo00AJeWooOomJeP9oo`03F]Ioogoo00Eo
o`05F]IoogooOomJeP04Ool00e[FOomJeP03Ool01E[FOomJeWooF]H00goo00=JeWooF]H027oo00=J
eWooF]H00goo00=JeWooF]H00Woo00AJeWooOomJeP9oo`03F]Ioogoo00ioo`0067oo00AJeWooOomo
o`9JeP=oo`05F]IoogooOomJeP03Ool00e[FOomoo`06Ool01U[FOomooe[FOomJeP9oo`03F]Ioogoo
00Ioo`03F]Ioogoo00Eoo`06F]IoogooF]Iooe[F0Woo00=JeWooOol00Woo00EJeWooOomooe[F009o
o`03F]Iooe[F009oo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`03F]Ioogoo00Ioo`05F]IoogooOomJ
eP03Ool01E[FOomoogooF]H017oo00IJeWooOomJeWooF]H2Ool00e[FOomoo`06Ool01E[FOomoogoo
F]H00goo00EJeWooOomooe[F00=oo`04F]IoogooOol2F]H3Ool00e[FOomoo`05Ool00e[FOomoo`02
Ool01E[FOomoogooF]H00Woo00=JeWooF]H00Woo00AJeWooOomoo`9JeP=oo`03F]Ioogoo00Eoo`05
F]IoogooOomJeP04Ool01U[FOomooe[FOomJeP9oo`06F]IoogooF]Iooe[F0Woo00=JeWooOol01Woo
00IJeWooOomJeWooF]H2Ool015[FOomoogoo0U[F0goo00=JeWooOol03Woo000HOol01E[FOomoogoo
F]H017oo2E[F27oo00EJeWooOomooe[F00=oo`03F]Ioogoo00Ioo`03F]Ioogoo00Eoo`05F]Ioogoo
OomJeP03Ool00e[FOomJeP05F]H00gooF]Ioo`02Ool01E[FOomoogooF]H027oo00=JeWooOol01Goo
00=JeWooOol01Woo1e[F00=ooe[FOol01U[F00=ooe[FOol00Woo00EJeWooOomooe[F00Qoo`MJeP03
OomJeWoo00MJeP=oo`03F]Ioogoo009oo`03F]Ioogoo00Eoo`03F]Iooe[F00EJeP03OomJeWoo009o
o`05F]IoogooOomJeP03Ool00e[FOomoo`02Ool00e[FOomoo`05Ool00e[FOomJeP05F]H00gooF]Io
o`02Ool01E[FOomoogooF]H00goo00EJeWooOomooe[F00Qoo`05F]IoogooOomJeP03Ool01E[FOomo
ogooF]H017oo00=JeWooOol03Woo000HOol015[FOomoogoo0U[F0goo00EJeWooOomooe[F00=oo`03
F]Ioogoo00Ioo`06F]IoogooF]Iooe[F0Woo00=JeWooOol01Woo00=JeWooOol01Goo00IJeWooOomJ
eWooF]H2Ool00e[FOomoo`02Ool01E[FOomoogooF]H00Woo00=JeWooF]H00Woo00=JeWooOol01Woo
00=JeWooOol01Goo00=JeWooOol01Woo00EJeWooOomooe[F00=oo`05F]IoogooOomJeP04Ool01U[F
Oomooe[FOomJeP9oo`03F]Ioogoo00Ioo`05F]IoogooOomJeP03Ool01E[FOomoogooF]H00goo00AJ
eWooOomoo`9JeP=oo`03F]Ioogoo00Eoo`03F]Ioogoo009oo`05F]IoogooOomJeP02Ool00e[FOomJ
eP02Ool015[FOomoogoo0U[F0goo00=JeWooOol01Goo00EJeWooOomooe[F00Aoo`06F]IoogooF]Io
oe[F0Woo00IJeWooOomJeWooF]H2Ool00e[FOomoo`06Ool01U[FOomooe[FOomJeP9oo`04F]Ioogoo
Ool2F]H3Ool00e[FOomoo`0>Ool001Qoo`04F]IoogooF]H2Ool015[FOomooe[F0goo00EJeWooOomo
oe[F00Qoo`03F]Iooe[F00=oo`03F]Iooe[F00Qoo`03F]Ioogoo00Eoo`03F]Iooe[F00=oo`03F]Io
oe[F00Aoo`07F]IoogooOomJeWooF]H00goo00=JeWooF]H027oo00=JeWooOol01Goo00=JeWooOol0
1Woo00EJeWooOomooe[F00=oo`05F]IoogooOomJeP04Ool00e[FOomJeP03Ool00e[FOomJeP08Ool0
1E[FOomoogooF]H00goo00EJeWooOomooe[F00=oo`04F]IoogooF]H2Ool015[FOomooe[F1goo00=J
eWooOol00Woo00MJeWooOomooe[FOomJeP03Ool00e[FOomJeP02Ool015[FOomooe[F0Woo00=JeWoo
Ool01Goo00EJeWooOomooe[F00Aoo`03F]Iooe[F00=oo`05F]Iooe[FOomJeP03Ool00e[FOomJeP08
Ool00e[FOomJeP03Ool00e[FOomJeP02Ool015[FOomooe[F0Woo00=JeWooOol03Woo000HOol00e[F
Oomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[F
Oomoo`05Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[F
Oomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[F
Oomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[F
Oomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[F
Oomoo`05Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[FOomoo`05Ool00e[FOomoo`06Ool00e[F
Oomoo`0>Ool001QooomJeQ1oo`00ogoo:7oo0000\
\>"],
  ImageRangeCache->{{{0, 294.75}, {50.5, 0}} -> {-0.033764, -0.00435576, 
  0.00362181, 0.00362181}, {{9.3125, 285.375}, {49.25, 
  40.375}} -> {-1.93344, -0.172903, 0.118318, 0.118318}, {{9.3125, 285.375}, {
  39.5, 30.5625}} -> {-1.87319, -1.32388, 0.117909, 0.117909}, {{9.3125, 
  285.375}, {29.6875, 20.75}} -> {-1.87319, -2.48086, 0.117909, 0.117909}, {{
  9.3125, 285.375}, {19.875, 11}} -> {-1.93344, -3.64849, 0.118318, 
  0.118318}, {{9.3125, 285.375}, {10.0625, 1.1875}} -> {-1.93344, -4.80949, 
  0.118318, 0.118318}}],

Cell[OutputFormData["\<\
$Aborted\
\>", "\<\
$Aborted\
\>"], "Output",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Code for Simulator in here ...", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True,
  FontSize->12,
  FontColor->RGBColor[1, 0, 0]],

Cell["\<\
DistributeKeyUsingQKD[desiredProbOfDetection_, nKeyBits_, \
opts___]:=
\tModule[{nTestBits, bitsA},
\t\tnTestBits = NumberOfBitsForDetectability[desiredProbOfDetection];
\t\tbitsA = Table[Random[Integer,{0,1}], {nTestBits + 4 * nKeyBits}];
\t\tQKD[bitsA, nTestBits, opts]
\t]
\t
DistributeKeyUsingQKD::usage =
\t\"DistributeKeyUsingQKD[p, n, options] generates a simulation of \\
\tquantum key distribution.  The desired probability of detecting \\
\teavesdropping is p and the desired number of bits in the \\
\tsecret shared key is n. If the option Eavesdropping->True is \\
\tgiven, then the actions of a malicious intermediary Eve are \\
\tsimulated. If the option Eavesdropping->False is given, then \\
\tthe simulator runs without Eve intervening.\";

Options[QKD] = {Eavesdropping->True};

QKD[bitString_String, nTestBits_Integer, opts___]:= 
\tQKD[Map[ToExpression, Characters[bitString]], nTestBits, opts]
\t
QKD[bits_, nTestBits_, opts___]:=
\tModule[{photonsAliceToBob, polarizersEve, bitsEve,
\t        photonsEveToBob, polarizersBob, bitsBob, whichBitsToTest,
\t        estErrorRate},
\t        
\t\tevePresentQ = Eavesdropping /. {opts} /. Options[QKD];
\t\tPrint[\"evePresentQ = \", evePresentQ];
\t\t
\t                (* Alice *)
\t\tpolarizersAlice   = PickPolarizers[ Length[bits] ];
\t\tphotonsAliceToBob = BitsToPhotons[bits, polarizersAlice];
\t\tPortrayEncoding[bits, polarizersAlice, photonsAliceToBob];
\t\t
\t\tpolarizersBob     = PickPolarizers[ Length[bits] ];
\t\t    (* I moved Bob's pick of polarizers ahead of anything Eve does
\t\t       so that we can set a seed in the random number generator
\t\t       that allows us to compare what Bob sees with and without 
\t\t       eavesdropping (assuming Alice makes the same choices for 
\t\t       bits and her polarizers and assuming Bob makes the same choices 
\t\t       for his polarizers. If this code was after Eve then I could
\t\t       not ensure Bob picked the same set of polarizers. *)
\t\t
\t\t            (* Eve *)
\t\tIf[evePresentQ,
\t\t   (polarizersEve     = PickPolarizers[ Length[photonsAliceToBob] ];
\t\t    bitsEve           = PhotonsToBits[photonsAliceToBob,polarizersEve];
\t\t    photonsEveToBob   = BitsToPhotons[bitsEve, polarizersEve];
\t\t    PortrayDecoding[photonsAliceToBob, polarizersEve, bitsEve]
\t\t   ),
\t\t   (polarizersEve     = polarizersAlice;
\t\t    photonsEveToBob   = photonsAliceToBob
\t\t   )
\t\t  ];
\t\t
\t\t            (* Bob *)
\t\tbitsBob           = PhotonsToBits[photonsEveToBob,polarizersBob];
\t\tPortrayDecoding[photonsEveToBob, polarizersBob, bitsBob];
\t\t
\t\t            (* Estimate Error Rate *)
\t\t
\t\twhichBitsToTest   = ChooseBitsToTest[polarizersAlice, polarizersBob, \
nTestBits];
\t\testErrorRate      = EstimateErrorRate[bits, bitsBob, whichBitsToTest];
\t\tPortrayEavesdroppingTest[bits, polarizersAlice, bitsBob, polarizersBob, \
whichBitsToTest];
\t\tPortrayKeyExchange[polarizersAlice, bits, polarizersBob, bitsBob, 
\t\t                   Map[{#}&, whichBitsToTest]];
\t\t
\t\t            (* Do Key Exchange or Abort *)
\t\tIf[EavesdroppingQ[estErrorRate],
\t\t   DoNotExchangeKey[],
\t\t   ExchangeKey[polarizersAlice, bits, polarizersBob, bitsBob, \
whichBitsToTest]
\t\t  ]
\t]
\t
\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
ChooseBitsToTest[polarizersA_, polarizersB_, nTestBits_]:=
\tModule[{candidates},
\t\tcandidates = Complement[MapThread[If[#1===#2, #3, Null]&, 
\t                                      {polarizersA,
\t                                       polarizersB,
\t                                       Range[Length[polarizersA]]
\t                                      }
\t                                     ],
\t                            {Null}];
\t                            Print[candidates];
\t    If[nTestBits <= Length[candidates],
\t       RandomKSubset[candidates, nTestBits],
\t       Message[ChooseBitsToTest::insufficientMatches]; Abort[]
\t      ]
\t]

ChooseBitsToTest::insufficientMatches =
\t\"Alice and Bob do not have enough polarizers set at the same angle \\
\tto attain the probability of detecting eavesdropping that you seek. \\
\tThis is just bad luck ... try again.\";
\t\t    \
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["\<\
Select Between Rectilinear and Diagional Polarizers at Random\
\>", 
  "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
PickPolarizers[n_]:=
\tTable[If[CoinToss[]===Heads, RectilinearBasis, DiagonalBasis], {n}]\t
\t\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Mapping Bits into Polarized Photons", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
BitStringToPolarizedPhotons[bitString_]:=
\tMap[{#[[2]], #[[3]]}&, EncodeBitString[bitString]]
\t
BitsToPhotons[bits_, polarizers_]:=
\tMapThread[BitToPhoton[#1, #2]&, {bits, polarizers}]

BitToPhoton[0, RectilinearBasis]:= {RectilinearBasis, ket[0]}
BitToPhoton[1, RectilinearBasis]:= {RectilinearBasis, ket[1]}
BitToPhoton[0, DiagonalBasis]:=    {DiagonalBasis, ket[0]}
BitToPhoton[1, DiagonalBasis]:=    {DiagonalBasis, ket[1]}\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Measuring Polarized Photons to Extract Bits", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
If the sebder and receiver both used the same basis, then the \
answer is certain and guaranteed correct (assuming perfection in \
measurement). If the sender and receiver are using polarizers at 45 degree to \
one another, then the bit obtained from the measurement is effectively \
random.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
PhotonsToBits[polarizedPhotons_, polarizers_]:=
\tMapThread[If[#1[[1]] === #2,
\t             #1[[2,1]],
\t             Random[Integer,{0,1}]]&, {polarizedPhotons, polarizers}]\
\>", 
  "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Measuring the Polarizations of a Stream of Photons", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
MeasurePolarizations[photons_]:=
\tMap[MeasurePolarization, photons]

MeasurePolarization[{ket_, unknownBasis_}]:=
\tModule[{measurementBasis},
\t\tmeasurementBasis = ChooseBasisForMeasurement[];
\t\tWhich[measurementBasis === unknownBasis, ket,
\t\t      measurementBasis =!= unknownBasis, RandomAnswer[]
\t\t       
ChooseBasisForMeasurement[]:=
\tIf[CoinToss[]===Heads, RectilinearBasis, DiagonalBasis]\
\>", "Input",
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Estimate Error Rate Step", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
EstimateErrorRate[bitsA_, bitsB_, indices_]:=
\tModule[{testBitsA, testBitsB, comparisons},
\t\ttestBitsA   = Map[bitsA[[#]]&, indices];
\t\ttestBitsB   = Map[bitsB[[#]]&, indices];
\t\tcomparisons = MapThread[If[#1 === #2, Right, Wrong]&, 
\t\t                        {testBitsA, testBitsB}];
\t\tN[Count[comparisons, Wrong] / Length[comparisons]]
\t]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Criterion for Whether Eavesdropping is Detected", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
This method for detecting eavesdropping is guaranteed to what \
probability? Is this criterion True? What is  the uncertainty on the \
estimate?\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
EavesdroppingQ[estimatedErrorRate_]:=
\tIf[estimatedErrorRate > 0, True, False]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Key Exchange Step", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
DoNotExchangeKey[]:=
\t(Print[\"Channel is insecure! Do not exchange a key.\"]; Abort[])

ExchangeKey[polarizersA_, bitsA_, polarizersB_, bitsB_, indices_]:=
\tModule[{throwOuts, polarTypesA, secretBitsA, polarTypesB, secretBitsB,
\t        positions},
\t\tthrowOuts = Map[{#}&, indices];
\t\tpolarTypesA = Delete[polarizersA, throwOuts];
\t\tsecretBitsA = Delete[bitsA, throwOuts];
\t\tpolarTypesB = Delete[polarizersB, throwOuts];
\t\tsecretBitsB = Delete[bitsB, throwOuts];
\t\tpositions   = SamePolarizersAtPositions[polarTypesA, polarTypesB];
\t\t{
\t\t Map[secretBitsA[[#]]&, positions],
\t\t Map[secretBitsB[[#]]&, positions]
\t\t}
\t]
\t\t
SamePolarizersAtPositions[polarTypesA_, polarTypesB_]:=
\tModule[{matches},
\t\tmatches = MapThread[If[#1===#2, Agree, Disagree]&, 
\t\t                    {polarTypesA, polarTypesB}];
\t\tPosition[matches, Agree]
\t] /; Length[polarTypesA] == Length[polarTypesB]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Graphics Utilities", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
PortrayEncoding[bits_, polarizers_, photons_]:=
\tModule[{g1, g2, g3},
\t\tg1 = PortrayBits[bits];
\t\tg2 = PortrayPolarizers[polarizers];
\t\tg3 = PortrayPolarizedPhotons[photons];
\t\tShow[GraphicsArray[{{g1}, {g2}, {g3}}, AspectRatio->Automatic]]
\t]

PortrayDecoding[photons_, polarizers_, bits_]:=
\tModule[{g1, g2, g3},
\t\tg1 = PortrayPolarizedPhotons[photons];
\t\tg2 = PortrayPolarizers[polarizers];
\t\tg3 = PortrayBits[bits];
\t\tShow[GraphicsArray[{{g1}, {g2}, {g3}}, AspectRatio->Automatic]]
\t]
\t
PortrayEavesdroppingTest[bitsA_, polarizersA_, bitsB_, 
                         polarizersB_, indicesToTest_]:=
\tModule[{excludeIndices, g1, g2, g3, g4},
\t\texcludeIndices = Map[{#}&, Complement[Range[Length[bitsA]], \
indicesToTest]];
\t\tg1 = PortrayBits[bitsA, excludeIndices];
\t\tg2 = PortrayPolarizers[polarizersA, excludeIndices];
\t\tg3 = PortrayPolarizers[polarizersB, excludeIndices];
\t\tg4 = PortrayBits[bitsB, excludeIndices];
\t\tShow[GraphicsArray[{{g1}, {g2}, {g3}, {g4}}, AspectRatio->Automatic]]
\t]
\t
PortrayKeyExchange[polarizersA_, bitsA_, polarizersB_, bitsB_, \
excludeIndices_]:=
\tModule[{g1, g2, g3, g4, matchingBits, g5},
\t\tg1 = PortrayPolarizers[polarizersA, excludeIndices];
\t\tg2 = PortrayPolarizers[polarizersB, excludeIndices];
\t\tg3 = PortrayBits[bitsB, excludeIndices];
\t\tg4 = PortrayMatchingTypes[polarizersA, polarizersB, excludeIndices];
\t\tmatchingBits = MapThread[If[#1 === #2, #3, \" \"]&,
\t\t                         {polarizersA, polarizersB, bitsB}];
\t\tg5 = PortrayBits[matchingBits, excludeIndices];
\t\t
\t\tShow[GraphicsArray[{{g1},{g2},{g3},{g4},{g5}},
\t\t                   AspectRatio->Automatic]]
\t]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Portray Bits", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
PortrayBits[bits_, excludeIndices_:{}]:=
\tModule[{usableBits},
\t\tusableBits = ReplacePart[bits, \" \", excludeIndices];
\t\tGraphics[Join[MakeBits[usableBits],
\t                  MyMakeBoxes[Length[bits]]],
\t             AspectRatio->Automatic]
\t]

MyMakeBoxes[n_]:=
\tJoin[{GrayLevel[0.7], AbsoluteThickness[1]}, 
\t     Table[Line[{{i,0},{i+1,0},{i+1,1},{i,1},{i,0}}], {i, 1, n}]]

MakeBits[bits_]:=
\tTable[Text[bits[[i]], {i+0.5, 0.5}], {i, 1, Length[bits]}]

\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Portray Polarizers", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
PortrayPolarizers[polarizers_, excludeIndices_:{}]:=
\tModule[{usablePolarizers},
\t\tusablePolarizers = ReplacePart[polarizers, \" \", excludeIndices];
\t\tGraphics[Join[MakePolarizers[usablePolarizers],
\t                  MyMakeBoxes[Length[polarizers]]],
\t             AspectRatio -> Automatic]
\t]
\t        
MakePolarizers[polarizers_]:=
\tMapIndexed[MakePolarizer, polarizers]

MakePolarizer[RectilinearBasis, {i_}]:= MakeCross[i, 0.6,  0]
MakePolarizer[DiagonalBasis,    {i_}]:= MakeCross[i, 0.6, N[Pi/4]]
MakePolarizer[\" \", {i_}]:={}
\t
MakeCross[i_, l_, theta_]:=
\tModule[{j,s,c},
\t\tj = i + 0.5; s = 0.5 l Sin[theta]; c = 0.5 l Cos[theta];
\t\t{GrayLevel[0.7], Line[{{j-c, 0.5 - s}, {j+c, 0.5 + s}}],
\t\t GrayLevel[0.7], Line[{{j-s, 0.5 + c}, {j+s, 0.5 - c}}]
\t\t}
\t]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Portray Polarized Photons", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
PortrayPolarizedPhotons[photons_]:=
\tModule[{boxes, lines},
\t\tboxes = MyMakeBoxes[Length[photons]];
\t\tlines = MapThread[PortrayPolarizedPhoton[#1, #2, 0.6]&, 
\t                      {photons, Range[Length[photons]]}];
\t    Graphics[Join[boxes, {GrayLevel[0], lines}], AspectRatio->Automatic]
\t]

PortrayPolarizedPhoton[{RectilinearBasis, ket[0]}, x0_, l_]:=
\t{Line[{{x0+0.5, .5 - l/2}, {x0+0.5, .5 + l/2}}]}

PortrayPolarizedPhoton[{RectilinearBasis, ket[1]}, x0_, l_]:=
\t{Line[{{x0 + .5 - l/2, 0.5}, {x0+.5+l/2, 0.5}}]}

PortrayPolarizedPhoton[{DiagonalBasis,    ket[0]}, x0_, l_]:=
\t{Line[{{x0 + .5 - l/2 Cos[Pi/4],   .5 - l/2 Sin[Pi/4]}, 
\t       {x0 + .5 + l/2 Cos[Pi/4],   .5 + l/2 Sin[Pi/4]}
\t      }]
\t}

PortrayPolarizedPhoton[{DiagonalBasis,    ket[1]}, x0_, l_]:=
\t{Line[{{x0 + .5 - l/2 Cos[Pi/4],   .5 + l/2 Sin[Pi/4]}, 
\t       {x0 + .5 + l/2 Cos[Pi/4],   .5 - l/2 Sin[Pi/4]}
\t      }]
\t}\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Portray Matching Types of Polarizers", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["Clear[PortrayMatchingFaces, GoodFace, BadFace]", "Input",
  AspectRatioFixed->True],

Cell["\<\
PortrayMatchingTypes[polarizersA_, polarizersB_, indices_]:=
\tModule[{faces, usableFaces, boxes},
\t\tfaces = MapThread[If[#1 === #2, 
\t                         GoodFace[ {#3+0.5, 0.5}, 0.4 ],
\t                         BadFace[  {#3+0.5, 0.5}, 0.4 ]
\t                        ]&,
\t                      {polarizersA, 
\t                       polarizersB,
\t                       Range[Length[polarizersA]]}
\t                     ];
\t     usableFaces = ReplacePart[faces, {}, indices];
\t     boxes = MyMakeBoxes[Length[polarizersA]];
\t     (* Show[usableFaces, boxes, AspectRatio -> Automatic] *)
\t     Graphics[Join[usableFaces, boxes], AspectRatio -> Automatic]
\t]

GoodFace[{x0_, y0_}, radius_]:=
\t{Circle[{x0,y0}, radius],
\t Disk[{x0 + .3 radius, y0 + .3 radius}, .15 radius],
\t Disk[{x0 - .3 radius, y0 + .3 radius}, .15 radius],
\t Line[{{x0 - .3 radius, y0 - .2 radius},
\t       {x0 - .2 radius, y0 - .4 radius},
\t       {x0 + .2 radius, y0 - .4 radius},
\t       {x0 + .3 radius, y0 - .2 radius}
\t      }
\t     ]
\t}
\t         
BadFace[{x0_, y0_}, radius_]:=
\t{Circle[{x0,y0}, radius],
\t Disk[{x0 + .3 radius, y0 + .3 radius}, .15 radius],
\t Disk[{x0 - .3 radius, y0 + .3 radius}, .15 radius],
\t Line[{{x0 - .3 radius, y0 - .4 radius},
\t       {x0 - .2 radius, y0 - .2 radius},
\t       {x0 + .2 radius, y0 - .2 radius},
\t       {x0 + .3 radius, y0 - .4 radius}
\t      }
\t     ]
\t}
\t          \
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Random Utilities", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["CoinToss[]:= If[Random[]<0.5, Heads, Tails]", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
RandomKSubset[list_, k_]:=
\tRandomKSubset[list, k, {}] /; k <= Length[list]
\t
RandomKSubset[_, 0, final_]:= final
RandomKSubset[list_, k_, soFar_]:=
\tModule[{i},
\t\ti = Random[Integer, {1, Length[list]}];
\t\tRandomKSubset[Drop[list, {i,i}], k-1, Append[soFar, list[[i]] ]]
\t]
\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Your experiments here ...", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["?DistributeKeyUsingQKD", "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    \("DistributeKeyUsingQKD[p, n, options] generates a simulation of  \
quantum key distribution.  The desired probability of detecting  \
eavesdropping is p and the desired number of bits in the  secret shared key \
is n. If the option Eavesdropping->True is  given, then the actions of a \
malicious intermediary Eve are  simulated. If the option Eavesdropping->False \
is given, then  the simulator runs without Eve intervening."\)], "Print"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"Microsoft Windows 3.0",
ScreenRectangle->{{0, 1280}, {0, 968}},
AutoGeneratedPackage->Automatic,
WindowToolbars->{},
CellGrouping->Manual,
WindowSize->{937, 833},
WindowMargins->{{36, Automatic}, {Automatic, 16}},
PrivateNotebookOptions->{"ColorPalette"->{RGBColor, -1}},
ShowCellLabel->True,
ShowCellTags->False,
RenderingOptions->{"ObjectDithering"->True,
"RasterDithering"->False}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1731, 51, 118, 3, 105, "Title",
  Evaluatable->False],
Cell[1852, 56, 108, 3, 53, "Subsubtitle",
  Evaluatable->False],
Cell[1963, 61, 81, 2, 30, "Input",
  InitializationCell->True],

Cell[CellGroupData[{
Cell[2069, 67, 102, 3, 53, "Section",
  Evaluatable->False],
Cell[2174, 72, 905, 19, 234, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[3116, 96, 110, 3, 53, "Section",
  Evaluatable->False],
Cell[3229, 101, 837, 17, 174, "Text",
  Evaluatable->False],
Cell[4069, 120, 86, 2, 30, "Input"],
Cell[4158, 124, 216, 6, 34, "Text",
  Evaluatable->False],
Cell[4377, 132, 90, 2, 30, "Input"],
Cell[4470, 136, 205, 6, 34, "Text",
  Evaluatable->False],
Cell[4678, 144, 83, 2, 30, "Input"],
Cell[4764, 148, 156, 6, 34, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[4957, 159, 176, 5, 53, "Section",
  Evaluatable->False],
Cell[5136, 166, 481, 9, 74, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[5642, 179, 77, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[5722, 183, 424, 10, 70, "Input",
  InitializationCell->True],
Cell[6149, 195, 706, 16, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[6892, 216, 80, 2, 29, "Subsubsection",
  Evaluatable->False],
Cell[6975, 220, 335, 7, 54, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[7335, 231, 295, 8, 120, "Input"],
Cell[7633, 241, 43923, 1101, 182, 9707, 673, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[51605, 1348, 80, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[51688, 1352, 246, 6, 54, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[51959, 1362, 261, 8, 102, "Input"],
Cell[52223, 1372, 27116, 1074, 182, 13865, 904, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[79400, 2453, 109, 2, 53, "Section",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[79534, 2459, 77, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[79614, 2463, 325, 12, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[79976, 2480, 80, 2, 29, "Subsubsection",
  Evaluatable->False],
Cell[80059, 2484, 103, 2, 34, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[80187, 2490, 76, 1, 30, "Input"],
Cell[80266, 2493, 854, 23, 87, "Output",
  Evaluatable->False]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[81181, 2523, 102, 2, 53, "Section",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[81308, 2529, 65, 1, 30, "Input"],
Cell[81376, 2532, 457, 9, 76, "Print"]
}, Open  ]],

Cell[CellGroupData[{
Cell[81870, 2546, 80, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[81953, 2550, 287, 7, 54, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[82265, 2561, 123, 4, 48, "Input"],
Cell[82391, 2567, 149, 5, 42, "Print",
  Evaluatable->False],
Cell[82543, 2574, 36518, 1257, 49, 13256, 963, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False],
Cell[119064, 3833, 36410, 1255, 49, 13246, 963, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False],
Cell[155477, 5090, 68162, 2675, 56, 47391, 2412, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False],
Cell[223642, 7767, 75768, 3153, 61, 51897, 2852, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False],
Cell[299413, 10922, 186, 7, 27, "Output",
  Evaluatable->False]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[299648, 10935, 80, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[299731, 10939, 161, 5, 34, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[299917, 10948, 122, 4, 48, "Input"],
Cell[300042, 10954, 192, 6, 59, "Print",
  Evaluatable->False],
Cell[300237, 10962, 36193, 1253, 48, 13256, 963, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False],
Cell[336433, 12217, 35598, 1245, 48, 13249, 963, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False],
Cell[372034, 13464, 35728, 1247, 48, 13237, 963, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False],
Cell[407765, 14713, 66586, 2656, 54, 47383, 2412, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False],
Cell[474354, 17371, 74623, 3139, 60, 51906, 2852, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False],
Cell[548980, 20512, 118, 6, 27, "Output",
  Evaluatable->False]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[549147, 20524, 148, 4, 45, "Subsection",
  Evaluatable->False],
Cell[549298, 20530, 3321, 82, 1362, "Input",
  InitializationCell->True],
Cell[552622, 20614, 977, 24, 390, "Input",
  InitializationCell->True],

Cell[CellGroupData[{
Cell[553624, 20642, 145, 5, 43, "Subsubsection",
  Evaluatable->False],
Cell[553772, 20649, 174, 6, 66, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[553983, 20660, 108, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[554094, 20664, 509, 13, 192, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[554640, 20682, 116, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[554759, 20686, 363, 8, 54, "Text",
  Evaluatable->False],
Cell[555125, 20696, 258, 8, 84, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[555420, 20709, 123, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[555546, 20713, 456, 13, 210, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[556039, 20731, 97, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[556139, 20735, 431, 11, 156, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[556607, 20751, 120, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[556730, 20755, 216, 6, 34, "Text",
  Evaluatable->False],
Cell[556949, 20763, 158, 5, 48, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[557144, 20773, 90, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[557237, 20777, 990, 27, 444, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[558264, 20809, 91, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[558358, 20813, 1756, 45, 732, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[560151, 20863, 85, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[560239, 20867, 549, 19, 300, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[560825, 20891, 91, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[560919, 20895, 865, 25, 408, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[561821, 20925, 98, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[561922, 20929, 997, 28, 462, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[562956, 20962, 109, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[563068, 20966, 89, 1, 30, "Input"],
Cell[563160, 20969, 1509, 43, 732, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[564706, 21017, 89, 2, 43, "Subsubsection",
  Evaluatable->False],
Cell[564798, 21021, 114, 2, 30, "Input",
  InitializationCell->True],
Cell[564915, 21025, 361, 13, 192, "Input",
  InitializationCell->True]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{
Cell[565337, 21045, 92, 2, 33, "Section",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[565454, 21051, 65, 1, 30, "Input"],
Cell[565522, 21054, 460, 6, 82, "Print"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

